!Module for exchanging data between main program and various routines
module defvar
type atomtype
integer index !Atomic index in periodic table
real*8 x,y,z,mass !Coordinate in Angstrom (x/y/z,atom), relative atomic mass in amu
end type

!Define constants
real*8,parameter :: R=8.3144648D0, kb=1.3806503D-23 !Ideal gas constant (J/mol/K), Boltzmann constant (J/K)
real*8,parameter :: NA=6.02214179D23 !Avogadro constant
real*8,parameter :: au2eV=27.2113838D0,au2kcal_mol=627.51D0,au2kJ_mol=2625.5D0,au2J=4.359744575D-18,au2cm_1=219474.6363D0 !Hartree to various units
real*8,parameter :: cal2J=4.184D0
real*8,parameter :: wave2freq=2.99792458D10 !cm^-1 to s^-1 (Hz)
real*8,parameter :: h=6.62606896D-34 !Planck constant, in J*s
real*8,parameter :: amu2kg=1.66053878D-27
real*8,parameter :: pi=3.141592653589793D0
real*8,parameter :: b2a=0.52917720859D0 !Bohr to Angstrom
real*8,parameter :: atm2Pa=101325 !atm to Pa
integer,parameter :: nelesupp=150
character*2 :: ind2name(0:nelesupp)=(/ "Bq","H ","He", &   !Bq(number 0) is ghost atom
"Li","Be","B ","C ","N ","O ","F ","Ne", & !3~10
"Na","Mg","Al","Si","P ","S ","Cl","Ar", & !11~18
"K ","Ca","Sc","Ti","V ","Cr","Mn","Fe","Co","Ni","Cu","Zn","Ga","Ge","As","Se","Br","Kr", & !19~36
"Rb","Sr","Y ","Zr","Nb","Mo","Tc","Ru","Rh","Pd","Ag","Cd","In","Sn","Sb","Te","I ","Xe", & !37~54
"Cs","Ba","La","Ce","Pr","Nd","Pm","Sm","Eu","Gd","Tb","Dy","Ho","Er","Tm","Yb","Lu", & !55~71
"Hf","Ta","W ","Re","Os","Ir","Pt","Au","Hg","Tl","Pb","Bi","Po","At","Rn", & !72~86
"Fr","Ra","Ac","Th","Pa","U ","Np","Pu","Am","Cm","Bk","Cf","Es","Fm","Md","No","Lr", & !87~103
"Rf","Db","Sg","Bh","Hs","Mt","Ds","Rg","Cn","Nh","Fl","Mc","Lv","Ts","Og","Un","Ux",("??",i=121,nelesupp) /) !104~all. Name is in line with NIST periodic table. Such as Uun is replaced by Un
integer,parameter :: maxiso=300
real*8 isomass(nelesupp,maxiso) !Isotope masses
real*8 isowei(nelesupp,maxiso) !Isotope composition
real*8 elemass(nelesupp) !Element masses

!Molecular information
integer ncenter !Number of atoms
type(atomtype),allocatable :: a(:)
integer ilinear !0/1: This system isn't / is a linear molecule
integer nfreq !Number of frequencies
real*8,allocatable :: freq(:),wavenum(:) !Frequencies (s^-1, i.e. Hz) and wavenumber (cm^-1)
real*8 inertmat(3,3) !Moments of inertia matrix in a.u. (amu*Bohr^2)
real*8 inert(3) !Inertia of principle axes in a.u. (amu*Bohr^2)
real*8 E !Electronic energy in a.u.
real*8 totmass !Mass of the whole system in amu
integer spinmult !Spin multiplicity. If .shm is used as input file, this is zero
integer rotsym !Rotational symmetry number
integer nelevel !Number of electron levels
real*8,allocatable :: elevel(:) !Electron levels with respect to ground state (eV)
integer,allocatable :: edegen(:) !Degeneracy of electron levels

!Parameters loaded from settings.ini or specified via arguments
character*4 :: PGlabelinit="?",PGlabel !Point group label, initially defined in settings.ini, and that used in Shermo
character*20 :: concstr="0" !Record concentration string
integer :: prtvib=0,ilowfreq=2,defmass=3,outshm=0,imode=0
real*8 :: T=298.15D0,Tlow,Thigh,Tstep=0
real*8 :: P=1D0,Plow,Phigh,Pstep=0
real*8 :: sclZPE=1,sclheat=1,sclS=1,sclCV=1
real*8 :: ravib=100,intpvib=100,imagreal=0
real*8 :: Eexter=0 !Electronic energy specified via "E" parameter

!Special
integer :: inoset=0 !If =1, do not try to load setting from settings.ini, mainly for avoiding conflict with settings.ini of Molclus

!Others
logical alive
character inputfile*200
#ifdef _WIN32
integer :: isys=1 !Windows
#else
integer :: isys=2 !Linux/MacOS
#endif
end module



!!!------------------------------------------------------------------
!!!------------------------- Shermo program -------------------------
!!!------------------------------------------------------------------
program Shermo
use defvar
use util
implicit real*8 (a-h,o-z)
character c80tmp*80,c200tmp*200
real*8 rotcst(3)

write(*,*) "Shermo: A general code for calculating molecular thermochemistry properties"
write(*,*) "Version 2.6.1  Release date: 2025-Oct-20"
write(*,*) "Developer: Tian Lu (sobereva@sina.com)"
write(*,*) "Beijing Kein Research Center for Natural Sciences (http://www.keinsci.com)"
write(*,*) "Official website: http://sobereva.com/soft/shermo"
write(*,*)
write(*,*) " **** If this code is utilized in your work, PLEASE CITE following paper ****"
write(*,"(a)") " Tian Lu, Qinxue Chen, Shermo: A general code for calculating molecular thermodynamic properties, Comput. Theor. Chem., 1200, 113249 (2021) DOI: 10.1016/j.comptc.2021.113249"

!***** Initialize isotope mass table
call initmass

!***** Loading running parameters
narg=command_argument_count()
do iarg=1,narg
    call get_command_argument(iarg,c80tmp)
    if (c80tmp=="-noset") then
        inoset=1
	    exit
    end if
end do
if (inoset==1) then
    write(*,*) "Do not try to load settings from settings.ini because of ""-noset"" argument"
else
    call loadsettings
end if
if (narg>1) call loadarguments !Load arguments to override those in settings.ini

!***** Printing running parameters
write(*,*)
write(*,*) "Running parameters:"
if (prtvib==1) then
    write(*,*) "Printing individual contribution of vibration modes: Yes"
else if (prtvib==-1) then
    write(*,*) "Printing individual contribution of vibration modes: Yes, to vibcontri.txt"
else if (prtvib==0) then
    write(*,*) "Printing individual contribution of vibration modes: No"
end if
if (Tstep==0) then
    write(*,"(' Temperature:',f12.3,' K')") T
else
    write(*,"(' Temperature scan, from',f10.3,' to',f10.3,', step:',f8.3,' K')") Tlow,Thigh,Tstep
end if
if (Pstep==0) then
    write(*,"(' Pressure:   ',f12.3,' atm')") P
else
    write(*,"(' Pressure scan, from',f10.3,' to',f10.3,', step:',f8.3,' atm')") Plow,Phigh,Pstep
end if
write(*,"(' Scale factor of vibrational frequencies for ZPE:      ',f8.4)") sclZPE
write(*,"(' Scale factor of vibrational frequencies for U(T)-U(0):',f8.4)") sclheat
write(*,"(' Scale factor of vibrational frequencies for S(T):     ',f8.4)") sclS
write(*,"(' Scale factor of vibrational frequencies for CV:       ',f8.4)") sclCV
if (ilowfreq==0) then
    write(*,*) "Low frequencies treatment: Harmonic approximation"
else if (ilowfreq==1) then
    write(*,*) "Low frequencies treatment: Raising low frequencies (Truhlar's treatment)"
    write(*,"(a,f7.2,a)") " Lower frequencies will be raised to",ravib," cm^-1 during calculating S, U(T)-U(0), CV and q"
else if (ilowfreq==2) then
    write(*,*) "Low frequencies treatment: Grimme's interpolation for entropy"
else if (ilowfreq==3) then
    write(*,*) "Low frequencies treatment: Minenkov's interpolation for entropy and internal energy"
end if
if (ilowfreq==2.or.ilowfreq==3) then
    write(*,"(a,f7.2,a)") " Vibrational frequency threshold used in the interpolation is",intpvib," cm^-1"
end if
if (imagreal/=0) then
    write(*,"(a,f7.2,a)") " Imaginary frequencies with norm <",imagreal," cm^-1 will be treated as real frequencies"
end if


!***** Loading input file
call get_command_argument(1,inputfile)
if (inputfile==" ") then
    write(*,*)
    write(*,*) "Input file path, e.g. C:\lovelive\A-rise.shm"
    write(*,"(a)") " Output file of frequency analysis task of Gaussian/ORCA/GAMESS-US/NWChem/CP2K is also supported"
    do while(.true.)
	    read(*,"(a)") inputfile
        do ic=1,len_trim(inputfile)
            if (inputfile(ic:ic)=='"') inputfile(ic:ic)=" "
        end do
	    inquire(file=inputfile,exist=alive)
	    if (alive) exit
	    write(*,*) "Cannot find the file, input again!"
    end do
else
	inquire(file=inputfile,exist=alive)
	if (.not.alive) then
        write(*,*) "Error: Unable to find "//trim(inputfile)
        stop
    end if
end if

if (index(inputfile,'.txt')/=0) then !Load and calculate weights and thermodynamic data for a ensemble of systems
    call ensemble
    goto 10
else if (index(inputfile,'.shm')/=0) then !Load data from .shm file
    write(*,"(/,a)") " Loading data from "//trim(inputfile)
    call loadshm
else !Determine which program outputs this file, and then properly load data
    call deterprog(iprog)
    if (iprog==0) then
        write(*,"(a)") " Error: Unable to identify the program that generated this file, press ENTER button to exit"
        write(*,*) "PS: If your input file is a list file, you must use .txt as suffix"
        read(*,*)
        stop
    end if
    write(*,*)
    if (defmass==1) write(*,"(' Default atomic masses: Element')")
    if (defmass==2) write(*,"(' Default atomic masses: Most abundant isotope')")
    if (defmass==3) write(*,"(' Default atomic masses: Same as the output file')")
    if (iprog==1) then
        write(*,*) "Loading Gaussian output file..."
        call loadgau
    else if (iprog==2) then
        write(*,*) "Loading ORCA output file..."
        call loadorca
    else if (iprog==3) then
        write(*,*) "Loading GAMESS-US output file..."
        call loadgms
    else if (iprog==4) then
        write(*,*) "Loading NWChem output file..."
        call loadnw
    else if (iprog==5) then
        write(*,*) "Loading CP2K output file..."
        call loadCP2K
        if (imode==0) then
            write(*,"(a)") " Note: If your system is a crystal, slab or adsorbate, you may need to set ""imode"" in settings.ini to 1, &
            so that translation and rotation contributions will be removed"
            write(*,*)
        end if
    else if (iprog==6) then
        write(*,*) "Loading xtb g98.out file..."
        call loadxtb
    end if
    !Applying modification on default mass
    call modmass
    !Convert spin multiplicity to electronic level information
    nelevel=1
    allocate(elevel(nelevel),edegen(nelevel))
    elevel=0
    edegen=spinmult
    !Output current data to .shm file
    if (outshm==1) call outshmfile
end if
if (Eexter/=0D0) then
    E=Eexter !Use electronic directly specified by "E" parameter
    write(*,*) "Note: The electronic energy specified by ""E"" parameter will be used"
else
    if (E/=0) write(*,*) "Note: The electronic energy extracted from inputted file will be used"
end if

if (imagreal/=0) then
    do ifreq=1,nfreq
        if (wavenum(ifreq)<0) then
            if (abs(wavenum(ifreq))<imagreal) then
                wavenum(ifreq)=abs(wavenum(ifreq))
                write(*,"(' Note: Imaginary frequency ',f7.2,' cm^-1 has been set to real frequency!')") wavenum(ifreq)
            end if
        end if
    end do
end if

!Calculate moment of inertia based on atomic masses and geometry
totmass=sum(a(:)%mass)
call calcinertia
!Determine if this is a linear molecule, because treatment of rotational contribution is different
if (any(inert<0.001D0)) then
    ilinear=1
else
    ilinear=0
end if
!Sort wavenumbers and inertia w.r.t. principal axes from low to high
call sortr8(wavenum)
call sortr8(inert)
!-- Detect point group
call detectPG(1)


!***** Printing molecular information
write(*,*)
write(*,*) "                      ======= Molecular information ======="
write(*,"(' Electronic energy:',f18.8,' a.u.')") E
if (spinmult/=0) then
    write(*,"(' Spin multiplicity:',i3)") spinmult
else
    do ie=1,nelevel
        write(*,"(' Electronic energy level',i3,'     E=',f12.6,' eV     Degeneracy=',i3)") ie,elevel(ie),edegen(ie)
    end do
end if

do iatm=1,ncenter
    write(*,"(' Atom',i5,' (',a,')   Mass:',f12.6,' amu')") iatm,ind2name(a(iatm)%index),a(iatm)%mass
end do
write(*,"(' Total mass:',f16.6,' amu')") totmass
write(*,*)
write(*,"(a)") " Point group: "//PGlabel

if (imode==0) then
    write(*,"(' Rotational symmetry number:',i3)") rotsym
    write(*,"(' Principal moments of inertia (amu*Bohr^2):',/,3f16.6)") inert(:)
    if (sum(inert)<1E-10) then !Show rotational constants here, however, which are not utilized in thermodynamic data calculation codes
        write(*,*) "This is a single atom system, rotational constant is zero"
    else
        if (ilinear==1) then
            rotcst1=h/(8D0*pi**2*inert(3)*amu2kg*(b2a*1D-10)**2)
            write(*,"(' Rotational constant (GHz):',f14.6)") rotcst1/1D9
	        write(*,"(' Rotational temperature (K):',f12.6)") rotcst1*h/kb
            write(*,*) "This is a linear molecule"
        else
            rotcst(:)=h/(8D0*pi**2*inert(1:3)*amu2kg*(b2a*1D-10)**2)
            write(*,"(' Rotational constants relative to principal axes (GHz):',/,3f14.6)") rotcst(:)/1D9
	        write(*,"(' Rotational temperatures (K):',3f12.6)") rotcst(:)*h/kb
            write(*,*) "This is not a linear molecule"
        end if
    end if
else if (imode==1) then
    write(*,*) "Rotation information is not shown here since imode=1"
end if

if (nfreq>0) write(*,"(/,a,i6,a)") " There are",nfreq," frequencies (cm^-1):"
do ifreq=1,nfreq
    write(*,"(f8.1)",advance='no') wavenum(ifreq)
    if (mod(ifreq,9)==0) write(*,*)
end do
if (mod(nfreq,9)/=0) write(*,*)
allocate(freq(nfreq))
freq=wavenum*wave2freq !Transform wavenumber to frequencies
nimag=count(freq<0)
if (nimag>0) write(*,"(' Note: There are',i3,' imaginary frequencies, they will be ignored in the calculation')") nimag



!***** Loading and preparation has finished, now output results!
!ZPE,U,H,G are recorded in kJ/mol, CV,CP,S are recorded as J/mol/K

if (Tstep==0.and.Pstep==0) then !Calculate at specific T and P
    call showthermo
else
    write(*,*)
    write(*,*) "Performing scan of temperature/pressure..."
    P1=P;P2=P;Ps=1
    T1=T;T2=T;Ts=1
    if (Tstep/=0) then
        T1=Tlow;T2=Thigh;Ts=Tstep
    end if
    if (Pstep/=0) then
        P1=Plow;P2=Phigh;Ps=Pstep
    end if
    open(10,file="scan_UHG.txt",status="replace")
    write(10,*) "Unit of Ucorr, Hcorr and Gcorr is kcal/mol, unit of U, H and G is a.u."
    write(10,*)
    write(10,"(a)") "     T(K)     P(atm)    Ucorr     Hcorr     Gcorr            U                H                G"
    open(11,file="scan_SCq.txt",status="replace")
    write(11,*) "Unit of S, CV and CP is cal/mol/K, q(V=0)/NA and q(bot)/NA are dimensionless"
    write(11,*)
    write(11,*) "    T(K)     P(atm)      S         CV        CP       q(V=0)/NA      q(bot)/NA"
    do T=T1,T2,Ts
        do P=P1,P2,Ps
            call calcthermo(corrU,corrH,corrG,S,CV,CP,QV,Qbot)
            write(10,"(2f10.3,3f10.3,3f17.6)") T,P,corrU/cal2J,corrH/cal2J,corrG/cal2J,corrU/au2KJ_mol+E,corrH/au2KJ_mol+E,corrG/au2KJ_mol+E
            write(11,"(2f10.3,3f10.3,2(1PE16.6E3))") T,P,S/cal2J,CV/cal2J,CP/cal2J,QV/NA,Qbot/NA
        end do
    end do
    close(10)
    close(11)
    write(*,"(/,a)") " Done! Thermochemistry quantities at various temperatures/pressures have been outputted to scan_UHG.txt and scan_SCq.txt"
    write(*,"(a)") " scan_UHG.txt includes thermal correction to U, H and G, as well as sum of each of them and electronic energy"
    write(*,"(a)") " scan_SCq.txt includes S, CV, CP, q(V=0) and q(bot)"
    !Output electronic CV in kJ/mol/K
    !open(10,file="scan_CVele.txt",status="replace")
    !do T=T1,T2,Ts
    !    call elecontri(tmpq,tmpheat,tmpCV,tmpS)
    !    write(10,"(f10.3,f12.6)") T,tmpCV
    !end do
    !close(10)
end if

10 continue
if (narg==0) then
    write(*,*)
    write(*,*) "Running finished! Press ENTER button to exit"
    read(*,*)
end if
end program