!%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
!
! EDFLIB: EDF data library (Ver.5, 2018.05.10)
!
! Authors: W. Zou, Z. Cai, and J. Wang
!
!%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
!
! Input  :  iz    : Atomic number
!           ncore : Number of core electrons replaced by pseudopotential
!
! Output :  nfun  : Number of S-Gaussian functions (< 45). nfun = 0 if ncore(iz) is not supported.
!           alf   : Array of S-Gaussian exponents, which are defined by the even-tempered formula:
!                   Alpha_i = 0.001 * 1.65^(i-1),  i=1~60
!                   Reference: M. Reiher and A. Wolf, J. Chem. Phys. 121, 10945 (2004).
!           coe   : Array of S-Gaussian coefficients
!
!%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
!
! Li-Og:
!   Calculated at the X2C(f.n.)+AMFI/HF level with Dyall's CV4Z basis set.
!
! E119, E120:
!   Calculated at the X2C(f.n.)+AMFI/HF level with Malli's universal Gaussian basis set.
!
! All the basis functions are uncontracted.
!
!  ===============================================================================================================================
!  Name IZ  Configuration     Ncore     |  Name IZ  Configuration           Ncore     |  Name IZ  Configuration           Ncore
!  -------------------------------------+-----------------------------------------------------------------------------------------
!  H     1  (1s)1             N.A.      !  Nb*  41  (4d5s)5                 28,36     |  Tl   81  (4f)14(5d)10(6s)2(6p)1  60,68,78
!  He    2  (1s)2             N.A.      !  Mo   42  (4d)5(5s)1              28,36     |  Pb   82  (4f)14(5d)10(6s)2(6p)2  60,68,78
!  Li    3  (2s)1             2         |  Tc   43  (4d)5(5s)2              28,36     |  Bi   83  (4f)14(5d)10(6s)2(6p)3  60,68,78
!  Be    4  (2s)2             2         |  Ru   44  (4d)7(5s)1              28,36     |  Po   84  (4f)14(5d)10(6s)2(6p)4  60,68,78
!  B     5  (2s)2(2p)1        2         |  Rh   45  (4d)8(5s)1              28,36     |  At   85  (4f)14(5d)10(6s)2(6p)5  60,68,78
!  C     6  (2s)2(2p)2        2         |  Pd   46  (4d)10                  28,36     |  Rn   86  (4f)14(5d)10(6s)2(6p)6  60,68,78
!  N     7  (2s)2(2p)3        2         |  Ag   47  (4d)10(5s)1             28,36,46  |  Fr   87  (7s)1                   60,78
!  O     8  (2s)2(2p)4        2         |  Cd   48  (4d)10(5s)2             28,36,46  |  Ra   88  (7s)2                   60,78
!  F     9  (2s)2(2p)5        2         |  In   49  (4d)10(5s)2(5p)1        28,36,46  |  Ac   89  (6d)1(7s)2              60,78
!  Ne   10  (2s)2(2p)6        2         |  Sn   50  (4d)10(5s)2(5p)2        28,36,46  |  Th   90  (6d)2(7s)2              60,78
!  Na   11  (3s)1             2,10      |  Sb   51  (4d)10(5s)2(5p)3        28,36,46  |  Pa   91  (5f)2(6d)1(7s)2         60,78
!  Mg   12  (3s)2             2,10      |  Te   52  (4d)10(5s)2(5p)4        28,36,46  |  U    92  (5f)3(6d)1(7s)2         60,78
!  Al   13  (3s)2(3p)1        10        |  I    53  (4d)10(5s)2(5p)5        28,36,46  |  Np   93  (5f)4(6d)1(7s)2         60,78
!  Si   14  (3s)2(3p)2        10        |  Xe   54  (4d)10(5s)2(5p)6        28,36,46  |  Pu   94  (5f)6(7s)2              60,78
!  P    15  (3s)2(3p)3        10        |  Cs   55  (6s)1                   28,46,54  |  Am   95  (5f)7(7s)2              60,78
!  S    16  (3s)2(3p)4        10        |  Ba   56  (6s)2                   28,46,54  |  Cm   96  (5f)7(6d)1(7s)2         60,78
!  Cl   17  (3s)2(3p)5        10        |  La*  57  (5d)2(6s)1              28,46,54  |  Bk   97  (5f)9(7s)2              60,78
!  Ar   18  (3s)2(3p)6        10        |  Ce*  58  (4f)1(5d)2(6s)1         28,46,54  |  Cf   98  (5f)10(7s)2             60,78
!  K    19  (4s)1             10,18     |  Pr   59  (4f)3(6s)2              28,54     |  Es   99  (5f)11(7s)2             60,78
!  Ca   20  (4s)2             10,18     |  Nd   60  (4f)4(6s)2              28,54     |  Fm  100  (5f)12(7s)2             60,78
!  Sc*  21  (3d)2(4s)1        10,18     |  Pm   61  (4f)5(6s)2              28,54     |  Md  101  (5f)13(7s)2             60,78
!  Ti*  22  (3d4s)4           10,18     |  Sm   62  (4f)6(6s)2              28,54     |  No  102  (5f)14(7s)2             60,78
!  V*   23  (3d4s)5           10,18     |  Eu   63  (4f)7(6s)2              28,54     |  Lr  103  (5f)14(6d)1(7s)2        60,78
!  Cr   24  (3d)5(4s)1        10,18     |  Gd   64  (4f)7(5d)1(6s)2         28,54     |  Rf  104  (5f)14(6d)2(7s)2        78,92
!  Mn   25  (3d)5(4s)2        10,18     |  Tb   65  (4f)9(6s)2              28,54     |  Db  105  (5f)14(6d)3(7s)2        78,92
!  Fe   26  (3d)6(4s)2        10,18     |  Dy   66  (4f)10(6s)2             28,54     |  Sg  106  (5f)14(6d)4(7s)2        78,92
!  Co   27  (3d)7(4s)2        10,18     |  Ho   67  (4f)11(6s)2             28,54     |  Bh  107  (5f)14(6d)5(7s)2        78,92
!  Ni   28  (3d)9(4s)1        10,18     |  Er   68  (4f)12(6s)2             28,54     |  Hs  108  (5f)14(6d)6(7s)2        78,92
!  Cu   29  (3d)10(4s)1       10,18,28  |  Tm   69  (4f)13(6s)2             28,54     |  Mt  109  (5f)14(6d)7(7s)2        78,92
!  Zn   30  (3d)10(4s)2       10,18,28  |  Yb   70  (4f)14(6s)2             28,54     |  Ds  110  (5f)14(6d)8(7s)2        78,92
!  Ga   31  (3d)10(4s)2(4p)1  10,18,28  |  Lu   71  (4f)14(5d)1(6s)2        28,54     |  Rg  111  (5f)14(6d)9(7s)2        78,92
!  Ge   32  (3d)10(4s)2(4p)2  10,18,28  |  Hf   72  (4f)14(5d)2(6s)2        60,68     |  Cn  112  (5f)14(6d)10(7s)2       78,92
!  As   33  (3d)10(4s)2(4p)3  10,18,28  |  Ta*  73  (4f)14(5d6s)5           60,68     |  Nh  113  (5f)14(6d)10(7s)2(7p)1  78,92
!  Se   34  (3d)10(4s)2(4p)4  10,18,28  |  W    74  (4f)14(5d)5(6s)1        60,68     |  Fl  114  (5f)14(6d)10(7s)2(7p)2  78,92
!  Br   35  (3d)10(4s)2(4p)5  10,18,28  |  Re*  75  (4f)14(5d6s)7           60,68     |  Mc  115  (5f)14(6d)10(7s)2(7p)3  78,92
!  Kr   36  (3d)10(4s)2(4p)6  10,18,28  |  Os   76  (4f)14(5d)6(6s)2        60,68     |  Lv  116  (5f)14(6d)10(7s)2(7p)4  78,92
!  Rb   37  (5s)1             28,36     |  Ir   77  (4f)14(5d)7(6s)2        60,68     |  Ts  117  (5f)14(6d)10(7s)2(7p)5  78,92
!  Sr   38  (5s)2             28,36     |  Pt   78  (4f)14(5d)9(6s)1        60,68     |  Og  118  (5f)14(6d)10(7s)2(7p)6  78,92
!  Y*   39  (4d)2(5s)1        28,36     |  Au   79  (4f)14(5d)10(6s)1       60,68,78  |  Uue 119  (8s)1                   92
!  Zr*  40  (4d5s)4           28,36     |  Hg   80  (4f)14(5d)10(6s)2       60,68,78  |  Ubn 120  (8s)2                   92
! ================================================================================================================================
!
! where the core electrons are
!   Ncore =  2  :  [He]              10 :  [Ne]              18 :  [Ar]              28 :  [Ar]3d            36 :  [Kr]
!            46 :  [Kr]4d            54 :  [Xe]              60 :  [Kr]4d4f          68 :  [Xe]4f            78 :  [Xe]4f5d
!            92 :  [Xe]4f5d5f
!
!-----------------------------------------------------------------------------------------------------------------------------------
subroutine EDFLIB(iz,ncore,nfun,alf,coe)
implicit real(kind=8) (a-h,o-z)
real(kind=8) :: alf(*),coe(*)

nfun = 0

select case(ncore)
  case( 2)                                           ! iz =  3- 12
    call EDFLIB02(iz,nfun,alf,coe)
  case(10)                                           !      11- 36
    call EDFLIB10(iz,nfun,alf,coe)
  case(18)                                           !      19- 36
    call EDFLIB18(iz,nfun,alf,coe)
  case(28)                                           !      29- 71
    call EDFLIB28(iz,nfun,alf,coe)
  case(36)                                           !      37- 54
    call EDFLIB36(iz,nfun,alf,coe)
  case(46)                                           !      47- 58
    call EDFLIB46(iz,nfun,alf,coe)
  case(54)                                           !      55- 71
    call EDFLIB54(iz,nfun,alf,coe)
  case(60)                                           !      72-103
    call EDFLIB60(iz,nfun,alf,coe)
  case(68)                                           !      72- 86
    call EDFLIB68(iz,nfun,alf,coe)
  case(78)                                           !      79-118
    call EDFLIB78(iz,nfun,alf,coe)
  case(92)                                           !     104-120
    call EDFLIB92(iz,nfun,alf,coe)
end select

return
end

!-----------------------------------------------------------------------------------------------------------------------------------
!---  EDF data library: ncore =  2, iz =  3- 12
!-----------------------------------------------------------------------------------------------------------------------------------
subroutine EDFLIB02(iz,nfun,alf,coe)
implicit real(kind=8) (a-h,o-z)
real(kind=8) :: alf(*),coe(*)

select case(iz)
  case(003)
    nfun = 20
    alf(1:nfun)=(/  0.15030386867905D+05,  0.55208032572654D+04,  0.20278432533574D+04,  0.12289959111257D+04,&
                    0.74484600674283D+03,  0.45142182226838D+03,  0.27358898319296D+03,  0.16581150496543D+03,&
                    0.10049182119117D+03,  0.60904134055255D+02,  0.36911596397124D+02,  0.22370664483105D+02,&
                    0.13557978474609D+02,  0.82169566512784D+01,  0.49799737280475D+01,  0.30181658957864D+01,&
                    0.18291914519917D+01,  0.11086008799950D+01,  0.67187932120909D+00,  0.40719958861157D+00/)
    coe(1:nfun)=(/  0.58983031656026D+02, -0.72737203740697D+02,  0.78916995999469D+02, -0.77546249302926D+02,&
                    0.46106987161780D+02, -0.20331246939862D+02,  0.90112335307749D+01, -0.21982003785542D+01,&
                    0.21027216790180D+01,  0.91378912360460D+00,  0.15704978111214D+01,  0.15386441157294D+01,&
                    0.15500911596470D+01,  0.12954686098725D+01,  0.89513038241381D+00,  0.45694947100746D+00,&
                    0.14676766311980D+00,  0.21867597753859D-01,  0.95974765757623D-03,  0.11486635340285D-04/)
  case(004)
    nfun = 18
    alf(1:nfun)=(/  0.15030386867905D+05,  0.55208032572654D+04,  0.20278432533574D+04,  0.12289959111257D+04,&
                    0.74484600674283D+03,  0.45142182226838D+03,  0.27358898319296D+03,  0.16581150496543D+03,&
                    0.10049182119117D+03,  0.60904134055255D+02,  0.36911596397124D+02,  0.22370664483105D+02,&
                    0.13557978474609D+02,  0.82169566512784D+01,  0.49799737280475D+01,  0.30181658957864D+01,&
                    0.18291914519917D+01,  0.11086008799950D+01/)
    coe(1:nfun)=(/  0.14038650791839D+03, -0.17248626795239D+03,  0.18675085725554D+03, -0.18220217558033D+03,&
                    0.10865160557412D+03, -0.47086318884736D+02,  0.21653690137839D+02, -0.43502525474141D+01,&
                    0.56887833542260D+01,  0.29505301239060D+01,  0.42556803378525D+01,  0.37709025931892D+01,&
                    0.31807010679969D+01,  0.20496022004473D+01,  0.94202176161338D+00,  0.24155902819972D+00,&
                    0.24681091288940D-01,  0.59203514952033D-03/)
  case(005)
    nfun = 19
    alf(1:nfun)=(/  0.91093253744880D+04,  0.55208032572654D+04,  0.33459413680397D+04,  0.20278432533574D+04,&
                    0.12289959111257D+04,  0.74484600674283D+03,  0.45142182226838D+03,  0.27358898319296D+03,&
                    0.16581150496543D+03,  0.10049182119117D+03,  0.60904134055255D+02,  0.36911596397124D+02,&
                    0.22370664483105D+02,  0.13557978474609D+02,  0.82169566512784D+01,  0.49799737280475D+01,&
                    0.30181658957864D+01,  0.18291914519917D+01,  0.11086008799950D+01/)
    coe(1:nfun)=(/  0.21232379469174D+03, -0.27611692254769D+03,  0.19031258266591D+03, -0.94236614678637D+02,&
                    0.44696020330404D+02, -0.12756414711890D+02,  0.10232989850287D+02,  0.34507598650307D+01,&
                    0.71363199593416D+01,  0.73039995581084D+01,  0.81682033571092D+01,  0.78913596076520D+01,&
                    0.66157675594535D+01,  0.43793462006678D+01,  0.20290109298211D+01,  0.52374081438737D+00,&
                    0.54188570911478D-01,  0.14212608948867D-02,  0.18875339302671D-04/)
  case(006)
    nfun = 19
    alf(1:nfun)=(/  0.15030386867905D+05,  0.91093253744880D+04,  0.55208032572654D+04,  0.33459413680397D+04,&
                    0.20278432533574D+04,  0.12289959111257D+04,  0.74484600674283D+03,  0.45142182226838D+03,&
                    0.27358898319296D+03,  0.16581150496543D+03,  0.10049182119117D+03,  0.60904134055255D+02,&
                    0.36911596397124D+02,  0.22370664483105D+02,  0.13557978474609D+02,  0.82169566512784D+01,&
                    0.49799737280475D+01,  0.30181658957864D+01,  0.18291914519917D+01/)
    coe(1:nfun)=(/  0.24431356399029D+02, -0.30375797638553D+02,  0.29418083664787D+02, -0.10858460465388D+02,&
                    0.11716345597961D+02,  0.30758523787372D+01,  0.85479164740682D+01,  0.86913694834366D+01,&
                    0.11156308784673D+02,  0.12810882765498D+02,  0.14207548067369D+02,  0.14304720249316D+02,&
                    0.12542692739861D+02,  0.88799283470084D+01,  0.45240132219303D+01,  0.13516631784194D+01,&
                    0.17565977810405D+00,  0.66741417660516D-02,  0.11807039242662D-03/)
  case(007)
    nfun = 20
    alf(1:nfun)=(/  0.24800138332043D+05,  0.15030386867905D+05,  0.91093253744880D+04,  0.55208032572654D+04,&
                    0.33459413680397D+04,  0.20278432533574D+04,  0.12289959111257D+04,  0.74484600674283D+03,&
                    0.45142182226838D+03,  0.27358898319296D+03,  0.16581150496543D+03,  0.10049182119117D+03,&
                    0.60904134055255D+02,  0.36911596397124D+02,  0.22370664483105D+02,  0.13557978474609D+02,&
                    0.82169566512784D+01,  0.49799737280475D+01,  0.30181658957864D+01,  0.18291914519917D+01/)
    coe(1:nfun)=(/  0.30045921853220D+02, -0.11767199135036D+02,  0.88263815356186D+00,  0.13759035021008D+02,&
                    0.38362862268696D+01,  0.95180166752543D+01,  0.11555269455435D+02,  0.13269863177738D+02,&
                    0.16893224070594D+02,  0.19605361320824D+02,  0.22337344390140D+02,  0.23387422438699D+02,&
                    0.21786681203474D+02,  0.16874732255375D+02,  0.98331751338930D+01,  0.36338028026544D+01,&
                    0.65282269776691D+00,  0.40452524456413D-01,  0.78461074311775D-03,  0.18473350505754D-04/)
  case(008)
    nfun = 19
    alf(1:nfun)=(/  0.24800138332043D+05,  0.15030386867905D+05,  0.91093253744880D+04,  0.55208032572654D+04,&
                    0.33459413680397D+04,  0.20278432533574D+04,  0.12289959111257D+04,  0.74484600674283D+03,&
                    0.45142182226838D+03,  0.27358898319296D+03,  0.16581150496543D+03,  0.10049182119117D+03,&
                    0.60904134055255D+02,  0.36911596397124D+02,  0.22370664483105D+02,  0.13557978474609D+02,&
                    0.82169566512784D+01,  0.49799737280475D+01,  0.30181658957864D+01/)
    coe(1:nfun)=(/  0.30024585443316D+02,  0.15735748658772D+02, -0.31068146914593D+02,  0.47666575273965D+02,&
                   -0.90738396160887D+01,  0.25632679540053D+02,  0.14718975228927D+02,  0.24885290614708D+02,&
                    0.27091530688565D+02,  0.32389923387196D+02,  0.34977198323204D+02,  0.34762604722453D+02,&
                    0.29566764055159D+02,  0.19892170635759D+02,  0.92266210713497D+01,  0.23554345760818D+01,&
                    0.24494068022429D+00,  0.75253219501619D-02,  0.19436172123388D-03/)
  case(009)
    nfun = 19
    alf(1:nfun)=(/  0.24800138332043D+05,  0.15030386867905D+05,  0.91093253744880D+04,  0.55208032572654D+04,&
                    0.33459413680397D+04,  0.20278432533574D+04,  0.12289959111257D+04,  0.74484600674283D+03,&
                    0.45142182226838D+03,  0.27358898319296D+03,  0.16581150496543D+03,  0.10049182119117D+03,&
                    0.60904134055255D+02,  0.36911596397124D+02,  0.22370664483105D+02,  0.13557978474609D+02,&
                    0.82169566512784D+01,  0.49799737280475D+01,  0.30181658957864D+01/)
    coe(1:nfun)=(/  0.52480276703485D+02,  0.13720820672461D+02, -0.35572727665771D+02,  0.66161429610394D+02,&
                   -0.86563592830207D+01,  0.38242532635282D+02,  0.24905391120876D+02,  0.38582046693977D+02,&
                    0.42491950708798D+02,  0.48917956253721D+02,  0.50992878504811D+02,  0.47453859712886D+02,&
                    0.36531771817179D+02,  0.20944465881057D+02,  0.74810803820294D+01,  0.12785542022189D+01,&
                    0.76004073551275D-01,  0.16301258720972D-02,  0.51176613033554D-04/)
  case(010)
    nfun = 22
    alf(1:nfun)=(/  0.11140532140483D+06,  0.67518376608988D+05,  0.40920228247872D+05,  0.24800138332043D+05,&
                    0.15030386867905D+05,  0.91093253744880D+04,  0.55208032572654D+04,  0.33459413680397D+04,&
                    0.20278432533574D+04,  0.12289959111257D+04,  0.74484600674283D+03,  0.45142182226838D+03,&
                    0.27358898319296D+03,  0.16581150496543D+03,  0.10049182119117D+03,  0.60904134055255D+02,&
                    0.36911596397124D+02,  0.22370664483105D+02,  0.13557978474609D+02,  0.82169566512784D+01,&
                    0.49799737280475D+01,  0.30181658957864D+01/)
    coe(1:nfun)=(/  0.39177175197471D+03, -0.37893713014154D+03,  0.17403199498681D+03, -0.56057715753559D+01,&
                    0.13445146938320D+01,  0.40363537505618D+02,  0.12945557065483D+02,  0.39026810376905D+02,&
                    0.33532315362478D+02,  0.48091531749742D+02,  0.53646998055490D+02,  0.63955884787458D+02,&
                    0.69424137066775D+02,  0.69863024369155D+02,  0.60429235034900D+02,  0.41655111281429D+02,&
                    0.19943032075624D+02,  0.53435688003036D+01,  0.59985664578895D+00,  0.21483577574088D-01,&
                    0.48021929175945D-03,  0.19895150820126D-04/)
  case(011)
    nfun = 22
    alf(1:nfun)=(/  0.11140532140483D+06,  0.67518376608988D+05,  0.40920228247872D+05,  0.24800138332043D+05,&
                    0.15030386867905D+05,  0.91093253744880D+04,  0.55208032572654D+04,  0.33459413680397D+04,&
                    0.20278432533574D+04,  0.12289959111257D+04,  0.74484600674283D+03,  0.45142182226838D+03,&
                    0.27358898319296D+03,  0.16581150496543D+03,  0.10049182119117D+03,  0.60904134055255D+02,&
                    0.36911596397124D+02,  0.22370664483105D+02,  0.13557978474609D+02,  0.82169566512784D+01,&
                    0.49799737280475D+01,  0.30181658957864D+01/)
    coe(1:nfun)=(/  0.64310617116699D+03, -0.70999965110025D+03,  0.42801537608611D+03, -0.14310564546869D+03,&
                    0.89497849832173D+02,  0.83682652657153D+01,  0.47549000273051D+02,  0.42008814503526D+02,&
                    0.56888295220604D+02,  0.65764745220396D+02,  0.78642004881607D+02,  0.89022512567102D+02,&
                    0.94866270133369D+02,  0.90514463114680D+02,  0.72618129966897D+02,  0.44245994454252D+02,&
                    0.17355372235049D+02,  0.34194241085531D+01,  0.24798907326655D+00,  0.64615358849096D-02,&
                    0.66303844022286D-04,  0.16064359580348D-04/)
  case(012)
    nfun = 22
    alf(1:nfun)=(/  0.11140532140483D+06,  0.67518376608988D+05,  0.40920228247872D+05,  0.24800138332043D+05,&
                    0.15030386867905D+05,  0.91093253744880D+04,  0.55208032572654D+04,  0.33459413680397D+04,&
                    0.20278432533574D+04,  0.12289959111257D+04,  0.74484600674283D+03,  0.45142182226838D+03,&
                    0.27358898319296D+03,  0.16581150496543D+03,  0.10049182119117D+03,  0.60904134055255D+02,&
                    0.36911596397124D+02,  0.22370664483105D+02,  0.13557978474609D+02,  0.82169566512784D+01,&
                    0.49799737280475D+01,  0.30181658957864D+01/)
    coe(1:nfun)=(/  0.84609031826351D+03, -0.91782612089231D+03,  0.54450017749774D+03, -0.16332079403696D+03,&
                    0.10221910833579D+03,  0.28175831710629D+02,  0.57894299100823D+02,  0.65179299650685D+02,&
                    0.77173943720816D+02,  0.93523114548880D+02,  0.10776082292166D+03,  0.12065829851853D+03,&
                    0.12383297158697D+03,  0.11230885699367D+03,  0.82772658845953D+02,  0.44106624155852D+02,&
                    0.13918719748982D+02,  0.19858959172912D+01,  0.92732190738488D-01,  0.22881284542251D-02,&
                   -0.27788213531110D-04,  0.99943257747537D-05/)
end select

return
end

!-----------------------------------------------------------------------------------------------------------------------------------
!---  EDF data library: ncore = 10, iz = 11- 36
!-----------------------------------------------------------------------------------------------------------------------------------
subroutine EDFLIB10(iz,nfun,alf,coe)
implicit real(kind=8) (a-h,o-z)
real(kind=8) :: alf(*),coe(*)

select case(iz)
  case(011)
    nfun = 27
    alf(1:nfun)=(/  0.11140532140483D+06,  0.67518376608988D+05,  0.40920228247872D+05,  0.24800138332043D+05,&
                    0.15030386867905D+05,  0.91093253744880D+04,  0.55208032572654D+04,  0.33459413680397D+04,&
                    0.20278432533574D+04,  0.12289959111257D+04,  0.74484600674283D+03,  0.45142182226838D+03,&
                    0.27358898319296D+03,  0.16581150496543D+03,  0.10049182119117D+03,  0.60904134055255D+02,&
                    0.36911596397124D+02,  0.22370664483105D+02,  0.13557978474609D+02,  0.82169566512784D+01,&
                    0.49799737280475D+01,  0.30181658957864D+01,  0.18291914519917D+01,  0.11086008799950D+01,&
                    0.67187932120909D+00,  0.40719958861157D+00,  0.24678762946156D+00/)
    coe(1:nfun)=(/  0.10610022659712D+04, -0.15128824328389D+04,  0.12375922040548D+04, -0.75991537966765D+03,&
                    0.49970269333944D+03, -0.23918917671544D+03,  0.19509779444197D+03, -0.37271068584872D+02,&
                    0.10532015693421D+03,  0.45854054476717D+02,  0.95668009221001D+02,  0.88522977244065D+02,&
                    0.10376520121115D+03,  0.94742915307572D+02,  0.77813180394030D+02,  0.45614512371561D+02,&
                    0.16294926109358D+02, -0.47626713265799D-01, -0.17551245523598D+01,  0.11132199719241D+01,&
                    0.29493349051562D+01,  0.23451209263125D+01,  0.10754491459785D+01,  0.27729001374463D+00,&
                    0.35089762836172D-01,  0.15210223223132D-02,  0.28190559410168D-04/)
  case(012)
    nfun = 26
    alf(1:nfun)=(/  0.11140532140483D+06,  0.67518376608988D+05,  0.40920228247872D+05,  0.24800138332043D+05,&
                    0.15030386867905D+05,  0.91093253744880D+04,  0.55208032572654D+04,  0.33459413680397D+04,&
                    0.20278432533574D+04,  0.12289959111257D+04,  0.74484600674283D+03,  0.45142182226838D+03,&
                    0.27358898319296D+03,  0.16581150496543D+03,  0.10049182119117D+03,  0.60904134055255D+02,&
                    0.36911596397124D+02,  0.22370664483105D+02,  0.13557978474609D+02,  0.82169566512784D+01,&
                    0.49799737280475D+01,  0.30181658957864D+01,  0.18291914519917D+01,  0.11086008799950D+01,&
                    0.67187932120909D+00,  0.40719958861157D+00/)
    coe(1:nfun)=(/  0.94726504261978D+03, -0.98969371836260D+03,  0.56699236450717D+03, -0.15383889928088D+03,&
                    0.92876447996125D+02,  0.40115397604182D+02,  0.56320223974530D+02,  0.71431389966514D+02,&
                    0.82186134595773D+02,  0.98731865131762D+02,  0.11583190231636D+03,  0.12808751533259D+03,&
                    0.13237956223883D+03,  0.11989229532236D+03,  0.87549069762041D+02,  0.45116464549825D+02,&
                    0.10309593833328D+02, -0.25742894881469D+01, -0.11074565234780D+01,  0.33491230940656D+01,&
                    0.43434567564235D+01,  0.26611528293858D+01,  0.85500946168796D+00,  0.12566418066591D+00,&
                    0.65706216433270D-02,  0.15310852110284D-03/)
  case(013)
    nfun = 26
    alf(1:nfun)=(/  0.11140532140483D+06,  0.67518376608988D+05,  0.40920228247872D+05,  0.24800138332043D+05,&
                    0.15030386867905D+05,  0.91093253744880D+04,  0.55208032572654D+04,  0.33459413680397D+04,&
                    0.20278432533574D+04,  0.12289959111257D+04,  0.74484600674283D+03,  0.45142182226838D+03,&
                    0.27358898319296D+03,  0.16581150496543D+03,  0.10049182119117D+03,  0.60904134055255D+02,&
                    0.36911596397124D+02,  0.22370664483105D+02,  0.13557978474609D+02,  0.82169566512784D+01,&
                    0.49799737280475D+01,  0.30181658957864D+01,  0.18291914519917D+01,  0.11086008799950D+01,&
                    0.67187932120909D+00,  0.40719958861157D+00/)
    coe(1:nfun)=(/  0.49079394540843D+03,  0.80313719565049D+02, -0.62401817305945D+03,  0.84636370021291D+03,&
                   -0.56614157484006D+03,  0.48034688820411D+03, -0.17065633950569D+03,  0.24174992378801D+03,&
                    0.31685054895468D+02,  0.17983359079785D+03,  0.13098138023948D+03,  0.18065156663035D+03,&
                    0.16167155749098D+03,  0.14568577330057D+03,  0.93895095736185D+02,  0.40706816510938D+02,&
                    0.40678196331228D+01, -0.42008419196859D+01,  0.11103381164558D+01,  0.60111007013377D+01,&
                    0.55664078218379D+01,  0.25289057779712D+01,  0.51725447209718D+00,  0.37474685886025D-01,&
                    0.95568729510020D-03,  0.10103048881938D-04/)
  case(014)
    nfun = 25
    alf(1:nfun)=(/  0.11140532140483D+06,  0.67518376608988D+05,  0.40920228247872D+05,  0.24800138332043D+05,&
                    0.15030386867905D+05,  0.91093253744880D+04,  0.55208032572654D+04,  0.33459413680397D+04,&
                    0.20278432533574D+04,  0.12289959111257D+04,  0.74484600674283D+03,  0.45142182226838D+03,&
                    0.27358898319296D+03,  0.16581150496543D+03,  0.10049182119117D+03,  0.60904134055255D+02,&
                    0.36911596397124D+02,  0.22370664483105D+02,  0.13557978474609D+02,  0.82169566512784D+01,&
                    0.49799737280475D+01,  0.30181658957864D+01,  0.18291914519917D+01,  0.11086008799950D+01,&
                    0.67187932120909D+00/)
    coe(1:nfun)=(/  0.23997291564941D+04, -0.31584094696045D+04,  0.24824263114929D+04, -0.14161632537842D+04,&
                    0.93305756568909D+03, -0.38782242584229D+03,  0.37444952297211D+03, -0.16008684158325D+02,&
                    0.23041436123848D+03,  0.13793514961004D+03,  0.22316902321577D+03,  0.20421518515050D+03,&
                    0.21078391750157D+03,  0.16181124888361D+03,  0.99367738021538D+02,  0.31886299785227D+02,&
                   -0.13857427318580D+01, -0.43193919127807D+01,  0.46937312679365D+01,  0.88678119213437D+01,&
                    0.62500410221255D+01,  0.19900399322614D+01,  0.23765317837206D+00,  0.84585512972808D-02,&
                    0.18277860186799D-03/)
  case(015)
    nfun = 29
    alf(1:nfun)=(/  0.50044662941568D+06,  0.30330098752465D+06,  0.18381878031797D+06,  0.11140532140483D+06,&
                    0.67518376608988D+05,  0.40920228247872D+05,  0.24800138332043D+05,  0.15030386867905D+05,&
                    0.91093253744880D+04,  0.55208032572654D+04,  0.33459413680397D+04,  0.20278432533574D+04,&
                    0.12289959111257D+04,  0.74484600674283D+03,  0.45142182226838D+03,  0.27358898319296D+03,&
                    0.16581150496543D+03,  0.10049182119117D+03,  0.60904134055255D+02,  0.36911596397124D+02,&
                    0.22370664483105D+02,  0.13557978474609D+02,  0.82169566512784D+01,  0.49799737280475D+01,&
                    0.30181658957864D+01,  0.18291914519917D+01,  0.11086008799950D+01,  0.67187932120909D+00,&
                    0.40719958861157D+00/)
    coe(1:nfun)=(/  0.96416220694780D+02,  0.19724303852394D+04, -0.37935622867495D+04,  0.40379923253544D+04,&
                   -0.30046624941118D+04,  0.21453237200920D+04, -0.11855588134872D+04,  0.86092089797091D+03,&
                   -0.31144275916531D+03,  0.38946101473295D+03,  0.33199168089428D+02,  0.27484104762180D+03,&
                    0.19186323668421D+03,  0.27805040093744D+03,  0.25705477609927D+03,  0.24925795186465D+03,&
                    0.18094897995130D+03,  0.96889925510271D+02,  0.22501097081147D+02, -0.65080236864410D+01,&
                   -0.20737094375161D+01,  0.91924178125034D+01,  0.11594055329403D+02,  0.61226585090719D+01,&
                    0.13021626022942D+01,  0.85150972805259D-01,  0.24917909088806D-02, -0.76201409871857D-04,&
                    0.13981420159404D-04/)
  case(016)
    nfun = 25
    alf(1:nfun)=(/  0.18381878031797D+06,  0.11140532140483D+06,  0.67518376608988D+05,  0.40920228247872D+05,&
                    0.24800138332043D+05,  0.15030386867905D+05,  0.91093253744880D+04,  0.55208032572654D+04,&
                    0.33459413680397D+04,  0.20278432533574D+04,  0.12289959111257D+04,  0.74484600674283D+03,&
                    0.45142182226838D+03,  0.27358898319296D+03,  0.16581150496543D+03,  0.10049182119117D+03,&
                    0.60904134055255D+02,  0.36911596397124D+02,  0.22370664483105D+02,  0.13557978474609D+02,&
                    0.82169566512784D+01,  0.49799737280475D+01,  0.30181658957864D+01,  0.18291914519917D+01,&
                    0.11086008799950D+01/)
    coe(1:nfun)=(/  0.11278206062317D+04, -0.14213970489502D+04,  0.14718984298706D+04, -0.87211665725708D+03,&
                    0.73958880996704D+03, -0.22625371932983D+03,  0.36697664451599D+03,  0.73829971313477D+02,&
                    0.27997856068611D+03,  0.22588849854469D+03,  0.30999509674311D+03,  0.31330867263675D+03,&
                    0.32869293794036D+03,  0.28128910364956D+03,  0.19850495252013D+03,  0.88393610302359D+02,&
                    0.12465481466614D+02, -0.99391864403151D+01,  0.23067421317101D+01,  0.14427125481656D+02,&
                    0.13612613815028D+02,  0.52549532666162D+01,  0.71597499603376D+00,  0.27054989069711D-01,&
                    0.70692791109650D-03/)
  case(017)
    nfun = 25
    alf(1:nfun)=(/  0.18381878031797D+06,  0.11140532140483D+06,  0.67518376608988D+05,  0.40920228247872D+05,&
                    0.24800138332043D+05,  0.15030386867905D+05,  0.91093253744880D+04,  0.55208032572654D+04,&
                    0.33459413680397D+04,  0.20278432533574D+04,  0.12289959111257D+04,  0.74484600674283D+03,&
                    0.45142182226838D+03,  0.27358898319296D+03,  0.16581150496543D+03,  0.10049182119117D+03,&
                    0.60904134055255D+02,  0.36911596397124D+02,  0.22370664483105D+02,  0.13557978474609D+02,&
                    0.82169566512784D+01,  0.49799737280475D+01,  0.30181658957864D+01,  0.18291914519917D+01,&
                    0.67187932120909D+00/)
    coe(1:nfun)=(/  0.23657751770020D+04, -0.38265813407898D+04,  0.40102751770020D+04, -0.27873534278870D+04,&
                    0.20704104042053D+04, -0.98142256546021D+03,  0.87792854118347D+03, -0.13925261926651D+03,&
                    0.48725195622444D+03,  0.21326946938038D+03,  0.42454632377625D+03,  0.36448140847683D+03,&
                    0.39993271611631D+03,  0.31455048507452D+03,  0.20885848337412D+03,  0.76535457368940D+02,&
                    0.17894758926705D+01, -0.10584631625563D+02,  0.85017793076113D+01,  0.20027009625046D+02,&
                    0.14433772398741D+02,  0.39824863937974D+01,  0.33821432539480D+00,  0.97947646354442D-02,&
                    0.29698367072228D-04/)
  case(018)
    nfun = 26
    alf(1:nfun)=(/  0.18381878031797D+06,  0.11140532140483D+06,  0.67518376608988D+05,  0.40920228247872D+05,&
                    0.24800138332043D+05,  0.15030386867905D+05,  0.91093253744880D+04,  0.55208032572654D+04,&
                    0.33459413680397D+04,  0.20278432533574D+04,  0.12289959111257D+04,  0.74484600674283D+03,&
                    0.45142182226838D+03,  0.27358898319296D+03,  0.16581150496543D+03,  0.10049182119117D+03,&
                    0.60904134055255D+02,  0.36911596397124D+02,  0.22370664483105D+02,  0.13557978474609D+02,&
                    0.82169566512784D+01,  0.49799737280475D+01,  0.30181658957864D+01,  0.18291914519917D+01,&
                    0.11086008799950D+01,  0.67187932120909D+00/)
    coe(1:nfun)=(/  0.17562941408055D+04, -0.26857190951519D+04,  0.29589765917566D+04, -0.19265973163918D+04,&
                    0.15703941800110D+04, -0.60317927222000D+03,  0.74168416593829D+03,  0.28553417488001D+02,&
                    0.50097459838993D+03,  0.32233512784890D+03,  0.49131475936066D+03,  0.45405143105818D+03,&
                    0.45842236139378D+03,  0.35270945986011D+03,  0.20895746707490D+03,  0.63030984511779D+02,&
                   -0.88445175441166D+01, -0.79210631298706D+01,  0.16276210022160D+02,  0.25376434784271D+02,&
                    0.13902103991900D+02,  0.26961431426665D+01,  0.13841931471731D+00,  0.42110954579471D-02,&
                   -0.29032593362821D-03,  0.32319166670680D-04/)
  case(019)
    nfun = 26
    alf(1:nfun)=(/  0.50044662941568D+06,  0.30330098752465D+06,  0.18381878031797D+06,  0.11140532140483D+06,&
                    0.67518376608988D+05,  0.40920228247872D+05,  0.24800138332043D+05,  0.15030386867905D+05,&
                    0.91093253744880D+04,  0.55208032572654D+04,  0.33459413680397D+04,  0.20278432533574D+04,&
                    0.12289959111257D+04,  0.74484600674283D+03,  0.45142182226838D+03,  0.27358898319296D+03,&
                    0.16581150496543D+03,  0.10049182119117D+03,  0.60904134055255D+02,  0.36911596397124D+02,&
                    0.22370664483105D+02,  0.13557978474609D+02,  0.82169566512784D+01,  0.49799737280475D+01,&
                    0.30181658957864D+01,  0.18291914519917D+01/)
    coe(1:nfun)=(/  0.81921821614653D+04, -0.88446109172404D+04,  0.45338815758526D+04, -0.72467363283038D+03,&
                   -0.23740499073267D+03,  0.90974653913826D+03, -0.31202690591663D+03,  0.66311305624619D+03,&
                    0.89837307145819D+02,  0.51274014521297D+03,  0.35412193563208D+03,  0.53791772293113D+03,&
                    0.51989060107828D+03,  0.57523695043556D+03,  0.50755210965464D+03,  0.39087273010553D+03,&
                    0.20087850031168D+03,  0.47359803920663D+02, -0.17567172725645D+02, -0.22764083338261D+01,&
                    0.25564909733217D+02,  0.29577918730382D+02,  0.12259451465097D+02,  0.16344018874286D+01,&
                    0.56454113240051D-01,  0.88436257153290D-03/)
  case(020)
    nfun = 27
    alf(1:nfun)=(/  0.50044662941568D+06,  0.30330098752465D+06,  0.18381878031797D+06,  0.11140532140483D+06,&
                    0.67518376608988D+05,  0.40920228247872D+05,  0.24800138332043D+05,  0.15030386867905D+05,&
                    0.91093253744880D+04,  0.55208032572654D+04,  0.33459413680397D+04,  0.20278432533574D+04,&
                    0.12289959111257D+04,  0.74484600674283D+03,  0.45142182226838D+03,  0.27358898319296D+03,&
                    0.16581150496543D+03,  0.10049182119117D+03,  0.60904134055255D+02,  0.36911596397124D+02,&
                    0.22370664483105D+02,  0.13557978474609D+02,  0.82169566512784D+01,  0.49799737280475D+01,&
                    0.30181658957864D+01,  0.18291914519917D+01,  0.11086008799950D+01/)
    coe(1:nfun)=(/  0.10082608639352D+05, -0.11752567547604D+05,  0.69025979824066D+04, -0.21063526220247D+04,&
                    0.59681133045256D+03,  0.55099228547513D+03, -0.36526647537947D+02,  0.61915972170793D+03,&
                    0.22278869418427D+03,  0.56996414896660D+03,  0.46344791718782D+03,  0.63552394849795D+03,&
                    0.62793220182275D+03,  0.66858365380223D+03,  0.57371328358332D+03,  0.41503437513394D+03,&
                    0.19007728452196D+03,  0.28374426337265D+02, -0.22617940114829D+02,  0.60843156699639D+01,&
                    0.35923830638006D+02,  0.32000143562950D+02,  0.99493727932941D+01,  0.90394422276643D+00,&
                    0.23675616433081D-01, -0.25161039484090D-03,  0.53667966801285D-04/)
  case(021)
    nfun = 29
    alf(1:nfun)=(/  0.13624659485842D+07,  0.82573693853586D+06,  0.50044662941568D+06,  0.30330098752465D+06,&
                    0.18381878031797D+06,  0.11140532140483D+06,  0.67518376608988D+05,  0.40920228247872D+05,&
                    0.24800138332043D+05,  0.15030386867905D+05,  0.91093253744880D+04,  0.55208032572654D+04,&
                    0.33459413680397D+04,  0.20278432533574D+04,  0.12289959111257D+04,  0.74484600674283D+03,&
                    0.45142182226838D+03,  0.27358898319296D+03,  0.16581150496543D+03,  0.10049182119117D+03,&
                    0.60904134055255D+02,  0.36911596397124D+02,  0.22370664483105D+02,  0.13557978474609D+02,&
                    0.82169566512784D+01,  0.49799737280475D+01,  0.30181658957864D+01,  0.18291914519917D+01,&
                    0.11086008799950D+01/)
    coe(1:nfun)=(/  0.25507597796693D+05, -0.72638534913123D+05,  0.85961278634697D+05, -0.60265925481826D+05,&
                    0.31311353279620D+05, -0.12897925587162D+05,  0.53473932818621D+04, -0.14487524849139D+04,&
                    0.98825005808286D+03,  0.23424106125813D+03,  0.54797150430735D+03,  0.54469380596769D+03,&
                    0.64012010887323D+03,  0.71667811565369D+03,  0.76432827092527D+03,  0.75782017739286D+03,&
                    0.64415979467968D+03,  0.42880837546070D+03,  0.17499799058764D+03,  0.79606694262111D+01,&
                   -0.23557811713077D+02,  0.17258300231765D+02,  0.46619034829333D+02,  0.32252252666232D+02,&
                    0.74897736731565D+01,  0.46077060025539D+00,  0.11959222937989D-01, -0.59183457114287D-03,&
                    0.72006058163154D-04/)
  case(022)
    nfun = 29
    alf(1:nfun)=(/  0.13624659485842D+07,  0.82573693853586D+06,  0.50044662941568D+06,  0.30330098752465D+06,&
                    0.18381878031797D+06,  0.11140532140483D+06,  0.67518376608988D+05,  0.40920228247872D+05,&
                    0.24800138332043D+05,  0.15030386867905D+05,  0.91093253744880D+04,  0.55208032572654D+04,&
                    0.33459413680397D+04,  0.20278432533574D+04,  0.12289959111257D+04,  0.74484600674283D+03,&
                    0.45142182226838D+03,  0.27358898319296D+03,  0.16581150496543D+03,  0.10049182119117D+03,&
                    0.60904134055255D+02,  0.36911596397124D+02,  0.22370664483105D+02,  0.13557978474609D+02,&
                    0.82169566512784D+01,  0.49799737280475D+01,  0.30181658957864D+01,  0.18291914519917D+01,&
                    0.11086008799950D+01/)
    coe(1:nfun)=(/  0.39220517925352D+05, -0.99061274706900D+05,  0.11344167827588D+06, -0.79770017234549D+05,&
                    0.42636517459318D+05, -0.18697308612064D+05,  0.83905722753331D+04, -0.28900242678616D+04,&
                    0.18630418704785D+04, -0.97332707493566D+02,  0.88205026726658D+03,  0.52978867740114D+03,&
                    0.83751192049799D+03,  0.80711748584447D+03,  0.91504609416419D+03,  0.84926036309684D+03,&
                    0.71291578361046D+03,  0.43368847257093D+03,  0.15493196486046D+03, -0.11751919802498D+02,&
                   -0.20319780015975D+02,  0.31208634170960D+02,  0.56394989603336D+02,  0.30514289054872D+02,&
                    0.52776874496323D+01,  0.22510492503039D+00,  0.60077436340151D-02, -0.39253072171874D-03,&
                    0.42901448297883D-04/)
  case(023)
    nfun = 28
    alf(1:nfun)=(/  0.13624659485842D+07,  0.82573693853586D+06,  0.50044662941568D+06,  0.30330098752465D+06,&
                    0.18381878031797D+06,  0.11140532140483D+06,  0.67518376608988D+05,  0.40920228247872D+05,&
                    0.24800138332043D+05,  0.15030386867905D+05,  0.91093253744880D+04,  0.55208032572654D+04,&
                    0.33459413680397D+04,  0.20278432533574D+04,  0.12289959111257D+04,  0.74484600674283D+03,&
                    0.45142182226838D+03,  0.27358898319296D+03,  0.16581150496543D+03,  0.10049182119117D+03,&
                    0.60904134055255D+02,  0.36911596397124D+02,  0.22370664483105D+02,  0.13557978474609D+02,&
                    0.82169566512784D+01,  0.49799737280475D+01,  0.30181658957864D+01,  0.18291914519917D+01/)
    coe(1:nfun)=(/  0.91128343995810D+04, -0.14802794414341D+05,  0.15524065148711D+05, -0.11380809667319D+05,&
                    0.84734912246764D+04, -0.46610249811113D+04,  0.35358946001381D+04, -0.12810736163855D+04,&
                    0.15885625339560D+04,  0.43113579504192D+02,  0.10411553132245D+04,  0.61514039147086D+03,&
                    0.10111815910542D+04,  0.93184516454139D+03,  0.10659445709235D+04,  0.94947300295590D+03,&
                    0.77382350736405D+03,  0.43169569363170D+03,  0.12928832879088D+03, -0.28730592684951D+02,&
                   -0.13028081790291D+02,  0.47789669258264D+02,  0.64183880253058D+02,  0.27280454264442D+02,&
                    0.34992434573686D+01,  0.11205147424113D+00,  0.18581082335771D-02,  0.16184427085619D-04/)
  case(024)
    nfun = 28
    alf(1:nfun)=(/  0.13624659485842D+07,  0.82573693853586D+06,  0.50044662941568D+06,  0.30330098752465D+06,&
                    0.18381878031797D+06,  0.11140532140483D+06,  0.67518376608988D+05,  0.40920228247872D+05,&
                    0.24800138332043D+05,  0.15030386867905D+05,  0.91093253744880D+04,  0.55208032572654D+04,&
                    0.33459413680397D+04,  0.20278432533574D+04,  0.12289959111257D+04,  0.74484600674283D+03,&
                    0.45142182226838D+03,  0.27358898319296D+03,  0.16581150496543D+03,  0.10049182119117D+03,&
                    0.60904134055255D+02,  0.36911596397124D+02,  0.22370664483105D+02,  0.13557978474609D+02,&
                    0.82169566512784D+01,  0.49799737280475D+01,  0.30181658957864D+01,  0.18291914519917D+01/)
    coe(1:nfun)=(/  0.59954168002605D+04, -0.12109797397554D+05,  0.14173251236022D+05, -0.10740889616460D+05,&
                    0.83922232199311D+04, -0.45482352564037D+04,  0.36441740951017D+04, -0.11937587542348D+04,&
                    0.17094841054212D+04,  0.15890751269460D+03,  0.11790501219109D+04,  0.75946606818121D+03,&
                    0.11711047313611D+04,  0.10911157732805D+04,  0.12145037521509D+04,  0.10586675850797D+04,&
                    0.82406061308339D+03,  0.42331231860214D+03,  0.98773912712692D+02, -0.41606819520461D+02,&
                   -0.16088521210513D+01,  0.66261460214374D+02,  0.69203448716653D+02,  0.23181002633288D+02,&
                    0.22041240497115D+01,  0.59644256070555D-01, -0.17922639490564D-03,  0.12763060414045D-03/)
  case(025)
    nfun = 28
    alf(1:nfun)=(/  0.13624659485842D+07,  0.82573693853586D+06,  0.50044662941568D+06,  0.30330098752465D+06,&
                    0.18381878031797D+06,  0.11140532140483D+06,  0.67518376608988D+05,  0.40920228247872D+05,&
                    0.24800138332043D+05,  0.15030386867905D+05,  0.91093253744880D+04,  0.55208032572654D+04,&
                    0.33459413680397D+04,  0.20278432533574D+04,  0.12289959111257D+04,  0.74484600674283D+03,&
                    0.45142182226838D+03,  0.27358898319296D+03,  0.16581150496543D+03,  0.10049182119117D+03,&
                    0.60904134055255D+02,  0.36911596397124D+02,  0.22370664483105D+02,  0.13557978474609D+02,&
                    0.82169566512784D+01,  0.49799737280475D+01,  0.30181658957864D+01,  0.18291914519917D+01/)
    coe(1:nfun)=(/  0.18707108187675D+03, -0.24451692616343D+04,  0.51629459616244D+04, -0.39218378983140D+04,&
                    0.41483445762396D+04, -0.18268111867309D+04,  0.22327642155141D+04, -0.20452694594860D+03,&
                    0.13394233599044D+04,  0.58156224994082D+03,  0.11687505996432D+04,  0.10177268855087D+04,&
                    0.12934981958065D+04,  0.12972350949415D+04,  0.13534641543620D+04,  0.11787097771380D+04,&
                    0.86027761977263D+03,  0.40955864384364D+03,  0.64217700368354D+02, -0.49218537939891D+02,&
                    0.13900548691661D+02,  0.85622448766567D+02,  0.71085861747702D+02,  0.18834060648612D+02,&
                    0.13282151623458D+01,  0.35580002281677D-01, -0.12300839608628D-02,  0.17760949338452D-03/)
  case(026)
    nfun = 27
    alf(1:nfun)=(/  0.82573693853586D+06,  0.50044662941568D+06,  0.30330098752465D+06,  0.18381878031797D+06,&
                    0.11140532140483D+06,  0.67518376608988D+05,  0.40920228247872D+05,  0.24800138332043D+05,&
                    0.15030386867905D+05,  0.91093253744880D+04,  0.55208032572654D+04,  0.33459413680397D+04,&
                    0.20278432533574D+04,  0.12289959111257D+04,  0.74484600674283D+03,  0.45142182226838D+03,&
                    0.27358898319296D+03,  0.16581150496543D+03,  0.10049182119117D+03,  0.60904134055255D+02,&
                    0.36911596397124D+02,  0.22370664483105D+02,  0.13557978474609D+02,  0.82169566512784D+01,&
                    0.49799737280475D+01,  0.30181658957864D+01,  0.18291914519917D+01/)
    coe(1:nfun)=(/  0.24706863876581D+04, -0.24618387249410D+04,  0.35586658916622D+04, -0.11938801196814D+04,&
                    0.18439836253822D+04,  0.20692310766876D+03,  0.11889750817977D+04,  0.76022677085549D+03,&
                    0.11533451801902D+04,  0.11028724603346D+04,  0.13389122332218D+04,  0.14091528582494D+04,&
                    0.15329491663844D+04,  0.14919420902979D+04,  0.13020695524581D+04,  0.88468056597418D+03,&
                    0.38870073564243D+03,  0.28158451586874D+02, -0.51494981164906D+02,  0.33853969837228D+02,&
                    0.10457261516614D+03,  0.69950959741637D+02,  0.14677470429720D+02,  0.77449783298880D+00,&
                    0.22237827278104D-01, -0.12685587487979D-02,  0.15373074108993D-03/)
  case(027)
    nfun = 27
    alf(1:nfun)=(/  0.82573693853586D+06,  0.50044662941568D+06,  0.30330098752465D+06,  0.18381878031797D+06,&
                    0.11140532140483D+06,  0.67518376608988D+05,  0.40920228247872D+05,  0.24800138332043D+05,&
                    0.15030386867905D+05,  0.91093253744880D+04,  0.55208032572654D+04,  0.33459413680397D+04,&
                    0.20278432533574D+04,  0.12289959111257D+04,  0.74484600674283D+03,  0.45142182226838D+03,&
                    0.27358898319296D+03,  0.16581150496543D+03,  0.10049182119117D+03,  0.60904134055255D+02,&
                    0.36911596397124D+02,  0.22370664483105D+02,  0.13557978474609D+02,  0.82169566512784D+01,&
                    0.49799737280475D+01,  0.30181658957864D+01,  0.18291914519917D+01/)
    coe(1:nfun)=(/  0.79257219650894D+04, -0.10636195561796D+05,  0.11462723463908D+05, -0.67476789322495D+04,&
                    0.56533539274782D+04, -0.18592854593098D+04,  0.26265710099339D+04,  0.19184814442508D+03,&
                    0.17463504972477D+04,  0.10594450305323D+04,  0.16787030497899D+04,  0.15457066670588D+04,&
                    0.17819450107145D+04,  0.16382784972742D+04,  0.14213187356929D+04,  0.89947882621763D+03,&
                    0.35943731158802D+03, -0.68764182522864D+01, -0.48375050195124D+02,  0.58217458248353D+02,&
                    0.12169269825963D+03,  0.66241525257585D+02,  0.11012710719707D+02,  0.44757026435224D+00,&
                    0.12623278368466D-01, -0.70290930652561D-03,  0.81859798529926D-04/)
  case(028)
    nfun = 26
    alf(1:nfun)=(/  0.82573693853586D+06,  0.50044662941568D+06,  0.30330098752465D+06,  0.18381878031797D+06,&
                    0.11140532140483D+06,  0.67518376608988D+05,  0.40920228247872D+05,  0.24800138332043D+05,&
                    0.15030386867905D+05,  0.91093253744880D+04,  0.55208032572654D+04,  0.33459413680397D+04,&
                    0.20278432533574D+04,  0.12289959111257D+04,  0.74484600674283D+03,  0.45142182226838D+03,&
                    0.27358898319296D+03,  0.16581150496543D+03,  0.10049182119117D+03,  0.60904134055255D+02,&
                    0.36911596397124D+02,  0.22370664483105D+02,  0.13557978474609D+02,  0.82169566512784D+01,&
                    0.49799737280475D+01,  0.30181658957864D+01/)
    coe(1:nfun)=(/  0.12534801648259D+05, -0.17009713411242D+05,  0.17530403097749D+05, -0.10828975037456D+05,&
                    0.85043497657925D+04, -0.32958598172069D+04,  0.37184775657356D+04, -0.13835192517564D+03,&
                    0.22328566978350D+04,  0.11144204821540D+04,  0.19952098119939D+04,  0.17287518595131D+04,&
                    0.20293747061766D+04,  0.17995996406806D+04,  0.15298845214247D+04,  0.90655856865129D+03,&
                    0.32109064773131D+03, -0.38702630587664D+02, -0.39763385615610D+02,  0.86692952264522D+02,&
                    0.13580143823663D+03,  0.60585059498911D+02,  0.79755476224473D+01,  0.26560231920587D+00,&
                    0.51202885229316D-02,  0.68812624370196D-04/)
  case(029)
    nfun = 26
    alf(1:nfun)=(/  0.82573693853586D+06,  0.50044662941568D+06,  0.30330098752465D+06,  0.18381878031797D+06,&
                    0.11140532140483D+06,  0.67518376608988D+05,  0.40920228247872D+05,  0.24800138332043D+05,&
                    0.15030386867905D+05,  0.91093253744880D+04,  0.55208032572654D+04,  0.33459413680397D+04,&
                    0.20278432533574D+04,  0.12289959111257D+04,  0.74484600674283D+03,  0.45142182226838D+03,&
                    0.27358898319296D+03,  0.16581150496543D+03,  0.10049182119117D+03,  0.60904134055255D+02,&
                    0.36911596397124D+02,  0.22370664483105D+02,  0.13557978474609D+02,  0.82169566512784D+01,&
                    0.49799737280475D+01,  0.30181658957864D+01/)
    coe(1:nfun)=(/  0.13703402264103D+05, -0.19586353114545D+05,  0.20746028412521D+05, -0.13078577505678D+05,&
                    0.10298789914042D+05, -0.40920911846012D+04,  0.44786967611834D+04, -0.24243400495499D+03,&
                    0.26241306543909D+04,  0.12636030490007D+04,  0.22921397611368D+04,  0.19574822208355D+04,&
                    0.22749872514263D+04,  0.19742363333369D+04,  0.16259655916994D+04,  0.90532347565514D+03,&
                    0.27480115516981D+03, -0.65963903057433D+02, -0.25274058745301D+02,  0.11821514709266D+03,&
                    0.14608243057303D+03,  0.53695676688941D+02,  0.56074070168529D+01,  0.16299181461306D+00,&
                    0.10500596753786D-02,  0.26305767413604D-03/)
  case(030)
    nfun = 26
    alf(1:nfun)=(/  0.82573693853586D+06,  0.50044662941568D+06,  0.30330098752465D+06,  0.18381878031797D+06,&
                    0.11140532140483D+06,  0.67518376608988D+05,  0.40920228247872D+05,  0.24800138332043D+05,&
                    0.15030386867905D+05,  0.91093253744880D+04,  0.55208032572654D+04,  0.33459413680397D+04,&
                    0.20278432533574D+04,  0.12289959111257D+04,  0.74484600674283D+03,  0.45142182226838D+03,&
                    0.27358898319296D+03,  0.16581150496543D+03,  0.10049182119117D+03,  0.60904134055255D+02,&
                    0.36911596397124D+02,  0.22370664483105D+02,  0.13557978474609D+02,  0.82169566512784D+01,&
                    0.49799737280475D+01,  0.30181658957864D+01/)
    coe(1:nfun)=(/  0.12557833690047D+05, -0.17241056515962D+05,  0.18867020988286D+05, -0.11303234146148D+05,&
                    0.94419048957974D+04, -0.32172333360240D+04,  0.42943204103224D+04,  0.23586265793815D+03,&
                    0.27226497189943D+04,  0.16192454960002D+04,  0.25094099472938D+04,  0.22657412148721D+04,&
                    0.24997752352918D+04,  0.21704311502138D+04,  0.17026824855528D+04,  0.89810549721069D+03,&
                    0.22057746488292D+03, -0.86873374984651D+02, -0.49571935986824D+01,  0.15174845232897D+03,&
                    0.15204476194795D+03,  0.46268101991892D+02,  0.38360028138115D+01,  0.10652305465374D+00,&
                   -0.16061894387747D-02,  0.38072048895188D-03/)
  case(031)
    nfun = 28
    alf(1:nfun)=(/  0.22480688151639D+07,  0.13624659485842D+07,  0.82573693853586D+06,  0.50044662941568D+06,&
                    0.30330098752465D+06,  0.18381878031797D+06,  0.11140532140483D+06,  0.67518376608988D+05,&
                    0.40920228247872D+05,  0.24800138332043D+05,  0.15030386867905D+05,  0.91093253744880D+04,&
                    0.55208032572654D+04,  0.33459413680397D+04,  0.20278432533574D+04,  0.12289959111257D+04,&
                    0.74484600674283D+03,  0.45142182226838D+03,  0.27358898319296D+03,  0.16581150496543D+03,&
                    0.10049182119117D+03,  0.60904134055255D+02,  0.36911596397124D+02,  0.22370664483105D+02,&
                    0.13557978474609D+02,  0.82169566512784D+01,  0.49799737280475D+01,  0.30181658957864D+01/)
    coe(1:nfun)=(/  0.91034303364277D+05, -0.12030645773596D+06,  0.85805026150465D+05, -0.42574325531125D+05,&
                    0.21330931062400D+05, -0.65154702226818D+04,  0.48330453977883D+04,  0.50105651279539D+03,&
                    0.23635818798989D+04,  0.17649775908682D+04,  0.22856937885415D+04,  0.23007273387015D+04,&
                    0.25957632714827D+04,  0.26743082313356D+04,  0.26963353109782D+04,  0.23874382480129D+04,&
                    0.17592188545038D+04,  0.88387524832180D+03,  0.16053138151730D+03, -0.10080422462165D+03,&
                    0.21519581398028D+02,  0.18591133614705D+03,  0.15366908846758D+03,  0.38840416850644D+02,&
                    0.25616747459146D+01,  0.72718628914345D-01, -0.27756535342576D-02,  0.40178479527953D-03/)
  case(032)
    nfun = 28
    alf(1:nfun)=(/  0.22480688151639D+07,  0.13624659485842D+07,  0.82573693853586D+06,  0.50044662941568D+06,&
                    0.30330098752465D+06,  0.18381878031797D+06,  0.11140532140483D+06,  0.67518376608988D+05,&
                    0.40920228247872D+05,  0.24800138332043D+05,  0.15030386867905D+05,  0.91093253744880D+04,&
                    0.55208032572654D+04,  0.33459413680397D+04,  0.20278432533574D+04,  0.12289959111257D+04,&
                    0.74484600674283D+03,  0.45142182226838D+03,  0.27358898319296D+03,  0.16581150496543D+03,&
                    0.10049182119117D+03,  0.60904134055255D+02,  0.36911596397124D+02,  0.22370664483105D+02,&
                    0.13557978474609D+02,  0.82169566512784D+01,  0.49799737280475D+01,  0.30181658957864D+01/)
    coe(1:nfun)=(/  0.90950623363197D+05, -0.11109557725573D+06,  0.71577985370278D+05, -0.28441494809210D+05,&
                    0.11163620353192D+05,  0.63676621598005D+03,  0.83631860722601D+03,  0.32873774619922D+04,&
                    0.11786690705568D+04,  0.28881222850755D+04,  0.21176349842893D+04,  0.28844715091158D+04,&
                    0.27720871731611D+04,  0.30712900569278D+04,  0.29196370614112D+04,  0.25959164427210D+04,&
                    0.18074987219541D+04,  0.85634303307929D+03,  0.99392011834483D+02, -0.10840086970230D+03,&
                    0.54763739224025D+02,  0.21900692163416D+03,  0.15130011621201D+03,  0.31808945909670D+02,&
                    0.16809586283100D+01,  0.49160041539153D-01, -0.26638368429133D-02,  0.32880604669572D-03/)
  case(033)
    nfun = 28
    alf(1:nfun)=(/  0.22480688151639D+07,  0.13624659485842D+07,  0.82573693853586D+06,  0.50044662941568D+06,&
                    0.30330098752465D+06,  0.18381878031797D+06,  0.11140532140483D+06,  0.67518376608988D+05,&
                    0.40920228247872D+05,  0.24800138332043D+05,  0.15030386867905D+05,  0.91093253744880D+04,&
                    0.55208032572654D+04,  0.33459413680397D+04,  0.20278432533574D+04,  0.12289959111257D+04,&
                    0.74484600674283D+03,  0.45142182226838D+03,  0.27358898319296D+03,  0.16581150496543D+03,&
                    0.10049182119117D+03,  0.60904134055255D+02,  0.36911596397124D+02,  0.22370664483105D+02,&
                    0.13557978474609D+02,  0.82169566512784D+01,  0.49799737280475D+01,  0.30181658957864D+01/)
    coe(1:nfun)=(/  0.96679488918781D+05, -0.11007114278907D+06,  0.64580283375680D+05, -0.19299269705892D+05,&
                    0.41803840709627D+04,  0.59553584636450D+04, -0.20564486911297D+04,  0.54995786280632D+04,&
                    0.36000844554976D+03,  0.38609953098483D+04,  0.20755828363476D+04,  0.34494329072866D+04,&
                    0.29988507527071D+04,  0.34770442271587D+04,  0.31585445852315D+04,  0.27984257855206D+04,&
                    0.18445017949131D+04,  0.81649827046315D+03,  0.38667567788587D+02, -0.10892847274114D+03,&
                    0.94515428104820D+02,  0.24962151197127D+03,  0.14544246996120D+03,  0.25450055443259D+02,&
                    0.10936534908351D+01,  0.30729842183575D-01, -0.17313058539088D-02,  0.19787582979447D-03/)
  case(034)
    nfun = 28
    alf(1:nfun)=(/  0.22480688151639D+07,  0.13624659485842D+07,  0.82573693853586D+06,  0.50044662941568D+06,&
                    0.30330098752465D+06,  0.18381878031797D+06,  0.11140532140483D+06,  0.67518376608988D+05,&
                    0.40920228247872D+05,  0.24800138332043D+05,  0.15030386867905D+05,  0.91093253744880D+04,&
                    0.55208032572654D+04,  0.33459413680397D+04,  0.20278432533574D+04,  0.12289959111257D+04,&
                    0.74484600674283D+03,  0.45142182226838D+03,  0.27358898319296D+03,  0.16581150496543D+03,&
                    0.10049182119117D+03,  0.60904134055255D+02,  0.36911596397124D+02,  0.22370664483105D+02,&
                    0.13557978474609D+02,  0.82169566512784D+01,  0.49799737280475D+01,  0.30181658957864D+01/)
    coe(1:nfun)=(/  0.10652328467056D+06, -0.11544433240432D+06,  0.63490836389542D+05, -0.14331254239976D+05,&
                   -0.88306643724442D+02,  0.97038025917262D+04, -0.39796958525628D+04,  0.72054399477616D+04,&
                   -0.11480151532963D+03,  0.46894487344678D+04,  0.21672801006781D+04,  0.39882072327817D+04,&
                    0.32852493842586D+04,  0.38843897876255D+04,  0.34167220705828D+04,  0.29896526422589D+04,&
                    0.18712511357967D+04,  0.76378649404674D+03, -0.19351464686588D+02, -0.10207263093417D+03,&
                    0.14035805396190D+03,  0.27645393094430D+03,  0.13676638976224D+03,  0.19916170014830D+02,&
                    0.71397750951735D+00,  0.15833278125456D-01, -0.47634170613264D-03,  0.52054427138515D-04/)
  case(035)
    nfun = 28
    alf(1:nfun)=(/  0.37093135450204D+07,  0.22480688151639D+07,  0.13624659485842D+07,  0.82573693853586D+06,&
                    0.50044662941568D+06,  0.30330098752465D+06,  0.18381878031797D+06,  0.11140532140483D+06,&
                    0.67518376608988D+05,  0.40920228247872D+05,  0.24800138332043D+05,  0.15030386867905D+05,&
                    0.91093253744880D+04,  0.55208032572654D+04,  0.33459413680397D+04,  0.20278432533574D+04,&
                    0.12289959111257D+04,  0.74484600674283D+03,  0.45142182226838D+03,  0.27358898319296D+03,&
                    0.16581150496543D+03,  0.10049182119117D+03,  0.60904134055255D+02,  0.36911596397124D+02,&
                    0.22370664483105D+02,  0.13557978474609D+02,  0.82169566512784D+01,  0.49799737280475D+01/)
    coe(1:nfun)=(/  0.74101594910979D+05, -0.12267678624415D+06,  0.12557890702701D+06, -0.88905468570709D+05,&
                    0.62453106315851D+05, -0.33282263321579D+05,  0.23802193374217D+05, -0.90539602013230D+04,&
                    0.97361154081821D+04, -0.63293983267993D+03,  0.54715442636199D+04,  0.23840602731891D+04,&
                    0.44928088053335D+04,  0.36429178019101D+04,  0.42845810260405D+04,  0.36981071973974D+04,&
                    0.31643961231295D+04,  0.18886859353340D+04,  0.69816079621602D+03, -0.72635674032354D+02,&
                   -0.87469713794781D+02,  0.19150337204991D+03,  0.29848529490475D+03,  0.12599547888958D+03,&
                    0.15268885256759D+02,  0.46962775542806D+00,  0.60152717354072D-02,  0.21178138483435D-03/)
  case(036)
    nfun = 28
    alf(1:nfun)=(/  0.37093135450204D+07,  0.22480688151639D+07,  0.13624659485842D+07,  0.82573693853586D+06,&
                    0.50044662941568D+06,  0.30330098752465D+06,  0.18381878031797D+06,  0.11140532140483D+06,&
                    0.67518376608988D+05,  0.40920228247872D+05,  0.24800138332043D+05,  0.15030386867905D+05,&
                    0.91093253744880D+04,  0.55208032572654D+04,  0.33459413680397D+04,  0.20278432533574D+04,&
                    0.12289959111257D+04,  0.74484600674283D+03,  0.45142182226838D+03,  0.27358898319296D+03,&
                    0.16581150496543D+03,  0.10049182119117D+03,  0.60904134055255D+02,  0.36911596397124D+02,&
                    0.22370664483105D+02,  0.13557978474609D+02,  0.82169566512784D+01,  0.49799737280475D+01/)
    coe(1:nfun)=(/  0.67169910450220D+05, -0.11644417661572D+06,  0.12283562803459D+06, -0.87097971702337D+05,&
                    0.62233690607786D+05, -0.32581158122778D+05,  0.24031688708901D+05, -0.85391603816748D+04,&
                    0.10054720398724D+05, -0.15179172677547D+03,  0.58489014723636D+04,  0.28576824758425D+04,&
                    0.49141820519054D+04,  0.40937372754207D+04,  0.46670066188518D+04,  0.40059838389286D+04,&
                    0.33182448584648D+04,  0.18973297642705D+04,  0.62020396351811D+03, -0.11945789179417D+03,&
                   -0.64763346939624D+02,  0.24688110537940D+03,  0.31497076390002D+03,  0.11388173769420D+03,&
                    0.11477862915954D+02,  0.31415499257628D+00, -0.11670899851879D-03,  0.50664785775467D-03/)
end select

return
end

!-----------------------------------------------------------------------------------------------------------------------------------
!---  EDF data library: ncore = 18, iz = 19- 36
!-----------------------------------------------------------------------------------------------------------------------------------
subroutine EDFLIB18(iz,nfun,alf,coe)
implicit real(kind=8) (a-h,o-z)
real(kind=8) :: alf(*),coe(*)

select case(iz)
  case(019)
    nfun = 36
    alf(1:nfun)=(/  0.61203673492837D+07,  0.37093135450204D+07,  0.22480688151639D+07,  0.13624659485842D+07,&
                    0.82573693853586D+06,  0.50044662941568D+06,  0.30330098752465D+06,  0.18381878031797D+06,&
                    0.11140532140483D+06,  0.67518376608988D+05,  0.40920228247872D+05,  0.24800138332043D+05,&
                    0.15030386867905D+05,  0.91093253744880D+04,  0.55208032572654D+04,  0.33459413680397D+04,&
                    0.20278432533574D+04,  0.12289959111257D+04,  0.74484600674283D+03,  0.45142182226838D+03,&
                    0.27358898319296D+03,  0.16581150496543D+03,  0.10049182119117D+03,  0.60904134055255D+02,&
                    0.36911596397124D+02,  0.22370664483105D+02,  0.13557978474609D+02,  0.82169566512784D+01,&
                    0.49799737280475D+01,  0.30181658957864D+01,  0.18291914519917D+01,  0.11086008799950D+01,&
                    0.67187932120909D+00,  0.40719958861157D+00,  0.24678762946156D+00,  0.14956826027973D+00/)
    coe(1:nfun)=(/ -0.10284267440557D+05,  0.53264242936134D+05, -0.65038826420784D+05,  0.55132298574209D+05,&
                   -0.37868959568501D+05,  0.24054719187498D+05, -0.14004229697466D+05,  0.83488096394241D+04,&
                   -0.44337580404133D+04,  0.27997489155531D+04, -0.12191398699284D+04,  0.10547010234445D+04,&
                   -0.15854357408173D+03,  0.57311903591547D+03,  0.24170367244026D+03,  0.51243752752524D+03,&
                    0.45568093649857D+03,  0.57287816341640D+03,  0.55344882289635D+03,  0.52699211953222D+03,&
                    0.38580255658383D+03,  0.20593653400028D+03,  0.42467139879875D+02, -0.19596616871106D+02,&
                   -0.33740895968258D+01,  0.29833847264547D+02,  0.34386325781897D+02,  0.13695649049048D+02,&
                   -0.14894393627852D+01, -0.17786033074900D+01,  0.64487396837541D+00,  0.90126561923344D+00,&
                    0.34417075470154D+00,  0.52599196477569D-01,  0.25245856748493D-02,  0.43295829941559D-04/)
  case(020)
    nfun = 35
    alf(1:nfun)=(/  0.61203673492837D+07,  0.37093135450204D+07,  0.22480688151639D+07,  0.13624659485842D+07,&
                    0.82573693853586D+06,  0.50044662941568D+06,  0.30330098752465D+06,  0.18381878031797D+06,&
                    0.11140532140483D+06,  0.67518376608988D+05,  0.40920228247872D+05,  0.24800138332043D+05,&
                    0.15030386867905D+05,  0.91093253744880D+04,  0.55208032572654D+04,  0.33459413680397D+04,&
                    0.20278432533574D+04,  0.12289959111257D+04,  0.74484600674283D+03,  0.45142182226838D+03,&
                    0.27358898319296D+03,  0.16581150496543D+03,  0.10049182119117D+03,  0.60904134055255D+02,&
                    0.36911596397124D+02,  0.22370664483105D+02,  0.13557978474609D+02,  0.82169566512784D+01,&
                    0.49799737280475D+01,  0.30181658957864D+01,  0.18291914519917D+01,  0.11086008799950D+01,&
                    0.67187932120909D+00,  0.40719958861157D+00,  0.24678762946156D+00/)
    coe(1:nfun)=(/  0.20623126139688D+06, -0.15496777391148D+06,  0.76749452705383D+05, -0.24675644781113D+05,&
                    0.60783941223621D+04,  0.44605751037598D+03, -0.10243988527060D+04,  0.13226551291347D+04,&
                   -0.48361479422450D+03,  0.69777040703595D+03,  0.29955224171281D+02,  0.45392224097997D+03,&
                    0.27928908802941D+03,  0.44260931722261D+03,  0.44624892840348D+03,  0.54221699220547D+03,&
                    0.59970343050873D+03,  0.65759521366074D+03,  0.66216989670938D+03,  0.58657941909379D+03,&
                    0.41451838440445D+03,  0.19194862176632D+03,  0.23265300958563D+02, -0.26058689790141D+02,&
                    0.68260804935198D+01,  0.42449614515381D+02,  0.38310871580788D+02,  0.95492159345990D+01,&
                   -0.38656819669602D+01, -0.11868292466681D+01,  0.14257162557348D+01,  0.10221988025464D+01,&
                    0.22904534565367D+00,  0.15783263146240D-01,  0.43908201772735D-03/)
  case(021)
    nfun = 36
    alf(1:nfun)=(/  0.10098606126318D+08,  0.61203673492837D+07,  0.37093135450204D+07,  0.22480688151639D+07,&
                    0.13624659485842D+07,  0.82573693853586D+06,  0.50044662941568D+06,  0.30330098752465D+06,&
                    0.18381878031797D+06,  0.11140532140483D+06,  0.67518376608988D+05,  0.40920228247872D+05,&
                    0.24800138332043D+05,  0.15030386867905D+05,  0.91093253744880D+04,  0.55208032572654D+04,&
                    0.33459413680397D+04,  0.20278432533574D+04,  0.12289959111257D+04,  0.74484600674283D+03,&
                    0.45142182226838D+03,  0.27358898319296D+03,  0.16581150496543D+03,  0.10049182119117D+03,&
                    0.60904134055255D+02,  0.36911596397124D+02,  0.22370664483105D+02,  0.13557978474609D+02,&
                    0.82169566512784D+01,  0.49799737280475D+01,  0.30181658957864D+01,  0.18291914519917D+01,&
                    0.11086008799950D+01,  0.67187932120909D+00,  0.40719958861157D+00,  0.24678762946156D+00/)
    coe(1:nfun)=(/  0.36720608641529D+06, -0.20313366131687D+06,  0.37700553151131D+05,  0.43087012607574D+05,&
                   -0.51039958721638D+05,  0.40132168445349D+05, -0.25613045923829D+05,  0.15965433634281D+05,&
                   -0.88291017700434D+04,  0.54885095518529D+04, -0.26011041471064D+04,  0.19806021200493D+04,&
                   -0.52389277271554D+03,  0.94135603377968D+03,  0.21021955582500D+03,  0.72490171643067D+03,&
                    0.55463872443466D+03,  0.77343563918374D+03,  0.74523817837378D+03,  0.78053305966908D+03,&
                    0.64248680996511D+03,  0.43666298959902D+03,  0.17100493784514D+03,  0.39025104475659D+01,&
                   -0.28433343296798D+02,  0.20874947511767D+02,  0.55289259835609D+02,  0.38994633296462D+02,&
                    0.42374512163460D+01, -0.49233325645757D+01, -0.27603583410155D-01,  0.18996644468320D+01,&
                    0.89595762451327D+00,  0.15240774887852D+00,  0.85112193695807D-02,  0.24919965078229D-03/)
  case(022)
    nfun = 31
    alf(1:nfun)=(/  0.82573693853586D+06,  0.50044662941568D+06,  0.30330098752465D+06,  0.18381878031797D+06,&
                    0.11140532140483D+06,  0.67518376608988D+05,  0.40920228247872D+05,  0.24800138332043D+05,&
                    0.15030386867905D+05,  0.91093253744880D+04,  0.55208032572654D+04,  0.33459413680397D+04,&
                    0.20278432533574D+04,  0.12289959111257D+04,  0.74484600674283D+03,  0.45142182226838D+03,&
                    0.27358898319296D+03,  0.16581150496543D+03,  0.10049182119117D+03,  0.60904134055255D+02,&
                    0.36911596397124D+02,  0.22370664483105D+02,  0.13557978474609D+02,  0.82169566512784D+01,&
                    0.49799737280475D+01,  0.30181658957864D+01,  0.18291914519917D+01,  0.11086008799950D+01,&
                    0.67187932120909D+00,  0.40719958861157D+00,  0.24678762946156D+00/)
    coe(1:nfun)=(/  0.15879486573994D+05, -0.18161696065754D+05,  0.15290193664998D+05, -0.95810346750617D+04,&
                    0.64444088823050D+04, -0.31520431842357D+04,  0.24427948407903D+04, -0.66078081538528D+03,&
                    0.11562977601569D+04,  0.24943729313836D+03,  0.87350177586451D+03,  0.66475571891712D+03,&
                    0.91464410994621D+03,  0.87078722107213D+03,  0.88837009571848D+03,  0.70386394686787D+03,&
                    0.44564693539806D+03,  0.14686995827843D+03, -0.16136854859746D+02, -0.25346308049015D+02,&
                    0.38097346430464D+02,  0.67749688473117D+02,  0.36419793438020D+02, -0.12472171423576D+01,&
                   -0.49623102933460D+01,  0.12829913915724D+01,  0.21819279362881D+01,  0.73921501530702D+00,&
                    0.91325526425742D-01,  0.35928375872363D-02,  0.11525882398500D-03/)
  case(023)
    nfun = 33
    alf(1:nfun)=(/  0.22480688151639D+07,  0.13624659485842D+07,  0.82573693853586D+06,  0.50044662941568D+06,&
                    0.30330098752465D+06,  0.18381878031797D+06,  0.11140532140483D+06,  0.67518376608988D+05,&
                    0.40920228247872D+05,  0.24800138332043D+05,  0.15030386867905D+05,  0.91093253744880D+04,&
                    0.55208032572654D+04,  0.33459413680397D+04,  0.20278432533574D+04,  0.12289959111257D+04,&
                    0.74484600674283D+03,  0.45142182226838D+03,  0.27358898319296D+03,  0.16581150496543D+03,&
                    0.10049182119117D+03,  0.60904134055255D+02,  0.36911596397124D+02,  0.22370664483105D+02,&
                    0.13557978474609D+02,  0.82169566512784D+01,  0.49799737280475D+01,  0.30181658957864D+01,&
                    0.18291914519917D+01,  0.11086008799950D+01,  0.67187932120909D+00,  0.40719958861157D+00,&
                    0.24678762946156D+00/)
    coe(1:nfun)=(/  0.14228902587891D+05, -0.16362531005859D+05,  0.14282420410156D+05, -0.90454221191406D+04,&
                    0.62854768676758D+04, -0.30458516845703D+04,  0.24302147521973D+04, -0.64461183166504D+03,&
                    0.11345937652588D+04,  0.22931031799316D+03,  0.81588371276855D+03,  0.62182390594482D+03,&
                    0.86122656345367D+03,  0.88965236997604D+03,  0.10157382433414D+04,  0.10373422586918D+04,&
                    0.98263741534948D+03,  0.76951910349727D+03,  0.44057429663837D+03,  0.12053503043205D+03,&
                   -0.35359710391611D+02, -0.16244453253224D+02,  0.58028881059960D+02,  0.78275007601827D+02,&
                    0.30570321861887D+02, -0.59416700183065D+01, -0.39591464802506D+01,  0.24983715466515D+01,&
                    0.22029014202890D+01,  0.57732616191242D+00,  0.57518930944298D-01,  0.18959848572422D-02,&
                    0.66198182622657D-04/)
  case(024)
    nfun = 34
    alf(1:nfun)=(/  0.37093135450204D+07,  0.22480688151639D+07,  0.13624659485842D+07,  0.82573693853586D+06,&
                    0.50044662941568D+06,  0.30330098752465D+06,  0.18381878031797D+06,  0.11140532140483D+06,&
                    0.67518376608988D+05,  0.40920228247872D+05,  0.24800138332043D+05,  0.15030386867905D+05,&
                    0.91093253744880D+04,  0.55208032572654D+04,  0.33459413680397D+04,  0.20278432533574D+04,&
                    0.12289959111257D+04,  0.74484600674283D+03,  0.45142182226838D+03,  0.27358898319296D+03,&
                    0.16581150496543D+03,  0.10049182119117D+03,  0.60904134055255D+02,  0.36911596397124D+02,&
                    0.22370664483105D+02,  0.13557978474609D+02,  0.82169566512784D+01,  0.49799737280475D+01,&
                    0.30181658957864D+01,  0.18291914519917D+01,  0.11086008799950D+01,  0.67187932120909D+00,&
                    0.40719958861157D+00,  0.24678762946156D+00/)
    coe(1:nfun)=(/  0.43157635200262D+05, -0.51138540953994D+05,  0.42258834789038D+05, -0.26970939736366D+05,&
                    0.17069251677513D+05, -0.90706815548539D+04,  0.59242122755051D+04, -0.24995206408352D+04,&
                    0.22642588311806D+04, -0.32700883699208D+03,  0.12103833908997D+04,  0.45772163376771D+03,&
                    0.10251171448464D+04,  0.86140146842925D+03,  0.11328780642091D+04,  0.11322766710555D+04,&
                    0.12130678843205D+04,  0.10795415876963D+04,  0.82788840602385D+03,  0.42668623935260D+03,&
                    0.89996713723805D+02, -0.50806856767787D+02, -0.17706733135474D+01,  0.80488607097064D+02,&
                    0.85506600768921D+02,  0.22334629722969D+02, -0.94249311295603D+01, -0.22675458883284D+01,&
                    0.35019083393710D+01,  0.20984763091238D+01,  0.43203486256733D+00,  0.33269016722691D-01,&
                    0.86924772143604D-03,  0.37581490352601D-04/)
  case(025)
    nfun = 32
    alf(1:nfun)=(/  0.22480688151639D+07,  0.13624659485842D+07,  0.82573693853586D+06,  0.50044662941568D+06,&
                    0.30330098752465D+06,  0.18381878031797D+06,  0.11140532140483D+06,  0.67518376608988D+05,&
                    0.40920228247872D+05,  0.24800138332043D+05,  0.15030386867905D+05,  0.91093253744880D+04,&
                    0.55208032572654D+04,  0.33459413680397D+04,  0.20278432533574D+04,  0.12289959111257D+04,&
                    0.74484600674283D+03,  0.45142182226838D+03,  0.27358898319296D+03,  0.16581150496543D+03,&
                    0.10049182119117D+03,  0.60904134055255D+02,  0.36911596397124D+02,  0.22370664483105D+02,&
                    0.13557978474609D+02,  0.82169566512784D+01,  0.49799737280475D+01,  0.30181658957864D+01,&
                    0.18291914519917D+01,  0.11086008799950D+01,  0.67187932120909D+00,  0.40719958861157D+00/)
    coe(1:nfun)=(/  0.30370627984524D+05, -0.17540087780714D+05, -0.55784574182034D+04,  0.17762014113665D+05,&
                   -0.16280036683559D+05,  0.13219261690438D+05, -0.76336414661407D+04,  0.57344895119667D+04,&
                   -0.22197475665659D+04,  0.24988311724961D+04, -0.54214964557439D+02,  0.15417690103762D+04,&
                    0.82972771575861D+03,  0.14204454125799D+04,  0.12510387773924D+04,  0.14045070506590D+04,&
                    0.11737100541894D+04,  0.87962828051246D+03,  0.40276839916507D+03,  0.57213482439824D+02,&
                   -0.61831622848593D+02,  0.18530640496429D+02,  0.10440501027355D+03,  0.88737808706977D+02,&
                    0.12503805349812D+02, -0.11510972221661D+02, -0.17011804242793D+00,  0.43295948394933D+01,&
                    0.19210651776349D+01,  0.29374646118197D+00,  0.14211926951404D-01,  0.34083364774329D-03/)
  case(026)
    nfun = 32
    alf(1:nfun)=(/  0.22480688151639D+07,  0.13624659485842D+07,  0.82573693853586D+06,  0.50044662941568D+06,&
                    0.30330098752465D+06,  0.18381878031797D+06,  0.11140532140483D+06,  0.67518376608988D+05,&
                    0.40920228247872D+05,  0.24800138332043D+05,  0.15030386867905D+05,  0.91093253744880D+04,&
                    0.55208032572654D+04,  0.33459413680397D+04,  0.20278432533574D+04,  0.12289959111257D+04,&
                    0.74484600674283D+03,  0.45142182226838D+03,  0.27358898319296D+03,  0.16581150496543D+03,&
                    0.10049182119117D+03,  0.60904134055255D+02,  0.36911596397124D+02,  0.22370664483105D+02,&
                    0.13557978474609D+02,  0.82169566512784D+01,  0.49799737280475D+01,  0.30181658957864D+01,&
                    0.18291914519917D+01,  0.11086008799950D+01,  0.67187932120909D+00,  0.40719958861157D+00/)
    coe(1:nfun)=(/  0.23189068541169D+05, -0.10872102563739D+05, -0.11174539232850D+05,  0.22145329801202D+05,&
                   -0.19157308440208D+05,  0.15262912468314D+05, -0.86924509000778D+04,  0.65544972936660D+04,&
                   -0.24751526208296D+04,  0.28597364457846D+04, -0.70515587031841D+01,  0.17719578103572D+04,&
                    0.99058124977164D+03,  0.16291145128882D+04,  0.14397626738425D+04,  0.15729583207117D+04,&
                    0.12834261686238D+04,  0.91211625187751D+03,  0.37505776275633D+03,  0.20664949099213D+02,&
                   -0.65412811233386D+02,  0.43733511947754D+02,  0.12845313015073D+03,  0.86639853901561D+02,&
                    0.24201481408263D+01, -0.11773912273294D+02,  0.21059943625341D+01,  0.47352008160126D+01,&
                    0.16499769130054D+01,  0.20333536124140D+00,  0.78070886345891D-02,  0.18397752257338D-03/)
  case(027)
    nfun = 32
    alf(1:nfun)=(/  0.22480688151639D+07,  0.13624659485842D+07,  0.82573693853586D+06,  0.50044662941568D+06,&
                    0.30330098752465D+06,  0.18381878031797D+06,  0.11140532140483D+06,  0.67518376608988D+05,&
                    0.40920228247872D+05,  0.24800138332043D+05,  0.15030386867905D+05,  0.91093253744880D+04,&
                    0.55208032572654D+04,  0.33459413680397D+04,  0.20278432533574D+04,  0.12289959111257D+04,&
                    0.74484600674283D+03,  0.45142182226838D+03,  0.27358898319296D+03,  0.16581150496543D+03,&
                    0.10049182119117D+03,  0.60904134055255D+02,  0.36911596397124D+02,  0.22370664483105D+02,&
                    0.13557978474609D+02,  0.82169566512784D+01,  0.49799737280475D+01,  0.30181658957864D+01,&
                    0.18291914519917D+01,  0.11086008799950D+01,  0.67187932120909D+00,  0.40719958861157D+00/)
    coe(1:nfun)=(/  0.39324079026282D+05, -0.39138171395540D+05,  0.15346207272291D+05,  0.32920627801418D+04,&
                   -0.70579123427272D+04,  0.83303938311636D+04, -0.45184782659113D+04,  0.44345725177079D+04,&
                   -0.10548044840768D+04,  0.23203284232654D+04,  0.56889173753746D+03,  0.17432595723458D+04,&
                    0.13266392123699D+04,  0.17723848303547D+04,  0.16900524363148D+04,  0.17227840999068D+04,&
                    0.14036390632336D+04,  0.92639956984934D+03,  0.34288703509790D+03, -0.17439590237744D+02,&
                   -0.61249763318410D+02,  0.73778606477612D+02,  0.15106454062107D+03,  0.79593082236608D+02,&
                   -0.69408333214396D+01, -0.10603303962282D+02,  0.42787861313270D+01,  0.48652340722745D+01,&
                    0.13732169991225D+01,  0.13794985083433D+00,  0.43047858317706D-02,  0.10331527199936D-03/)
  case(028)
    nfun = 30
    alf(1:nfun)=(/  0.82573693853586D+06,  0.50044662941568D+06,  0.30330098752465D+06,  0.18381878031797D+06,&
                    0.11140532140483D+06,  0.67518376608988D+05,  0.40920228247872D+05,  0.24800138332043D+05,&
                    0.15030386867905D+05,  0.91093253744880D+04,  0.55208032572654D+04,  0.33459413680397D+04,&
                    0.20278432533574D+04,  0.12289959111257D+04,  0.74484600674283D+03,  0.45142182226838D+03,&
                    0.27358898319296D+03,  0.16581150496543D+03,  0.10049182119117D+03,  0.60904134055255D+02,&
                    0.36911596397124D+02,  0.22370664483105D+02,  0.13557978474609D+02,  0.82169566512784D+01,&
                    0.49799737280475D+01,  0.30181658957864D+01,  0.18291914519917D+01,  0.11086008799950D+01,&
                    0.67187932120909D+00,  0.40719958861157D+00/)
    coe(1:nfun)=(/  0.53164786319137D+04, -0.61774804787636D+04,  0.75662871499062D+04, -0.35592112509310D+04,&
                    0.38302735334188D+04, -0.45456220042706D+03,  0.20892829399556D+04,  0.81283613363281D+03,&
                    0.17257084836680D+04,  0.14266630670438D+04,  0.18552667631637D+04,  0.18422656911826D+04,&
                    0.20052826211441D+04,  0.18505632447879D+04,  0.15331676200598D+04,  0.92112628939582D+03,&
                    0.30741806863128D+03, -0.55416427629360D+02, -0.48474479591190D+02,  0.10782558565676D+03,&
                    0.17012140747467D+03,  0.67988130563304D+02, -0.14356252120372D+02, -0.81647180946813D+01,&
                    0.60269382505037D+01,  0.46583944028085D+01,  0.11403643450388D+01,  0.10880505672483D+00,&
                    0.33473009518055D-02,  0.10120949677983D-03/)
  case(029)
    nfun = 30
    alf(1:nfun)=(/  0.82573693853586D+06,  0.50044662941568D+06,  0.30330098752465D+06,  0.18381878031797D+06,&
                    0.11140532140483D+06,  0.67518376608988D+05,  0.40920228247872D+05,  0.24800138332043D+05,&
                    0.15030386867905D+05,  0.91093253744880D+04,  0.55208032572654D+04,  0.33459413680397D+04,&
                    0.20278432533574D+04,  0.12289959111257D+04,  0.74484600674283D+03,  0.45142182226838D+03,&
                    0.27358898319296D+03,  0.16581150496543D+03,  0.10049182119117D+03,  0.60904134055255D+02,&
                    0.36911596397124D+02,  0.22370664483105D+02,  0.13557978474609D+02,  0.82169566512784D+01,&
                    0.49799737280475D+01,  0.30181658957864D+01,  0.18291914519917D+01,  0.11086008799950D+01,&
                    0.67187932120909D+00,  0.40719958861157D+00/)
    coe(1:nfun)=(/  0.15813707230732D+05, -0.23144727128565D+05,  0.24238556628883D+05, -0.15675301806688D+05,&
                    0.12025755381823D+05, -0.51160240881220D+04,  0.51033084945455D+04, -0.56940778064355D+03,&
                    0.28406057679933D+04,  0.11771823416287D+04,  0.23796098547843D+04,  0.19506943982937D+04,&
                    0.23233381520791D+04,  0.19897930502786D+04,  0.16513148062293D+04,  0.90641475102925D+03,&
                    0.26367123513069D+03, -0.88919881932641D+02, -0.28521275573847D+02,  0.14623723858993D+03,&
                    0.18467482482872D+03,  0.53351737633065D+02, -0.20165040004852D+02, -0.51291902063715D+01,&
                    0.75650067365144D+01,  0.44297771496537D+01,  0.92044756685169D+00,  0.73064853889719D-01,&
                    0.19790547793086D-02,  0.55604800344611D-04/)
  case(030)
    nfun = 29
    alf(1:nfun)=(/  0.82573693853586D+06,  0.50044662941568D+06,  0.30330098752465D+06,  0.18381878031797D+06,&
                    0.11140532140483D+06,  0.67518376608988D+05,  0.40920228247872D+05,  0.24800138332043D+05,&
                    0.15030386867905D+05,  0.91093253744880D+04,  0.55208032572654D+04,  0.33459413680397D+04,&
                    0.20278432533574D+04,  0.12289959111257D+04,  0.74484600674283D+03,  0.45142182226838D+03,&
                    0.27358898319296D+03,  0.16581150496543D+03,  0.10049182119117D+03,  0.60904134055255D+02,&
                    0.36911596397124D+02,  0.22370664483105D+02,  0.13557978474609D+02,  0.82169566512784D+01,&
                    0.49799737280475D+01,  0.30181658957864D+01,  0.18291914519917D+01,  0.11086008799950D+01,&
                    0.67187932120909D+00/)
    coe(1:nfun)=(/  0.27287408303082D+05, -0.38943580421269D+05,  0.38769060514808D+05, -0.25675251258731D+05,&
                    0.18723065302283D+05, -0.87705939538181D+04,  0.75506803632081D+04, -0.15797115688846D+04,&
                    0.37807623685487D+04,  0.10701390620675D+04,  0.28605944695696D+04,  0.21198931868430D+04,&
                    0.26317824994549D+04,  0.21450866057612D+04,  0.17531190039896D+04,  0.88333874173940D+03,&
                    0.21218868046146D+03, -0.11627967223909D+03, -0.98092177782382D+00,  0.18803686539555D+03,&
                    0.19366730183454D+03,  0.36476142229374D+02, -0.24224922647910D+02, -0.16147288067682D+01,&
                    0.89385932262658D+01,  0.41568080479964D+01,  0.69390909011815D+00,  0.38635089753868D-01,&
                    0.98901069397073D-03/)
  case(031)
    nfun = 29
    alf(1:nfun)=(/  0.82573693853586D+06,  0.50044662941568D+06,  0.30330098752465D+06,  0.18381878031797D+06,&
                    0.11140532140483D+06,  0.67518376608988D+05,  0.40920228247872D+05,  0.24800138332043D+05,&
                    0.15030386867905D+05,  0.91093253744880D+04,  0.55208032572654D+04,  0.33459413680397D+04,&
                    0.20278432533574D+04,  0.12289959111257D+04,  0.74484600674283D+03,  0.45142182226838D+03,&
                    0.27358898319296D+03,  0.16581150496543D+03,  0.10049182119117D+03,  0.60904134055255D+02,&
                    0.36911596397124D+02,  0.22370664483105D+02,  0.13557978474609D+02,  0.82169566512784D+01,&
                    0.49799737280475D+01,  0.30181658957864D+01,  0.18291914519917D+01,  0.11086008799950D+01,&
                    0.67187932120909D+00/)
    coe(1:nfun)=(/  0.17484058285981D+05, -0.20570434487462D+05,  0.20886061727524D+05, -0.11625321346700D+05,&
                    0.98207716421187D+04, -0.29660620281547D+04,  0.45756123312414D+04,  0.49808780172467D+03,&
                    0.30600435849950D+04,  0.19111959982216D+04,  0.28682604657179D+04,  0.25791472800272D+04,&
                    0.28079796815985D+04,  0.23788747848268D+04,  0.18025124856307D+04,  0.87003557681106D+03,&
                    0.14453573608195D+03, -0.13054213811649D+03,  0.31562104999342D+02,  0.23325770909073D+03,&
                    0.19517637170861D+03,  0.18724691588597D+02, -0.26393212692248D+02,  0.23618358051340D+01,&
                    0.10010628119945D+02,  0.37759938628142D+01,  0.47827701009024D+00,  0.18356499832741D-01,&
                    0.32186488219565D-03/)
  case(032)
    nfun = 29
    alf(1:nfun)=(/  0.82573693853586D+06,  0.50044662941568D+06,  0.30330098752465D+06,  0.18381878031797D+06,&
                    0.11140532140483D+06,  0.67518376608988D+05,  0.40920228247872D+05,  0.24800138332043D+05,&
                    0.15030386867905D+05,  0.91093253744880D+04,  0.55208032572654D+04,  0.33459413680397D+04,&
                    0.20278432533574D+04,  0.12289959111257D+04,  0.74484600674283D+03,  0.45142182226838D+03,&
                    0.27358898319296D+03,  0.16581150496543D+03,  0.10049182119117D+03,  0.60904134055255D+02,&
                    0.36911596397124D+02,  0.22370664483105D+02,  0.13557978474609D+02,  0.82169566512784D+01,&
                    0.49799737280475D+01,  0.30181658957864D+01,  0.18291914519917D+01,  0.11086008799950D+01,&
                    0.67187932120909D+00/)
    coe(1:nfun)=(/  0.21170166787595D+05, -0.25543974913895D+05,  0.25965728505433D+05, -0.14841997795373D+05,&
                    0.12283615146160D+05, -0.39938493433744D+04,  0.55899247451276D+04,  0.38522730813548D+03,&
                    0.35924240314998D+04,  0.21129257280249D+04,  0.32656874618363D+04,  0.28623558661155D+04,&
                    0.31023618602317D+04,  0.25539618611212D+04,  0.18707773614486D+04,  0.82738894529955D+03,&
                    0.83537646053388D+02, -0.14193629544538D+03,  0.75341081650971D+02,  0.27694464152709D+03,&
                    0.19047840144243D+03,  0.29776315110576D+00, -0.26289458529282D+02,  0.64758167820046D+01,&
                    0.10824949451373D+02,  0.32545110033934D+01,  0.30050377145307D+00,  0.78372418585744D-02,&
                    0.11507785850107D-03/)
  case(033)
    nfun = 32
    alf(1:nfun)=(/  0.22480688151639D+07,  0.13624659485842D+07,  0.82573693853586D+06,  0.50044662941568D+06,&
                    0.30330098752465D+06,  0.18381878031797D+06,  0.11140532140483D+06,  0.67518376608988D+05,&
                    0.40920228247872D+05,  0.24800138332043D+05,  0.15030386867905D+05,  0.91093253744880D+04,&
                    0.55208032572654D+04,  0.33459413680397D+04,  0.20278432533574D+04,  0.12289959111257D+04,&
                    0.74484600674283D+03,  0.45142182226838D+03,  0.27358898319296D+03,  0.16581150496543D+03,&
                    0.10049182119117D+03,  0.60904134055255D+02,  0.36911596397124D+02,  0.22370664483105D+02,&
                    0.13557978474609D+02,  0.82169566512784D+01,  0.49799737280475D+01,  0.30181658957864D+01,&
                    0.18291914519917D+01,  0.11086008799950D+01,  0.67187932120909D+00,  0.40719958861157D+00/)
    coe(1:nfun)=(/  0.13421613366076D+06, -0.18126270170748D+06,  0.13640412636298D+06, -0.74268626272559D+05,&
                    0.40603052557230D+05, -0.16284439567506D+05,  0.11013252133116D+05, -0.19182156123295D+04,&
                    0.45943383833803D+04,  0.15496604235675D+04,  0.34245280560451D+04,  0.27666779172574D+04,&
                    0.34496977392458D+04,  0.33029398166623D+04,  0.33301677924916D+04,  0.27689343822707D+04,&
                    0.19016329641505D+04,  0.78561627480750D+03,  0.14730616641275D+02, -0.14053683404349D+03,&
                    0.12546398831867D+03,  0.31974193541721D+03,  0.17819759405032D+03, -0.17110807678399D+02,&
                   -0.24332516081531D+02,  0.10746912811244D+02,  0.11237208339877D+02,  0.26498005423856D+01,&
                    0.16927576943972D+00,  0.41308077073623D-02, -0.11130173560048D-03,  0.17956518123530D-04/)
  case(034)
    nfun = 33
    alf(1:nfun)=(/  0.37093135450204D+07,  0.22480688151639D+07,  0.13624659485842D+07,  0.82573693853586D+06,&
                    0.50044662941568D+06,  0.30330098752465D+06,  0.18381878031797D+06,  0.11140532140483D+06,&
                    0.67518376608988D+05,  0.40920228247872D+05,  0.24800138332043D+05,  0.15030386867905D+05,&
                    0.91093253744880D+04,  0.55208032572654D+04,  0.33459413680397D+04,  0.20278432533574D+04,&
                    0.12289959111257D+04,  0.74484600674283D+03,  0.45142182226838D+03,  0.27358898319296D+03,&
                    0.16581150496543D+03,  0.10049182119117D+03,  0.60904134055255D+02,  0.36911596397124D+02,&
                    0.22370664483105D+02,  0.13557978474609D+02,  0.82169566512784D+01,  0.49799737280475D+01,&
                    0.30181658957864D+01,  0.18291914519917D+01,  0.11086008799950D+01,  0.67187932120909D+00,&
                    0.40719958861157D+00/)
    coe(1:nfun)=(/  0.30813222167969D+05, -0.57456883789062D+05,  0.64305116210938D+05, -0.44786401367188D+05,&
                    0.33407689941406D+05, -0.16298948730469D+05,  0.13433489379883D+05, -0.36759549255371D+04,&
                    0.62148381958008D+04,  0.80938688659668D+03,  0.41407167968750D+04,  0.25761318817139D+04,&
                    0.38293355636597D+04,  0.34590253372192D+04,  0.38742200512886D+04,  0.35069339089394D+04,&
                    0.30113881758451D+04,  0.19000029100180D+04,  0.74180449759960D+03, -0.58083035007119D+02,&
                   -0.12641938172281D+03,  0.18207000196725D+03,  0.35939834678359D+03,  0.15885708559398D+03,&
                   -0.32667396801524D+02, -0.20648541407660D+02,  0.15013273688324D+02,  0.11183420028014D+02,&
                    0.20231924700129D+01,  0.87206714706554D-01,  0.25808498066908D-02, -0.15923628347991D-03,&
                    0.18788437444073D-04/)
  case(035)
    nfun = 31
    alf(1:nfun)=(/  0.37093135450204D+07,  0.22480688151639D+07,  0.13624659485842D+07,  0.82573693853586D+06,&
                    0.50044662941568D+06,  0.30330098752465D+06,  0.18381878031797D+06,  0.11140532140483D+06,&
                    0.67518376608988D+05,  0.40920228247872D+05,  0.24800138332043D+05,  0.15030386867905D+05,&
                    0.91093253744880D+04,  0.55208032572654D+04,  0.33459413680397D+04,  0.20278432533574D+04,&
                    0.12289959111257D+04,  0.74484600674283D+03,  0.45142182226838D+03,  0.27358898319296D+03,&
                    0.16581150496543D+03,  0.10049182119117D+03,  0.60904134055255D+02,  0.36911596397124D+02,&
                    0.22370664483105D+02,  0.13557978474609D+02,  0.82169566512784D+01,  0.49799737280475D+01,&
                    0.30181658957864D+01,  0.18291914519917D+01,  0.11086008799950D+01/)
    coe(1:nfun)=(/  0.11767451279926D+06, -0.19449175742769D+06,  0.19475103603172D+06, -0.14046574941397D+06,&
                    0.96251575286150D+05, -0.53763005574942D+05,  0.35830276442885D+05, -0.15852305606425D+05,&
                    0.13640714597344D+05, -0.27471498020291D+04,  0.67307662556022D+04,  0.17650467962585D+04,&
                    0.49279307776988D+04,  0.34956796520334D+04,  0.44656106665330D+04,  0.36918323598434D+04,&
                    0.32457378625930D+04,  0.18833501372450D+04,  0.68668770722544D+03, -0.12754323393671D+03,&
                   -0.10167622159162D+03,  0.24597222662400D+03,  0.39319873092871D+03,  0.13361838553728D+03,&
                   -0.45829938225694D+02, -0.15340566113615D+02,  0.19055692770810D+02,  0.10651296489401D+02,&
                    0.14413436642443D+01,  0.44650973800003D-01,  0.10455613143776D-02/)
  case(036)
    nfun = 32
    alf(1:nfun)=(/  0.37093135450204D+07,  0.22480688151639D+07,  0.13624659485842D+07,  0.82573693853586D+06,&
                    0.50044662941568D+06,  0.30330098752465D+06,  0.18381878031797D+06,  0.11140532140483D+06,&
                    0.67518376608988D+05,  0.40920228247872D+05,  0.24800138332043D+05,  0.15030386867905D+05,&
                    0.91093253744880D+04,  0.55208032572654D+04,  0.33459413680397D+04,  0.20278432533574D+04,&
                    0.12289959111257D+04,  0.74484600674283D+03,  0.45142182226838D+03,  0.27358898319296D+03,&
                    0.16581150496543D+03,  0.10049182119117D+03,  0.60904134055255D+02,  0.36911596397124D+02,&
                    0.22370664483105D+02,  0.13557978474609D+02,  0.82169566512784D+01,  0.49799737280475D+01,&
                    0.30181658957864D+01,  0.18291914519917D+01,  0.11086008799950D+01,  0.67187932120909D+00/)
    coe(1:nfun)=(/  0.16038152245045D+06, -0.28977317323184D+06,  0.29741369487596D+06, -0.22080182346630D+06,&
                    0.15092661915278D+06, -0.86851559710324D+05,  0.55926868138522D+05, -0.26706046524540D+05,&
                    0.20386602800503D+05, -0.58513463383988D+04,  0.91177641994543D+04,  0.11351628169026D+04,&
                    0.59749824867435D+04,  0.36132291291943D+04,  0.50456161663427D+04,  0.39007662035683D+04,&
                    0.34600217702710D+04,  0.18563323127166D+04,  0.61836105486780D+03, -0.19019907741528D+03,&
                   -0.66441161071685D+02,  0.31661732668397D+03,  0.41917430418067D+03,  0.10369233237590D+03,&
                   -0.56175498963794D+02, -0.85794384902283D+01,  0.22659728699807D+02,  0.96889803820064D+01,&
                    0.95929989651938D+00,  0.22949053189265D-01,  0.65499353617327D-04,  0.27746757859326D-04/)
end select

return
end

!-----------------------------------------------------------------------------------------------------------------------------------
!---  EDF data library: ncore = 28, iz = 29- 71
!-----------------------------------------------------------------------------------------------------------------------------------
subroutine EDFLIB28(iz,nfun,alf,coe)
implicit real(kind=8) (a-h,o-z)
real(kind=8) :: alf(*),coe(*)

select case(iz)
  case(029)
    nfun = 32
    alf(1:nfun)=(/  0.82573693853586D+06,  0.50044662941568D+06,  0.30330098752465D+06,  0.18381878031797D+06,&
                    0.11140532140483D+06,  0.67518376608988D+05,  0.40920228247872D+05,  0.24800138332043D+05,&
                    0.15030386867905D+05,  0.91093253744880D+04,  0.55208032572654D+04,  0.33459413680397D+04,&
                    0.20278432533574D+04,  0.12289959111257D+04,  0.74484600674283D+03,  0.45142182226838D+03,&
                    0.27358898319296D+03,  0.16581150496543D+03,  0.10049182119117D+03,  0.60904134055255D+02,&
                    0.36911596397124D+02,  0.22370664483105D+02,  0.13557978474609D+02,  0.82169566512784D+01,&
                    0.49799737280475D+01,  0.30181658957864D+01,  0.18291914519917D+01,  0.11086008799950D+01,&
                    0.67187932120909D+00,  0.40719958861157D+00,  0.24678762946156D+00,  0.14956826027973D+00/)
    coe(1:nfun)=(/  0.32847216124684D+04, -0.76320840451121D+04,  0.11747559525967D+05, -0.77188400598615D+04,&
                    0.74020302290469D+04, -0.25474360655546D+04,  0.36882423132025D+04,  0.20725492488965D+03,&
                    0.24131474715751D+04,  0.14124079409242D+04,  0.22501625636225D+04,  0.20216738300466D+04,&
                    0.22847090460358D+04,  0.20105040144849D+04,  0.16405022161411D+04,  0.91197140678734D+03,&
                    0.26111407505473D+03, -0.87391407428244D+02, -0.28924123279219D+02,  0.14608631628642D+03,&
                    0.18301612660460D+03,  0.50251091272230D+02, -0.22831918942398D+02, -0.43913822878301D+01,&
                    0.10127845442366D+02,  0.65309769923737D+01,  0.20099578784876D+01,  0.50277993045091D+00,&
                    0.13765072528921D+00,  0.31785436427754D-01,  0.49510085549374D-02,  0.35925855133911D-03/)
  case(030)
    nfun = 31
    alf(1:nfun)=(/  0.82573693853586D+06,  0.50044662941568D+06,  0.30330098752465D+06,  0.18381878031797D+06,&
                    0.11140532140483D+06,  0.67518376608988D+05,  0.40920228247872D+05,  0.24800138332043D+05,&
                    0.15030386867905D+05,  0.91093253744880D+04,  0.55208032572654D+04,  0.33459413680397D+04,&
                    0.20278432533574D+04,  0.12289959111257D+04,  0.74484600674283D+03,  0.45142182226838D+03,&
                    0.27358898319296D+03,  0.16581150496543D+03,  0.10049182119117D+03,  0.60904134055255D+02,&
                    0.36911596397124D+02,  0.22370664483105D+02,  0.13557978474609D+02,  0.82169566512784D+01,&
                    0.49799737280475D+01,  0.30181658957864D+01,  0.18291914519917D+01,  0.11086008799950D+01,&
                    0.67187932120909D+00,  0.40719958861157D+00,  0.24678762946156D+00/)
    coe(1:nfun)=(/  0.50551752878070D+05, -0.59366748977005D+05,  0.51343835889161D+05, -0.31898535704494D+05,&
                    0.21754997814059D+05, -0.10226179382429D+05,  0.82854548720941D+04, -0.19575741715021D+04,&
                    0.39819530850109D+04,  0.96198203412443D+03,  0.29195826251181D+04,  0.20876486726827D+04,&
                    0.26495564688616D+04,  0.21351747543311D+04,  0.17588586681371D+04,  0.88002717138309D+03,&
                    0.21447712993721D+03, -0.11723161042549D+03, -0.15908080407826D-01,  0.18683538578805D+03,&
                    0.19156003074337D+03,  0.31979858715591D+02, -0.26815947112224D+02,  0.87888420623131D-01,&
                    0.12361091105750D+02,  0.65910376114381D+01,  0.18649713995952D+01,  0.43823643857195D+00,&
                    0.99495924981028D-01,  0.13807413056002D-01,  0.11211937220477D-02/)
  case(031)
    nfun = 31
    alf(1:nfun)=(/  0.82573693853586D+06,  0.50044662941568D+06,  0.30330098752465D+06,  0.18381878031797D+06,&
                    0.11140532140483D+06,  0.67518376608988D+05,  0.40920228247872D+05,  0.24800138332043D+05,&
                    0.15030386867905D+05,  0.91093253744880D+04,  0.55208032572654D+04,  0.33459413680397D+04,&
                    0.20278432533574D+04,  0.12289959111257D+04,  0.74484600674283D+03,  0.45142182226838D+03,&
                    0.27358898319296D+03,  0.16581150496543D+03,  0.10049182119117D+03,  0.60904134055255D+02,&
                    0.36911596397124D+02,  0.22370664483105D+02,  0.13557978474609D+02,  0.82169566512784D+01,&
                    0.49799737280475D+01,  0.30181658957864D+01,  0.18291914519917D+01,  0.11086008799950D+01,&
                    0.67187932120909D+00,  0.40719958861157D+00,  0.24678762946156D+00/)
    coe(1:nfun)=(/  0.18037918410495D+05, -0.22715816142708D+05,  0.23434583729029D+05, -0.13744543308944D+05,&
                    0.11277915866137D+05, -0.38778482241333D+04,  0.51150570547581D+04,  0.18831581060588D+03,&
                    0.32344502350055D+04,  0.18142767857127D+04,  0.29214190739901D+04,  0.25504583731634D+04,&
                    0.28231253238260D+04,  0.23711001712044D+04,  0.18064950301728D+04,  0.86818605604485D+03,&
                    0.14582767934374D+03, -0.13062005084922D+03,  0.31946468029091D+02,  0.23193283664759D+03,&
                    0.19170893779857D+03,  0.13072295957996D+02, -0.28703630149036D+02,  0.53917741158859D+01,&
                    0.14269161949211D+02,  0.65356276545494D+01,  0.16597021462802D+01,  0.34963380324800D+00,&
                    0.53887401598949D-01,  0.40449976607650D-02,  0.11731216667288D-03/)
  case(032)
    nfun = 30
    alf(1:nfun)=(/  0.82573693853586D+06,  0.50044662941568D+06,  0.30330098752465D+06,  0.18381878031797D+06,&
                    0.11140532140483D+06,  0.67518376608988D+05,  0.40920228247872D+05,  0.24800138332043D+05,&
                    0.15030386867905D+05,  0.91093253744880D+04,  0.55208032572654D+04,  0.33459413680397D+04,&
                    0.20278432533574D+04,  0.12289959111257D+04,  0.74484600674283D+03,  0.45142182226838D+03,&
                    0.27358898319296D+03,  0.16581150496543D+03,  0.10049182119117D+03,  0.60904134055255D+02,&
                    0.36911596397124D+02,  0.22370664483105D+02,  0.13557978474609D+02,  0.82169566512784D+01,&
                    0.49799737280475D+01,  0.30181658957864D+01,  0.18291914519917D+01,  0.11086008799950D+01,&
                    0.67187932120909D+00,  0.40719958861157D+00/)
    coe(1:nfun)=(/  0.30171207332879D+05, -0.34156743331969D+05,  0.31794539022326D+05, -0.18093943332553D+05,&
                    0.14049413622677D+05, -0.49319710387886D+04,  0.60968592952639D+04,  0.11121788129956D+03,&
                    0.37418409119211D+04,  0.20317868964076D+04,  0.33095136938877D+04,  0.28391677607582D+04,&
                    0.31141902372949D+04,  0.25483526715999D+04,  0.18732652170998D+04,  0.82670368377550D+03,&
                    0.84125505976845D+02, -0.14133998158870D+03,  0.75223605281753D+02,  0.27520668198835D+03,&
                    0.18537048876931D+03, -0.65086888444730D+01, -0.27825455963621D+02,  0.11027689386106D+02,&
                    0.15928136105691D+02,  0.62666367912932D+01,  0.14040859604753D+01,  0.23978200919899D+00,&
                    0.22890170316231D-01,  0.10548065138716D-02/)
  case(033)
    nfun = 32
    alf(1:nfun)=(/  0.22480688151639D+07,  0.13624659485842D+07,  0.82573693853586D+06,  0.50044662941568D+06,&
                    0.30330098752465D+06,  0.18381878031797D+06,  0.11140532140483D+06,  0.67518376608988D+05,&
                    0.40920228247872D+05,  0.24800138332043D+05,  0.15030386867905D+05,  0.91093253744880D+04,&
                    0.55208032572654D+04,  0.33459413680397D+04,  0.20278432533574D+04,  0.12289959111257D+04,&
                    0.74484600674283D+03,  0.45142182226838D+03,  0.27358898319296D+03,  0.16581150496543D+03,&
                    0.10049182119117D+03,  0.60904134055255D+02,  0.36911596397124D+02,  0.22370664483105D+02,&
                    0.13557978474609D+02,  0.82169566512784D+01,  0.49799737280475D+01,  0.30181658957864D+01,&
                    0.18291914519917D+01,  0.11086008799950D+01,  0.67187932120909D+00,  0.40719958861157D+00/)
    coe(1:nfun)=(/  0.14738647088242D+06, -0.19723209766054D+06,  0.14949879483485D+06, -0.83030345690966D+05,&
                    0.46002395226240D+05, -0.19443196241558D+05,  0.12820905541509D+05, -0.29365056484342D+04,&
                    0.51642682021931D+04,  0.12323831163943D+04,  0.36007292656638D+04,  0.26691104273293D+04,&
                    0.35035441652620D+04,  0.32734743320807D+04,  0.33460020917773D+04,  0.27607751904523D+04,&
                    0.19057489381208D+04,  0.78403178699961D+03,  0.16014256597147D+02, -0.14016109256916D+03,&
                    0.12531611019881D+03,  0.31700223568463D+03,  0.17129001560488D+03, -0.24995800915960D+02,&
                   -0.24525490765225D+02,  0.16942599229424D+02,  0.17184790542772D+02,  0.57888508996616D+01,&
                    0.11066111129616D+01,  0.14219111763714D+00,  0.84843531409895D-02,  0.25584764587894D-03/)
  case(034)
    nfun = 34
    alf(1:nfun)=(/  0.37093135450204D+07,  0.22480688151639D+07,  0.13624659485842D+07,  0.82573693853586D+06,&
                    0.50044662941568D+06,  0.30330098752465D+06,  0.18381878031797D+06,  0.11140532140483D+06,&
                    0.67518376608988D+05,  0.40920228247872D+05,  0.24800138332043D+05,  0.15030386867905D+05,&
                    0.91093253744880D+04,  0.55208032572654D+04,  0.33459413680397D+04,  0.20278432533574D+04,&
                    0.12289959111257D+04,  0.74484600674283D+03,  0.45142182226838D+03,  0.27358898319296D+03,&
                    0.16581150496543D+03,  0.10049182119117D+03,  0.60904134055255D+02,  0.36911596397124D+02,&
                    0.22370664483105D+02,  0.13557978474609D+02,  0.82169566512784D+01,  0.49799737280475D+01,&
                    0.30181658957864D+01,  0.18291914519917D+01,  0.11086008799950D+01,  0.67187932120909D+00,&
                    0.40719958861157D+00,  0.24678762946156D+00/)
    coe(1:nfun)=(/  0.57329183551908D+05, -0.10891469554782D+06,  0.11684516309500D+06, -0.85373207284749D+05,&
                    0.60410295529008D+05, -0.32884892698795D+05,  0.23168335342988D+05, -0.92489391766340D+04,&
                    0.93593142669424D+04, -0.95052760797739D+03,  0.51207720896415D+04,  0.20321408823356D+04,&
                    0.41304307320160D+04,  0.32928850465184D+04,  0.39655696894487D+04,  0.34568769365796D+04,&
                    0.30388760877711D+04,  0.18850527495834D+04,  0.75047534756738D+03, -0.62039186088412D+02,&
                   -0.12307362436364D+03,  0.18000574550671D+03,  0.35637590105771D+03,  0.14917447516888D+03,&
                   -0.40896752270015D+02, -0.19204969667283D+02,  0.23055669775894D+02,  0.17900064756008D+02,&
                    0.51175407202874D+01,  0.81021176177624D+00,  0.72631925527325D-01,  0.37925825345759D-02,&
                   -0.87209311730341D-04,  0.14134335033944D-04/)
  case(035)
    nfun = 33
    alf(1:nfun)=(/  0.37093135450204D+07,  0.22480688151639D+07,  0.13624659485842D+07,  0.82573693853586D+06,&
                    0.50044662941568D+06,  0.30330098752465D+06,  0.18381878031797D+06,  0.11140532140483D+06,&
                    0.67518376608988D+05,  0.40920228247872D+05,  0.24800138332043D+05,  0.15030386867905D+05,&
                    0.91093253744880D+04,  0.55208032572654D+04,  0.33459413680397D+04,  0.20278432533574D+04,&
                    0.12289959111257D+04,  0.74484600674283D+03,  0.45142182226838D+03,  0.27358898319296D+03,&
                    0.16581150496543D+03,  0.10049182119117D+03,  0.60904134055255D+02,  0.36911596397124D+02,&
                    0.22370664483105D+02,  0.13557978474609D+02,  0.82169566512784D+01,  0.49799737280475D+01,&
                    0.30181658957864D+01,  0.18291914519917D+01,  0.11086008799950D+01,  0.67187932120909D+00,&
                    0.40719958861157D+00/)
    coe(1:nfun)=(/  0.11438158911133D+06, -0.19119944384766D+06,  0.19242602441406D+06, -0.13910597851562D+06,&
                    0.95484446533203D+05, -0.53341285156250D+05,  0.35596183593750D+05, -0.15722081359863D+05,&
                    0.13567173095703D+05, -0.27049936065674D+04,  0.67059160766602D+04,  0.17802789421082D+04,&
                    0.49180458431244D+04,  0.35025682802200D+04,  0.44605055131912D+04,  0.36957334766388D+04,&
                    0.32429376730919D+04,  0.18855398242474D+04,  0.68579918307066D+03, -0.12584532940388D+03,&
                   -0.10137371809781D+03,  0.24520904906094D+03,  0.38752798539400D+03,  0.12209798738360D+03,&
                   -0.54640602122061D+02, -0.11392761402996D+02,  0.28861311623245D+02,  0.18059827109159D+02,&
                    0.43059196362447D+01,  0.54339772532694D+00,  0.37089391495101D-01,  0.98652496217255D-03,&
                    0.25314045871028D-04/)
  case(036)
    nfun = 32
    alf(1:nfun)=(/  0.37093135450204D+07,  0.22480688151639D+07,  0.13624659485842D+07,  0.82573693853586D+06,&
                    0.50044662941568D+06,  0.30330098752465D+06,  0.18381878031797D+06,  0.11140532140483D+06,&
                    0.67518376608988D+05,  0.40920228247872D+05,  0.24800138332043D+05,  0.15030386867905D+05,&
                    0.91093253744880D+04,  0.55208032572654D+04,  0.33459413680397D+04,  0.20278432533574D+04,&
                    0.12289959111257D+04,  0.74484600674283D+03,  0.45142182226838D+03,  0.27358898319296D+03,&
                    0.16581150496543D+03,  0.10049182119117D+03,  0.60904134055255D+02,  0.36911596397124D+02,&
                    0.22370664483105D+02,  0.13557978474609D+02,  0.82169566512784D+01,  0.49799737280475D+01,&
                    0.30181658957864D+01,  0.18291914519917D+01,  0.11086008799950D+01,  0.67187932120909D+00/)
    coe(1:nfun)=(/  0.18623146058464D+06, -0.30774729727650D+06,  0.30498738781452D+06, -0.22210668258715D+06,&
                    0.15035036876273D+06, -0.85966745610833D+05,  0.55250524350464D+05, -0.26261778749704D+05,&
                    0.20118904353499D+05, -0.56943389007002D+04,  0.90273064022660D+04,  0.11874688146673D+04,&
                    0.59443576496337D+04,  0.36317563524283D+04,  0.50338770323163D+04,  0.39086208744654D+04,&
                    0.34547668736919D+04,  0.18602036924738D+04,  0.61653392792691D+03, -0.18760415192819D+03,&
                   -0.66545693956752D+02,  0.31541826355799D+03,  0.41134469533873D+03,  0.89552213616975D+02,&
                   -0.64556765544181D+02, -0.19195278603938D+01,  0.34378317844222D+02,  0.17553473345367D+02,&
                    0.34322077582787D+01,  0.34488574128352D+00,  0.16714339024830D-01,  0.43302346242768D-03/)
  case(037)
    nfun = 32
    alf(1:nfun)=(/  0.37093135450204D+07,  0.22480688151639D+07,  0.13624659485842D+07,  0.82573693853586D+06,&
                    0.50044662941568D+06,  0.30330098752465D+06,  0.18381878031797D+06,  0.11140532140483D+06,&
                    0.67518376608988D+05,  0.40920228247872D+05,  0.24800138332043D+05,  0.15030386867905D+05,&
                    0.91093253744880D+04,  0.55208032572654D+04,  0.33459413680397D+04,  0.20278432533574D+04,&
                    0.12289959111257D+04,  0.74484600674283D+03,  0.45142182226838D+03,  0.27358898319296D+03,&
                    0.16581150496543D+03,  0.10049182119117D+03,  0.60904134055255D+02,  0.36911596397124D+02,&
                    0.22370664483105D+02,  0.13557978474609D+02,  0.82169566512784D+01,  0.49799737280475D+01,&
                    0.30181658957864D+01,  0.18291914519917D+01,  0.11086008799950D+01,  0.67187932120909D+00/)
    coe(1:nfun)=(/  0.15919631856799D+06, -0.30692676411629D+06,  0.32285898599291D+06, -0.24242322714520D+06,&
                    0.16695123732495D+06, -0.96239305386186D+05,  0.62223435372233D+05, -0.29602234897494D+05,&
                    0.22713971555293D+05, -0.64125224266201D+04,  0.10148005714856D+05,  0.13272258360460D+04,&
                    0.66196800259333D+04,  0.40067373371189D+04,  0.55057111521526D+04,  0.41923007898470D+04,&
                    0.36184115933771D+04,  0.18381448656007D+04,  0.52781371921074D+03, -0.23667883309472D+03,&
                   -0.22929239703866D+02,  0.39182949991598D+03,  0.42463128158385D+03,  0.53906883433472D+02,&
                   -0.70706127803760D+02,  0.92718278311804D+01,  0.39189991754433D+02,  0.16413905292838D+02,&
                    0.25992854695270D+01,  0.20866495130988D+00,  0.62605128095612D-02,  0.18209317941675D-03/)
  case(038)
    nfun = 32
    alf(1:nfun)=(/  0.37093135450204D+07,  0.22480688151639D+07,  0.13624659485842D+07,  0.82573693853586D+06,&
                    0.50044662941568D+06,  0.30330098752465D+06,  0.18381878031797D+06,  0.11140532140483D+06,&
                    0.67518376608988D+05,  0.40920228247872D+05,  0.24800138332043D+05,  0.15030386867905D+05,&
                    0.91093253744880D+04,  0.55208032572654D+04,  0.33459413680397D+04,  0.20278432533574D+04,&
                    0.12289959111257D+04,  0.74484600674283D+03,  0.45142182226838D+03,  0.27358898319296D+03,&
                    0.16581150496543D+03,  0.10049182119117D+03,  0.60904134055255D+02,  0.36911596397124D+02,&
                    0.22370664483105D+02,  0.13557978474609D+02,  0.82169566512784D+01,  0.49799737280475D+01,&
                    0.30181658957864D+01,  0.18291914519917D+01,  0.11086008799950D+01,  0.67187932120909D+00/)
    coe(1:nfun)=(/  0.12650512671041D+06, -0.26023889104891D+06,  0.28241348285270D+06, -0.21269190978909D+06,&
                    0.14880527935302D+06, -0.84530926767111D+05,  0.56230103239417D+05, -0.25401847937375D+05,&
                    0.21124857457444D+05, -0.47173206036091D+04,  0.99943052987382D+04,  0.22415929016545D+04,&
                    0.69254590921514D+04,  0.46369740703981D+04,  0.58712818759945D+04,  0.45457645390972D+04,&
                    0.37323144114783D+04,  0.18179814870553D+04,  0.42218530485383D+03, -0.27213747214800D+03,&
                    0.30396689780275D+02,  0.47236784419222D+03,  0.42646244032820D+03,  0.16582854832723D+02,&
                   -0.72750632382516D+02,  0.21842176269381D+02,  0.43009752517336D+02,  0.14743043322421D+02,&
                    0.18779610372487D+01,  0.11958885405758D+00,  0.15165023302970D-02,  0.12576257333841D-03/)
  case(039)
    nfun = 32
    alf(1:nfun)=(/  0.37093135450204D+07,  0.22480688151639D+07,  0.13624659485842D+07,  0.82573693853586D+06,&
                    0.50044662941568D+06,  0.30330098752465D+06,  0.18381878031797D+06,  0.11140532140483D+06,&
                    0.67518376608988D+05,  0.40920228247872D+05,  0.24800138332043D+05,  0.15030386867905D+05,&
                    0.91093253744880D+04,  0.55208032572654D+04,  0.33459413680397D+04,  0.20278432533574D+04,&
                    0.12289959111257D+04,  0.74484600674283D+03,  0.45142182226838D+03,  0.27358898319296D+03,&
                    0.16581150496543D+03,  0.10049182119117D+03,  0.60904134055255D+02,  0.36911596397124D+02,&
                    0.22370664483105D+02,  0.13557978474609D+02,  0.82169566512784D+01,  0.49799737280475D+01,&
                    0.30181658957864D+01,  0.18291914519917D+01,  0.11086008799950D+01,  0.67187932120909D+00/)
    coe(1:nfun)=(/  0.57509950872421D+05, -0.11982415635276D+06,  0.13870162137318D+06, -0.99825695057154D+05,&
                    0.74432778926730D+05, -0.37834568722844D+05,  0.29707753128111D+05, -0.93577963252664D+04,&
                    0.12951415707752D+05,  0.72162124598771D+03,  0.78434557242468D+04,  0.43263159262612D+04,&
                    0.66494246789496D+04,  0.56390934220012D+04,  0.60648971736571D+04,  0.50033665813287D+04,&
                    0.37734278397431D+04,  0.18120917256089D+04,  0.29392815275840D+03, -0.28800099428682D+03,&
                    0.91345843239586D+02,  0.55616947572251D+03,  0.41524561373928D+03, -0.20301086808362D+02,&
                   -0.70723495648482D+02,  0.35468312145728D+02,  0.45518062990228D+02,  0.12755704759579D+02,&
                    0.12880422109921D+01,  0.63876993193787D-01, -0.16557991664001D-04,  0.10040403312495D-03/)
  case(040)
    nfun = 33
    alf(1:nfun)=(/  0.61203673492837D+07,  0.37093135450204D+07,  0.22480688151639D+07,  0.13624659485842D+07,&
                    0.82573693853586D+06,  0.50044662941568D+06,  0.30330098752465D+06,  0.18381878031797D+06,&
                    0.11140532140483D+06,  0.67518376608988D+05,  0.40920228247872D+05,  0.24800138332043D+05,&
                    0.15030386867905D+05,  0.91093253744880D+04,  0.55208032572654D+04,  0.33459413680397D+04,&
                    0.20278432533574D+04,  0.12289959111257D+04,  0.74484600674283D+03,  0.45142182226838D+03,&
                    0.27358898319296D+03,  0.16581150496543D+03,  0.10049182119117D+03,  0.60904134055255D+02,&
                    0.36911596397124D+02,  0.22370664483105D+02,  0.13557978474609D+02,  0.82169566512784D+01,&
                    0.49799737280475D+01,  0.30181658957864D+01,  0.18291914519917D+01,  0.11086008799950D+01,&
                    0.67187932120909D+00/)
    coe(1:nfun)=(/  0.41247166015625D+05, -0.96314599609375D+05,  0.12040158789062D+06, -0.89325767578125D+05,&
                    0.72759857910156D+05, -0.37815072753906D+05,  0.31678421386719D+05, -0.99372374267578D+04,&
                    0.14228815185547D+05,  0.63577978515625D+03,  0.85440988769531D+04,  0.44563128662109D+04,&
                    0.71614522590637D+04,  0.60289473381042D+04,  0.68800235862732D+04,  0.61604316759109D+04,&
                    0.55209352250099D+04,  0.37631888313293D+04,  0.18082249809504D+04,  0.15139819139242D+03,&
                   -0.28607140004635D+03,  0.16142653284967D+03,  0.64003663777560D+03,  0.39135266748816D+03,&
                   -0.55409971872345D+02, -0.64612292311154D+02,  0.49726791594410D+02,  0.46527428123285D+02,&
                    0.10601855504472D+02,  0.84907467174344D+00,  0.32907387176238D-01, -0.32420644402009D-03,&
                    0.67252264443596D-04/)
  case(041)
    nfun = 33
    alf(1:nfun)=(/  0.61203673492837D+07,  0.37093135450204D+07,  0.22480688151639D+07,  0.13624659485842D+07,&
                    0.82573693853586D+06,  0.50044662941568D+06,  0.30330098752465D+06,  0.18381878031797D+06,&
                    0.11140532140483D+06,  0.67518376608988D+05,  0.40920228247872D+05,  0.24800138332043D+05,&
                    0.15030386867905D+05,  0.91093253744880D+04,  0.55208032572654D+04,  0.33459413680397D+04,&
                    0.20278432533574D+04,  0.12289959111257D+04,  0.74484600674283D+03,  0.45142182226838D+03,&
                    0.27358898319296D+03,  0.16581150496543D+03,  0.10049182119117D+03,  0.60904134055255D+02,&
                    0.36911596397124D+02,  0.22370664483105D+02,  0.13557978474609D+02,  0.82169566512784D+01,&
                    0.49799737280475D+01,  0.30181658957864D+01,  0.18291914519917D+01,  0.11086008799950D+01,&
                    0.67187932120909D+00/)
    coe(1:nfun)=(/  0.32713309667969D+06, -0.55455397851562D+06,  0.55823279394531D+06, -0.41292694921875D+06,&
                    0.28527801220703D+06, -0.16571967724609D+06,  0.10745261682129D+06, -0.52064954345703D+05,&
                    0.38985985961914D+05, -0.12134249176025D+05,  0.16666522293091D+05,  0.10038984603882D+04,&
                    0.10099931591034D+05,  0.54234616184235D+04,  0.81624890804291D+04,  0.62646412582397D+04,&
                    0.60364482531548D+04,  0.37332876951694D+04,  0.17878706783056D+04,  0.73969515562057D+01,&
                   -0.27112987282872D+03,  0.24398000091314D+03,  0.71938688397408D+03,  0.35614685153961D+03,&
                   -0.87850654648617D+02, -0.54203100558370D+02,  0.63920602975879D+02,  0.46037518364727D+02,&
                    0.84719128533616D+01,  0.54201022250345D+00,  0.15338211629569D-01, -0.34511105241108D-04,&
                    0.19500890743984D-04/)
  case(042)
    nfun = 32
    alf(1:nfun)=(/  0.61203673492837D+07,  0.37093135450204D+07,  0.22480688151639D+07,  0.13624659485842D+07,&
                    0.82573693853586D+06,  0.50044662941568D+06,  0.30330098752465D+06,  0.18381878031797D+06,&
                    0.11140532140483D+06,  0.67518376608988D+05,  0.40920228247872D+05,  0.24800138332043D+05,&
                    0.15030386867905D+05,  0.91093253744880D+04,  0.55208032572654D+04,  0.33459413680397D+04,&
                    0.20278432533574D+04,  0.12289959111257D+04,  0.74484600674283D+03,  0.45142182226838D+03,&
                    0.27358898319296D+03,  0.16581150496543D+03,  0.10049182119117D+03,  0.60904134055255D+02,&
                    0.36911596397124D+02,  0.22370664483105D+02,  0.13557978474609D+02,  0.82169566512784D+01,&
                    0.49799737280475D+01,  0.30181658957864D+01,  0.18291914519917D+01,  0.11086008799950D+01/)
    coe(1:nfun)=(/  0.48180739843798D+06, -0.87179486345005D+06,  0.89040445250607D+06, -0.67081007617188D+06,&
                    0.45933763182211D+06, -0.27177845300794D+06,  0.17111469835734D+06, -0.87376983649611D+05,&
                    0.60080909810364D+05, -0.22790661857098D+05,  0.23719913049862D+05, -0.17739537734091D+04,&
                    0.12747740025599D+05,  0.50468535804041D+04,  0.93716099109575D+04,  0.64376324271001D+04,&
                    0.65144140798096D+04,  0.37001557412045D+04,  0.17421072107289D+04, -0.13093451555440D+03,&
                   -0.24457928595680D+03,  0.33971942838051D+03,  0.79098508545427D+03,  0.31085966293357D+03,&
                   -0.11659841801935D+03, -0.39621803616390D+02,  0.77479143769441D+02,  0.44087935284114D+02,&
                    0.65210433007142D+01,  0.34150157504905D+00,  0.58918633240239D-02,  0.19028947242106D-03/)
  case(043)
    nfun = 32
    alf(1:nfun)=(/  0.61203673492837D+07,  0.37093135450204D+07,  0.22480688151639D+07,  0.13624659485842D+07,&
                    0.82573693853586D+06,  0.50044662941568D+06,  0.30330098752465D+06,  0.18381878031797D+06,&
                    0.11140532140483D+06,  0.67518376608988D+05,  0.40920228247872D+05,  0.24800138332043D+05,&
                    0.15030386867905D+05,  0.91093253744880D+04,  0.55208032572654D+04,  0.33459413680397D+04,&
                    0.20278432533574D+04,  0.12289959111257D+04,  0.74484600674283D+03,  0.45142182226838D+03,&
                    0.27358898319296D+03,  0.16581150496543D+03,  0.10049182119117D+03,  0.60904134055255D+02,&
                    0.36911596397124D+02,  0.22370664483105D+02,  0.13557978474609D+02,  0.82169566512784D+01,&
                    0.49799737280475D+01,  0.30181658957864D+01,  0.18291914519917D+01,  0.11086008799950D+01/)
    coe(1:nfun)=(/  0.47230935036230D+06, -0.96386703894901D+06,  0.10242979308748D+07, -0.78862767220545D+06,&
                    0.54463494560766D+06, -0.32476778820610D+06,  0.20425906486750D+06, -0.10529980589712D+06,&
                    0.71557848841310D+05, -0.28000465083480D+05,  0.27861976642922D+05, -0.28540913712382D+04,&
                    0.14544163638957D+05,  0.52107786801346D+04,  0.10334780748293D+05,  0.67736355357049D+04,&
                    0.68985494548660D+04,  0.36932324331871D+04,  0.16540479942854D+04, -0.25104406708886D+03,&
                   -0.21153572776308D+03,  0.45149114620211D+03,  0.85008660266496D+03,  0.25809318577383D+03,&
                   -0.14144127931614D+03, -0.20688777776428D+02,  0.89585277199737D+02,  0.40931784287800D+02,&
                    0.48411180819835D+01,  0.21949999581608D+00,  0.34733914719709D-03,  0.36008104650875D-03/)
  case(044)
    nfun = 32
    alf(1:nfun)=(/  0.10098606126318D+08,  0.61203673492837D+07,  0.22480688151639D+07,  0.13624659485842D+07,&
                    0.82573693853586D+06,  0.50044662941568D+06,  0.30330098752465D+06,  0.18381878031797D+06,&
                    0.11140532140483D+06,  0.67518376608988D+05,  0.40920228247872D+05,  0.24800138332043D+05,&
                    0.15030386867905D+05,  0.91093253744880D+04,  0.55208032572654D+04,  0.33459413680397D+04,&
                    0.20278432533574D+04,  0.12289959111257D+04,  0.74484600674283D+03,  0.45142182226838D+03,&
                    0.27358898319296D+03,  0.16581150496543D+03,  0.10049182119117D+03,  0.60904134055255D+02,&
                    0.36911596397124D+02,  0.22370664483105D+02,  0.13557978474609D+02,  0.82169566512784D+01,&
                    0.49799737280475D+01,  0.30181658957864D+01,  0.18291914519917D+01,  0.11086008799950D+01/)
    coe(1:nfun)=(/  0.34587220077181D+06, -0.45200521553898D+06,  0.38267204302120D+06, -0.44815152715254D+06,&
                    0.38461231050682D+06, -0.24938628341961D+06,  0.17013899609876D+06, -0.87235253368020D+05,&
                    0.63722817898780D+05, -0.22574510116071D+05,  0.26288308580041D+05, -0.70251762278378D+03,&
                    0.14654843434010D+05,  0.63720537871383D+04,  0.10802879969521D+05,  0.74054713893449D+04,&
                    0.71132381150112D+04,  0.37510349515430D+04,  0.15033680094158D+04, -0.33938858858717D+03,&
                   -0.17658279888312D+03,  0.58079853844835D+03,  0.89287927083140D+03,  0.19986183204361D+03,&
                   -0.16146494873236D+03,  0.21779130112282D+01,  0.99574587617381D+02,  0.36938008636039D+02,&
                    0.34661642706949D+01,  0.14446416559355D+00, -0.24888821641440D-02,  0.47561859849310D-03/)
  case(045)
    nfun = 32
    alf(1:nfun)=(/  0.61203673492837D+07,  0.37093135450204D+07,  0.22480688151639D+07,  0.13624659485842D+07,&
                    0.82573693853586D+06,  0.50044662941568D+06,  0.30330098752465D+06,  0.18381878031797D+06,&
                    0.11140532140483D+06,  0.67518376608988D+05,  0.40920228247872D+05,  0.24800138332043D+05,&
                    0.15030386867905D+05,  0.91093253744880D+04,  0.55208032572654D+04,  0.33459413680397D+04,&
                    0.20278432533574D+04,  0.12289959111257D+04,  0.74484600674283D+03,  0.45142182226838D+03,&
                    0.27358898319296D+03,  0.16581150496543D+03,  0.10049182119117D+03,  0.60904134055255D+02,&
                    0.36911596397124D+02,  0.22370664483105D+02,  0.13557978474609D+02,  0.82169566512784D+01,&
                    0.49799737280475D+01,  0.30181658957864D+01,  0.18291914519917D+01,  0.11086008799950D+01/)
    coe(1:nfun)=(/  0.16952230697393D+06, -0.47169321400928D+06,  0.56045181329536D+06, -0.44080249824095D+06,&
                    0.32199083053160D+06, -0.18669289340806D+06,  0.12715933931017D+06, -0.58462657290161D+05,&
                    0.48204019554138D+05, -0.12169294969395D+05,  0.21882948058724D+05,  0.31760060869977D+04,&
                    0.13884897343397D+05,  0.80997522613853D+04,  0.11008144003130D+05,  0.82054633008623D+04,&
                    0.72240152375870D+04,  0.38362927563391D+04,  0.13109325889111D+04, -0.40425149473647D+03,&
                   -0.13357072570248D+03,  0.72308424180601D+03,  0.91907469233306D+03,  0.13734649742669D+03,&
                   -0.17599155507473D+03,  0.28470353143090D+02,  0.10705689587260D+03,  0.32358750435827D+02,&
                    0.24152136265520D+01,  0.98484519187650D-01, -0.37664480905590D-02,  0.50528210332601D-03/)
  case(046)
    nfun = 30
    alf(1:nfun)=(/  0.22480688151639D+07,  0.13624659485842D+07,  0.82573693853586D+06,  0.50044662941568D+06,&
                    0.30330098752465D+06,  0.18381878031797D+06,  0.11140532140483D+06,  0.67518376608988D+05,&
                    0.40920228247872D+05,  0.24800138332043D+05,  0.15030386867905D+05,  0.91093253744880D+04,&
                    0.55208032572654D+04,  0.33459413680397D+04,  0.20278432533574D+04,  0.12289959111257D+04,&
                    0.74484600674283D+03,  0.45142182226838D+03,  0.27358898319296D+03,  0.16581150496543D+03,&
                    0.10049182119117D+03,  0.60904134055255D+02,  0.36911596397124D+02,  0.22370664483105D+02,&
                    0.13557978474609D+02,  0.82169566512784D+01,  0.49799737280475D+01,  0.30181658957864D+01,&
                    0.18291914519917D+01,  0.11086008799950D+01/)
    coe(1:nfun)=(/  0.25731225677729D+05, -0.18889748571157D+05,  0.45831333602190D+05, -0.17313428983688D+05,&
                    0.30539822581291D+05, -0.18044668139219D+04,  0.18050794027418D+05,  0.62547135449201D+04,&
                    0.13222799579479D+05,  0.94986190905795D+04,  0.11862187263511D+05,  0.10595562354423D+05,&
                    0.10839566488802D+05,  0.92314635917749D+04,  0.71958611328031D+04,  0.39657374064443D+04,&
                    0.10691016686294D+04, -0.43868607601250D+03, -0.84182736339804D+02,  0.87763604057876D+03,&
                    0.92615035896168D+03,  0.72753046613878D+02, -0.18446753636366D+03,  0.57505170724351D+02,&
                    0.11171245574691D+03,  0.27583885599461D+02,  0.16366760237223D+01,  0.64327497518434D-01,&
                   -0.38116360602345D-02,  0.40604122492205D-03/)
  case(047)
    nfun = 33
    alf(1:nfun)=(/  0.10098606126318D+08,  0.61203673492837D+07,  0.37093135450204D+07,  0.22480688151639D+07,&
                    0.13624659485842D+07,  0.82573693853586D+06,  0.50044662941568D+06,  0.30330098752465D+06,&
                    0.18381878031797D+06,  0.11140532140483D+06,  0.67518376608988D+05,  0.40920228247872D+05,&
                    0.24800138332043D+05,  0.15030386867905D+05,  0.91093253744880D+04,  0.55208032572654D+04,&
                    0.33459413680397D+04,  0.20278432533574D+04,  0.12289959111257D+04,  0.74484600674283D+03,&
                    0.45142182226838D+03,  0.27358898319296D+03,  0.16581150496543D+03,  0.10049182119117D+03,&
                    0.60904134055255D+02,  0.36911596397124D+02,  0.22370664483105D+02,  0.13557978474609D+02,&
                    0.82169566512784D+01,  0.49799737280475D+01,  0.30181658957864D+01,  0.18291914519917D+01,&
                    0.11086008799950D+01/)
    coe(1:nfun)=(/  0.21094990625000D+06, -0.58837662109375D+06,  0.71129897656250D+06, -0.55783122460938D+06,&
                    0.40538522558594D+06, -0.23178798876953D+06,  0.15732957812500D+06, -0.71093476074219D+05,&
                    0.58790139160156D+05, -0.14479672973633D+05,  0.26256416870117D+05,  0.37794635162354D+04,&
                    0.16378774635315D+05,  0.96272954559326D+04,  0.13286992568970D+05,  0.10612090143204D+05,&
                    0.10307278268814D+05,  0.71218254013062D+04,  0.40871072232723D+04,  0.80766867518425D+03,&
                   -0.45583674609661D+03, -0.19750393867493D+02,  0.10374552256912D+04,  0.91577459795028D+03,&
                    0.67311148643494D+01, -0.18597821191140D+03,  0.88274515966419D+02,  0.11332046270324D+03,&
                    0.22870616074943D+02,  0.11099159516743D+01,  0.40339466926525D-01, -0.26817999623745D-02,&
                    0.26918316314095D-03/)
  case(048)
    nfun = 31
    alf(1:nfun)=(/  0.37093135450204D+07,  0.22480688151639D+07,  0.13624659485842D+07,  0.82573693853586D+06,&
                    0.50044662941568D+06,  0.30330098752465D+06,  0.18381878031797D+06,  0.11140532140483D+06,&
                    0.67518376608988D+05,  0.40920228247872D+05,  0.24800138332043D+05,  0.15030386867905D+05,&
                    0.91093253744880D+04,  0.55208032572654D+04,  0.33459413680397D+04,  0.20278432533574D+04,&
                    0.12289959111257D+04,  0.74484600674283D+03,  0.45142182226838D+03,  0.27358898319296D+03,&
                    0.16581150496543D+03,  0.10049182119117D+03,  0.60904134055255D+02,  0.36911596397124D+02,&
                    0.22370664483105D+02,  0.13557978474609D+02,  0.82169566512784D+01,  0.49799737280475D+01,&
                    0.30181658957864D+01,  0.18291914519917D+01,  0.11086008799950D+01/)
    coe(1:nfun)=(/  0.34851407158673D+06, -0.54725834683490D+06,  0.56479864850092D+06, -0.40035574727035D+06,&
                    0.28682255860436D+06, -0.15294277198827D+06,  0.11039627691215D+06, -0.42490039979488D+05,&
                    0.44113312206969D+05, -0.44236713463739D+04,  0.22723179303695D+05,  0.77764556124471D+04,&
                    0.15850341272768D+05,  0.10501878758736D+05,  0.11331733052127D+05,  0.70554301503068D+04,&
                    0.41681078759416D+04,  0.54723990890593D+03, -0.46469625516707D+03,  0.66579875947151D+02,&
                    0.11961511056639D+04,  0.88935451066200D+03, -0.60086521850921D+02, -0.17953164985175D+03,&
                    0.11952009269431D+03,  0.11208010805013D+03,  0.18465308014801D+02,  0.76499818610569D+00,&
                    0.19387839552280D-01, -0.82529293464972D-03,  0.73932975256596D-04/)
  case(049)
    nfun = 30
    alf(1:nfun)=(/  0.37093135450204D+07,  0.22480688151639D+07,  0.13624659485842D+07,  0.82573693853586D+06,&
                    0.50044662941568D+06,  0.30330098752465D+06,  0.18381878031797D+06,  0.11140532140483D+06,&
                    0.67518376608988D+05,  0.40920228247872D+05,  0.24800138332043D+05,  0.15030386867905D+05,&
                    0.91093253744880D+04,  0.55208032572654D+04,  0.33459413680397D+04,  0.20278432533574D+04,&
                    0.12289959111257D+04,  0.74484600674283D+03,  0.45142182226838D+03,  0.27358898319296D+03,&
                    0.16581150496543D+03,  0.10049182119117D+03,  0.60904134055255D+02,  0.36911596397124D+02,&
                    0.22370664483105D+02,  0.13557978474609D+02,  0.82169566512784D+01,  0.49799737280475D+01,&
                    0.30181658957864D+01,  0.18291914519917D+01/)
    coe(1:nfun)=(/  0.43517838994265D+06, -0.73595169138956D+06,  0.77032394431019D+06, -0.55997421341181D+06,&
                    0.39701689034319D+06, -0.21868957032967D+06,  0.15151237188232D+06, -0.64014380727768D+05,&
                    0.58424328415483D+05, -0.10531309618190D+05,  0.28063107488081D+05,  0.65599752809256D+04,&
                    0.18159163938217D+05,  0.10570912165500D+05,  0.12269826153040D+05,  0.70121306205136D+04,&
                    0.41983239417870D+04,  0.29629595566657D+03, -0.46721229293698D+03,  0.17727403630514D+03,&
                    0.13493580993553D+04,  0.84760783481550D+03, -0.12641645485721D+03, -0.16460961657005D+03,&
                    0.15002260593417D+03,  0.10825496869240D+03,  0.14541151705009D+02,  0.54066366931556D+00,&
                    0.43674531600004D-02,  0.37179026142553D-03/)
  case(050)
    nfun = 30
    alf(1:nfun)=(/  0.37093135450204D+07,  0.22480688151639D+07,  0.13624659485842D+07,  0.82573693853586D+06,&
                    0.50044662941568D+06,  0.30330098752465D+06,  0.18381878031797D+06,  0.11140532140483D+06,&
                    0.67518376608988D+05,  0.40920228247872D+05,  0.24800138332043D+05,  0.15030386867905D+05,&
                    0.91093253744880D+04,  0.55208032572654D+04,  0.33459413680397D+04,  0.20278432533574D+04,&
                    0.12289959111257D+04,  0.74484600674283D+03,  0.45142182226838D+03,  0.27358898319296D+03,&
                    0.16581150496543D+03,  0.10049182119117D+03,  0.60904134055255D+02,  0.36911596397124D+02,&
                    0.22370664483105D+02,  0.13557978474609D+02,  0.82169566512784D+01,  0.49799737280475D+01,&
                    0.30181658957864D+01,  0.18291914519917D+01/)
    coe(1:nfun)=(/  0.45253609473109D+06, -0.82939265169525D+06,  0.89236353722906D+06, -0.66042301792479D+06,&
                    0.46984834676600D+06, -0.26182541076851D+06,  0.17964997298717D+06, -0.77886476577044D+05,&
                    0.68513690050244D+05, -0.14065948278531D+05,  0.32056075249635D+05,  0.62276837703735D+04,&
                    0.20041802955579D+05,  0.10929745172080D+05,  0.13049508484581D+05,  0.70300242897766D+04,&
                    0.41562790286511D+04,  0.68503243204439D+02, -0.46771660367516D+03,  0.31557915165205D+03,&
                    0.14921082715746D+04,  0.79188144340906D+03, -0.19100790685377D+03, -0.14099898222783D+03,&
                    0.17864089864105D+03,  0.10223014685930D+03,  0.11188489885174D+02,  0.39402207027729D+00,&
                   -0.53839993761358D-02,  0.97306258245311D-03/)
  case(051)
    nfun = 30
    alf(1:nfun)=(/  0.37093135450204D+07,  0.22480688151639D+07,  0.13624659485842D+07,  0.82573693853586D+06,&
                    0.50044662941568D+06,  0.30330098752465D+06,  0.18381878031797D+06,  0.11140532140483D+06,&
                    0.67518376608988D+05,  0.40920228247872D+05,  0.24800138332043D+05,  0.15030386867905D+05,&
                    0.91093253744880D+04,  0.55208032572654D+04,  0.33459413680397D+04,  0.20278432533574D+04,&
                    0.12289959111257D+04,  0.74484600674283D+03,  0.45142182226838D+03,  0.27358898319296D+03,&
                    0.16581150496543D+03,  0.10049182119117D+03,  0.60904134055255D+02,  0.36911596397124D+02,&
                    0.22370664483105D+02,  0.13557978474609D+02,  0.82169566512784D+01,  0.49799737280475D+01,&
                    0.30181658957864D+01,  0.18291914519917D+01/)
    coe(1:nfun)=(/  0.42407885186577D+06, -0.81315766114902D+06,  0.89618288872766D+06, -0.66644876216030D+06,&
                    0.47930846822643D+06, -0.26547807139111D+06,  0.18473720350754D+06, -0.78322076829851D+05,&
                    0.71188127694398D+05, -0.13290192919433D+05,  0.33784268643521D+05,  0.72665889558569D+04,&
                    0.21242549099527D+05,  0.11712238613581D+05,  0.13594600772039D+05,  0.71481263832068D+04,&
                    0.40192354077597D+04, -0.12068895829958D+03, -0.47165176576370D+03,  0.48478627507767D+03,&
                    0.16193157268038D+04,  0.72397165838339D+03, -0.25276168294995D+03, -0.10859638620108D+03,&
                    0.20420375358512D+03,  0.94531608363844D+02,  0.84164421406831D+01,  0.29899214059833D+00,&
                   -0.11929135479988D-01,  0.14045178009354D-02/)
  case(052)
    nfun = 32
    alf(1:nfun)=(/  0.10098606126318D+08,  0.61203673492837D+07,  0.37093135450204D+07,  0.22480688151639D+07,&
                    0.13624659485842D+07,  0.82573693853586D+06,  0.50044662941568D+06,  0.30330098752465D+06,&
                    0.18381878031797D+06,  0.11140532140483D+06,  0.67518376608988D+05,  0.40920228247872D+05,&
                    0.24800138332043D+05,  0.15030386867905D+05,  0.91093253744880D+04,  0.55208032572654D+04,&
                    0.33459413680397D+04,  0.20278432533574D+04,  0.12289959111257D+04,  0.74484600674283D+03,&
                    0.45142182226838D+03,  0.27358898319296D+03,  0.16581150496543D+03,  0.10049182119117D+03,&
                    0.60904134055255D+02,  0.36911596397124D+02,  0.22370664483105D+02,  0.13557978474609D+02,&
                    0.82169566512784D+01,  0.49799737280475D+01,  0.30181658957864D+01,  0.18291914519917D+01/)
    coe(1:nfun)=(/  0.65426515731239D+06, -0.19469096601601D+07,  0.23050982962723D+07, -0.18463406167707D+07,&
                    0.13048586625795D+07, -0.77785704816151D+06,  0.49491283233237D+06, -0.25216422100675D+06,&
                    0.17374440395570D+06, -0.67326162423134D+05,  0.66975653636754D+05, -0.82919919616580D+04,&
                    0.33236891292237D+05,  0.97053596318401D+04,  0.21741435546326D+05,  0.12925680195924D+05,&
                    0.13899053853216D+05,  0.73650362305031D+04,  0.37883687818137D+04, -0.26912100217980D+03,&
                   -0.47733853404818D+03,  0.68416016330069D+03,  0.17277535965579D+04,  0.64511958344535D+03,&
                   -0.31015288421372D+03, -0.67816986230817D+02,  0.22579644092805D+03,  0.85680827732124D+02,&
                    0.62036775321185D+01,  0.23270294185318D+00, -0.15248408251466D-01,  0.16035088688255D-02/)
  case(053)
    nfun = 30
    alf(1:nfun)=(/  0.37093135450204D+07,  0.22480688151639D+07,  0.13624659485842D+07,  0.82573693853586D+06,&
                    0.50044662941568D+06,  0.30330098752465D+06,  0.18381878031797D+06,  0.11140532140483D+06,&
                    0.67518376608988D+05,  0.40920228247872D+05,  0.24800138332043D+05,  0.15030386867905D+05,&
                    0.91093253744880D+04,  0.55208032572654D+04,  0.33459413680397D+04,  0.20278432533574D+04,&
                    0.12289959111257D+04,  0.74484600674283D+03,  0.45142182226838D+03,  0.27358898319296D+03,&
                    0.16581150496543D+03,  0.10049182119117D+03,  0.60904134055255D+02,  0.36911596397124D+02,&
                    0.22370664483105D+02,  0.13557978474609D+02,  0.82169566512784D+01,  0.49799737280475D+01,&
                    0.30181658957864D+01,  0.18291914519917D+01/)
    coe(1:nfun)=(/  0.23638262476087D+06, -0.44475428832722D+06,  0.53567450949812D+06, -0.38173301548553D+06,&
                    0.29812371064162D+06, -0.14771161299634D+06,  0.12186077269351D+06, -0.36598962361336D+05,&
                    0.53040503560901D+05,  0.19424654279351D+04,  0.30037823309436D+05,  0.13698707265299D+05,&
                    0.21470198045731D+05,  0.14601299684566D+05,  0.13942246415981D+05,  0.76890514993151D+04,&
                    0.34593971436394D+04, -0.37190270751342D+03, -0.48413254383684D+03,  0.91266242317631D+03,&
                    0.18145780264127D+04,  0.55661935302183D+03, -0.36154803222955D+03, -0.19457079214724D+02,&
                    0.24274117614493D+03,  0.76186552569931D+02,  0.44964452948252D+01,  0.18083559594538D+00,&
                   -0.15629294832479D-01,  0.15637577206107D-02/)
  case(054)
    nfun = 30
    alf(1:nfun)=(/  0.37093135450204D+07,  0.22480688151639D+07,  0.13624659485842D+07,  0.82573693853586D+06,&
                    0.50044662941568D+06,  0.30330098752465D+06,  0.18381878031797D+06,  0.11140532140483D+06,&
                    0.67518376608988D+05,  0.40920228247872D+05,  0.24800138332043D+05,  0.15030386867905D+05,&
                    0.91093253744880D+04,  0.55208032572654D+04,  0.33459413680397D+04,  0.20278432533574D+04,&
                    0.12289959111257D+04,  0.74484600674283D+03,  0.45142182226838D+03,  0.27358898319296D+03,&
                    0.16581150496543D+03,  0.10049182119117D+03,  0.60904134055255D+02,  0.36911596397124D+02,&
                    0.22370664483105D+02,  0.13557978474609D+02,  0.82169566512784D+01,  0.49799737280475D+01,&
                    0.30181658957864D+01,  0.18291914519917D+01/)
    coe(1:nfun)=(/  0.99273944761038D+05, -0.13672084026527D+06,  0.21693582445192D+06, -0.12635211992836D+06,&
                    0.13113883230448D+06, -0.40809074811101D+05,  0.62449319880724D+05,  0.69949698433280D+03,&
                    0.34981867466211D+05,  0.14867249143019D+05,  0.25430980041496D+05,  0.18614097396538D+05,&
                    0.20761351959461D+05,  0.16556922629541D+05,  0.13818824758126D+05,  0.80656680864091D+04,&
                    0.30622830853709D+04, -0.44180383027880D+03, -0.48240396708343D+03,  0.11642233567975D+04,&
                    0.18795203291818D+04,  0.45883650102765D+03, -0.40482733475038D+03,  0.35163844953280D+02,&
                    0.25468125165705D+03,  0.66495508169799D+02,  0.32256200207921D+01,  0.13491809565755D+00,&
                   -0.13519194528897D-01,  0.13111841444200D-02/)
  case(055)
    nfun = 31
    alf(1:nfun)=(/  0.61203673492837D+07,  0.37093135450204D+07,  0.22480688151639D+07,  0.13624659485842D+07,&
                    0.82573693853586D+06,  0.50044662941568D+06,  0.30330098752465D+06,  0.18381878031797D+06,&
                    0.11140532140483D+06,  0.67518376608988D+05,  0.40920228247872D+05,  0.24800138332043D+05,&
                    0.15030386867905D+05,  0.91093253744880D+04,  0.55208032572654D+04,  0.33459413680397D+04,&
                    0.20278432533574D+04,  0.12289959111257D+04,  0.74484600674283D+03,  0.45142182226838D+03,&
                    0.27358898319296D+03,  0.16581150496543D+03,  0.10049182119117D+03,  0.60904134055255D+02,&
                    0.36911596397124D+02,  0.22370664483105D+02,  0.13557978474609D+02,  0.82169566512784D+01,&
                    0.49799737280475D+01,  0.30181658957864D+01,  0.18291914519917D+01/)
    coe(1:nfun)=(/  0.14127462515354D+06, -0.24830349604034D+06,  0.36025635204601D+06, -0.23577440242529D+06,&
                    0.21480186209726D+06, -0.86648581275940D+05,  0.95509597839117D+05, -0.13100367268741D+05,&
                    0.47186743056983D+05,  0.12050873353019D+05,  0.30695285723507D+05,  0.19369855762698D+05,&
                    0.24456577157136D+05,  0.19622357204432D+05,  0.18780666271681D+05,  0.13533590757252D+05,&
                    0.84881918196618D+04,  0.26024196909538D+04, -0.47917191598541D+03, -0.46886878215318D+03,&
                    0.14354936974938D+04,  0.19210365677906D+04,  0.35316193850922D+03, -0.43841146864088D+03,&
                    0.94693675889223D+02,  0.26142148249726D+03,  0.57018100271900D+02,  0.23049533201579D+01,&
                    0.93278363558966D-01, -0.99324634205686D-02,  0.93371555228572D-03/)
  case(056)
    nfun = 31
    alf(1:nfun)=(/  0.61203673492837D+07,  0.37093135450204D+07,  0.22480688151639D+07,  0.13624659485842D+07,&
                    0.82573693853586D+06,  0.50044662941568D+06,  0.30330098752465D+06,  0.18381878031797D+06,&
                    0.11140532140483D+06,  0.67518376608988D+05,  0.40920228247872D+05,  0.24800138332043D+05,&
                    0.15030386867905D+05,  0.91093253744880D+04,  0.55208032572654D+04,  0.33459413680397D+04,&
                    0.20278432533574D+04,  0.12289959111257D+04,  0.74484600674283D+03,  0.45142182226838D+03,&
                    0.27358898319296D+03,  0.16581150496543D+03,  0.10049182119117D+03,  0.60904134055255D+02,&
                    0.36911596397124D+02,  0.22370664483105D+02,  0.13557978474609D+02,  0.82169566512784D+01,&
                    0.49799737280475D+01,  0.30181658957864D+01,  0.18291914519917D+01/)
    coe(1:nfun)=(/  0.66960208778477D+06, -0.10455870175433D+07,  0.11101300002646D+07, -0.77732684966612D+06,&
                    0.57034467069244D+06, -0.29642585175800D+06,  0.22218655845976D+06, -0.81332824291110D+05,&
                    0.89380691395491D+05, -0.80839062641412D+04,  0.45096072581232D+05,  0.14289915163189D+05,&
                    0.29877785326716D+05,  0.18790796894876D+05,  0.20866888176595D+05,  0.13302809468186D+05,&
                    0.88364344083962D+04,  0.21459137824348D+04, -0.51595064846415D+03, -0.42381605927908D+03,&
                    0.17139782658922D+04,  0.19426165130901D+04,  0.23864920137829D+03, -0.45958329966294D+03,&
                    0.15697898593418D+03,  0.26324576076591D+03,  0.48003671022003D+02,  0.16728154758673D+01,&
                    0.51680954544281D-01, -0.49453332923122D-02,  0.43065622439187D-03/)
  case(057)
    nfun = 29
    alf(1:nfun)=(/  0.61203673492837D+07,  0.37093135450204D+07,  0.22480688151639D+07,  0.13624659485842D+07,&
                    0.82573693853586D+06,  0.50044662941568D+06,  0.30330098752465D+06,  0.18381878031797D+06,&
                    0.11140532140483D+06,  0.67518376608988D+05,  0.40920228247872D+05,  0.24800138332043D+05,&
                    0.15030386867905D+05,  0.91093253744880D+04,  0.55208032572654D+04,  0.33459413680397D+04,&
                    0.20278432533574D+04,  0.12289959111257D+04,  0.74484600674283D+03,  0.45142182226838D+03,&
                    0.27358898319296D+03,  0.16581150496543D+03,  0.10049182119117D+03,  0.60904134055255D+02,&
                    0.36911596397124D+02,  0.22370664483105D+02,  0.13557978474609D+02,  0.82169566512784D+01,&
                    0.49799737280475D+01/)
    coe(1:nfun)=(/  0.97350602015686D+06, -0.16217065590591D+07,  0.17083176705589D+07, -0.12328757168179D+07,&
                    0.87971369039917D+06, -0.48118358384132D+06,  0.33557971885204D+06, -0.14196324456692D+06,&
                    0.12768684672284D+06, -0.25856256397367D+05,  0.58422101294816D+05,  0.99622312273681D+04,&
                    0.35016248765796D+05,  0.18190176015146D+05,  0.22851578417473D+05,  0.13104687249735D+05,&
                    0.91171434461828D+04,  0.16936216821419D+04, -0.55160620135488D+03, -0.34352994944714D+03,&
                    0.19948286340943D+04,  0.19435993318130D+04,  0.11710276438092D+03, -0.46713150491143D+03,&
                    0.22032294454753D+03,  0.26038384981424D+03,  0.39715799268441D+02,  0.12481437766623D+01,&
                    0.14130782814618D-01/)
  case(058)
    nfun = 30
    alf(1:nfun)=(/  0.61203673492837D+07,  0.37093135450204D+07,  0.22480688151639D+07,  0.13624659485842D+07,&
                    0.82573693853586D+06,  0.50044662941568D+06,  0.30330098752465D+06,  0.18381878031797D+06,&
                    0.11140532140483D+06,  0.67518376608988D+05,  0.40920228247872D+05,  0.24800138332043D+05,&
                    0.15030386867905D+05,  0.91093253744880D+04,  0.55208032572654D+04,  0.33459413680397D+04,&
                    0.20278432533574D+04,  0.12289959111257D+04,  0.74484600674283D+03,  0.45142182226838D+03,&
                    0.27358898319296D+03,  0.16581150496543D+03,  0.10049182119117D+03,  0.60904134055255D+02,&
                    0.36911596397124D+02,  0.22370664483105D+02,  0.13557978474609D+02,  0.82169566512784D+01,&
                    0.49799737280475D+01,  0.30181658957864D+01/)
    coe(1:nfun)=(/  0.12399559785042D+07, -0.21183533887024D+07,  0.22244715820045D+07, -0.16236214976654D+07,&
                    0.11463090282912D+07, -0.63908448478651D+06,  0.43354312128973D+06, -0.19333719256389D+06,&
                    0.16103625723058D+06, -0.40569186647892D+05,  0.70252186896473D+05,  0.66533950546980D+04,&
                    0.39710803531125D+05,  0.17919479086418D+05,  0.24670446136549D+05,  0.12973333651271D+05,&
                    0.93100419362998D+04,  0.12562180625852D+04, -0.58681559483916D+03, -0.22657237282710D+03,&
                    0.22754394246101D+04,  0.19237522780745D+04, -0.10599077305815D+02, -0.45996975353091D+03,&
                    0.28368239160104D+03,  0.25293990251069D+03,  0.32350096053409D+02,  0.94022444830807D+00,&
                   -0.31279043363384D-02,  0.11637145173893D-02/)
  case(059)
    nfun = 30
    alf(1:nfun)=(/  0.61203673492837D+07,  0.37093135450204D+07,  0.22480688151639D+07,  0.13624659485842D+07,&
                    0.82573693853586D+06,  0.50044662941568D+06,  0.30330098752465D+06,  0.18381878031797D+06,&
                    0.11140532140483D+06,  0.67518376608988D+05,  0.40920228247872D+05,  0.24800138332043D+05,&
                    0.15030386867905D+05,  0.91093253744880D+04,  0.55208032572654D+04,  0.33459413680397D+04,&
                    0.20278432533574D+04,  0.12289959111257D+04,  0.74484600674283D+03,  0.45142182226838D+03,&
                    0.27358898319296D+03,  0.16581150496543D+03,  0.10049182119117D+03,  0.60904134055255D+02,&
                    0.36911596397124D+02,  0.22370664483105D+02,  0.13557978474609D+02,  0.82169566512784D+01,&
                    0.49799737280475D+01,  0.30181658957864D+01/)
    coe(1:nfun)=(/  0.12455648135490D+07, -0.22557069506645D+07,  0.24245119471779D+07, -0.17904386131706D+07,&
                    0.12712499386802D+07, -0.71193750003529D+06,  0.48274565698743D+06, -0.21664856159472D+06,&
                    0.17892900293434D+06, -0.46370119090438D+05,  0.77353677515149D+05,  0.61525090377033D+04,&
                    0.42979199476853D+05,  0.18515381789263D+05,  0.26025274996785D+05,  0.13067724869714D+05,&
                    0.93251879901113D+04,  0.88627301707352D+03, -0.64758291210420D+03, -0.56081489498960D+02,&
                    0.25435224085303D+04,  0.18868261801734D+04, -0.14473122324207D+03, -0.43579214288705D+03,&
                    0.34440038502195D+03,  0.24179954228477D+03,  0.25887582529418D+02,  0.75454745591969D+00,&
                   -0.19260268840753D-01,  0.25591015699860D-02/)
  case(060)
    nfun = 30
    alf(1:nfun)=(/  0.61203673492837D+07,  0.37093135450204D+07,  0.22480688151639D+07,  0.13624659485842D+07,&
                    0.82573693853586D+06,  0.50044662941568D+06,  0.30330098752465D+06,  0.18381878031797D+06,&
                    0.11140532140483D+06,  0.67518376608988D+05,  0.40920228247872D+05,  0.24800138332043D+05,&
                    0.15030386867905D+05,  0.91093253744880D+04,  0.55208032572654D+04,  0.33459413680397D+04,&
                    0.20278432533574D+04,  0.12289959111257D+04,  0.74484600674283D+03,  0.45142182226838D+03,&
                    0.27358898319296D+03,  0.16581150496543D+03,  0.10049182119117D+03,  0.60904134055255D+02,&
                    0.36911596397124D+02,  0.22370664483105D+02,  0.13557978474609D+02,  0.82169566512784D+01,&
                    0.49799737280475D+01,  0.30181658957864D+01/)
    coe(1:nfun)=(/  0.11686142441607D+07, -0.21943146795425D+07,  0.24104928693562D+07, -0.17859398970346D+07,&
                    0.12811785662341D+07, -0.71295950729084D+06,  0.49009150236559D+06, -0.21555825814795D+06,&
                    0.18339288992834D+06, -0.44357879613042D+05,  0.80332511008486D+05,  0.81352352513075D+04,&
                    0.44997228669621D+05,  0.19877217487410D+05,  0.26965655099116D+05,  0.13355025968920D+05,&
                    0.91787327395375D+04,  0.57800689560687D+03, -0.72603552058619D+03,  0.16525590161572D+03,&
                    0.27956247535329D+04,  0.18307411537978D+04, -0.28030669277233D+03, -0.39531671863164D+03,&
                    0.40011445022604D+03,  0.22794755993862D+03,  0.20427788583937D+02,  0.62963688219088D+00,&
                   -0.31948315817431D-01,  0.36103491068928D-02/)
  case(061)
    nfun = 30
    alf(1:nfun)=(/  0.61203673492837D+07,  0.37093135450204D+07,  0.22480688151639D+07,  0.13624659485842D+07,&
                    0.82573693853586D+06,  0.50044662941568D+06,  0.30330098752465D+06,  0.18381878031797D+06,&
                    0.11140532140483D+06,  0.67518376608988D+05,  0.40920228247872D+05,  0.24800138332043D+05,&
                    0.15030386867905D+05,  0.91093253744880D+04,  0.55208032572654D+04,  0.33459413680397D+04,&
                    0.20278432533574D+04,  0.12289959111257D+04,  0.74484600674283D+03,  0.45142182226838D+03,&
                    0.27358898319296D+03,  0.16581150496543D+03,  0.10049182119117D+03,  0.60904134055255D+02,&
                    0.36911596397124D+02,  0.22370664483105D+02,  0.13557978474609D+02,  0.82169566512784D+01,&
                    0.49799737280475D+01,  0.30181658957864D+01/)
    coe(1:nfun)=(/  0.10054413687067D+07, -0.19187609641657D+07,  0.21639733552961D+07, -0.15946848467875D+07,&
                    0.11656250262632D+07, -0.63558616462731D+06,  0.45170049569583D+06, -0.18775341455841D+06,&
                    0.17311418805444D+06, -0.33772632696271D+05,  0.78779583515450D+05,  0.12833576177478D+05,&
                    0.45638577072926D+05,  0.22074134622138D+05,  0.27448682375964D+05,  0.13855095259789D+05,&
                    0.88587440818311D+04,  0.34027344572544D+03, -0.82324964854866D+03,  0.43910939352994D+03,&
                    0.30278001154009D+04,  0.17562428740239D+04, -0.41565372865451D+03, -0.33865754906869D+03,&
                    0.44990634150897D+03,  0.21195269522085D+03,  0.15893552759391D+02,  0.54279644292092D+00,&
                   -0.40030498459942D-01,  0.42527155156904D-02/)
  case(062)
    nfun = 30
    alf(1:nfun)=(/  0.61203673492837D+07,  0.37093135450204D+07,  0.22480688151639D+07,  0.13624659485842D+07,&
                    0.82573693853586D+06,  0.50044662941568D+06,  0.30330098752465D+06,  0.18381878031797D+06,&
                    0.11140532140483D+06,  0.67518376608988D+05,  0.40920228247872D+05,  0.24800138332043D+05,&
                    0.15030386867905D+05,  0.91093253744880D+04,  0.55208032572654D+04,  0.33459413680397D+04,&
                    0.20278432533574D+04,  0.12289959111257D+04,  0.74484600674283D+03,  0.45142182226838D+03,&
                    0.27358898319296D+03,  0.16581150496543D+03,  0.10049182119117D+03,  0.60904134055255D+02,&
                    0.36911596397124D+02,  0.22370664483105D+02,  0.13557978474609D+02,  0.82169566512784D+01,&
                    0.49799737280475D+01,  0.30181658957864D+01/)
    coe(1:nfun)=(/  0.75209389159966D+06, -0.13905603318138D+07,  0.16353400733843D+07, -0.11738692222328D+07,&
                    0.89473947466612D+06, -0.46096732004833D+06,  0.35640600735974D+06, -0.12675678467441D+06,&
                    0.14442088889092D+06, -0.12525664184451D+05,  0.71532003404140D+05,  0.20899879720859D+05,&
                    0.44539606149733D+05,  0.25308074803978D+05,  0.27350927344320D+05,  0.14638724670109D+05,&
                    0.83195961882807D+04,  0.20445300150686D+03, -0.95334739195742D+03,  0.77310922323924D+03,&
                    0.32334208720546D+04,  0.16640304813758D+04, -0.54740530493042D+03, -0.26721353819284D+03,&
                    0.49293560712201D+03,  0.19441694584944D+03,  0.12259968282841D+02,  0.46192394572745D+00,&
                   -0.41525024151952D-01,  0.42626156931789D-02/)
  case(063)
    nfun = 30
    alf(1:nfun)=(/  0.61203673492837D+07,  0.37093135450204D+07,  0.22480688151639D+07,  0.13624659485842D+07,&
                    0.82573693853586D+06,  0.50044662941568D+06,  0.30330098752465D+06,  0.18381878031797D+06,&
                    0.11140532140483D+06,  0.67518376608988D+05,  0.40920228247872D+05,  0.24800138332043D+05,&
                    0.15030386867905D+05,  0.91093253744880D+04,  0.55208032572654D+04,  0.33459413680397D+04,&
                    0.20278432533574D+04,  0.12289959111257D+04,  0.74484600674283D+03,  0.45142182226838D+03,&
                    0.27358898319296D+03,  0.16581150496543D+03,  0.10049182119117D+03,  0.60904134055255D+02,&
                    0.36911596397124D+02,  0.22370664483105D+02,  0.13557978474609D+02,  0.82169566512784D+01,&
                    0.49799737280475D+01,  0.30181658957864D+01/)
    coe(1:nfun)=(/  0.50489510407782D+06, -0.87949635469055D+06,  0.11284156761885D+07, -0.76994132521439D+06,&
                    0.63621908128595D+06, -0.29328733294702D+06,  0.26578697495806D+06, -0.67976477155566D+05,&
                    0.11741601215678D+06,  0.81044001185894D+04,  0.64902813899979D+05,  0.28825527874544D+05,&
                    0.43657633226294D+05,  0.28476610592354D+05,  0.27298429431831D+05,  0.15334164445469D+05,&
                    0.77824541993383D+04,  0.42832725947490D+02, -0.10400867712270D+04,  0.11285812137657D+04,&
                    0.34247408332570D+04,  0.15502720593002D+04, -0.67282987318662D+03, -0.18116022330749D+03,&
                    0.52763374159397D+03,  0.17635636821384D+03,  0.93072911210229D+01,  0.40837179985147D+00,&
                   -0.42967356673671D-01,  0.43182949390683D-02/)
  case(064)
    nfun = 30
    alf(1:nfun)=(/  0.61203673492837D+07,  0.37093135450204D+07,  0.22480688151639D+07,  0.13624659485842D+07,&
                    0.82573693853586D+06,  0.50044662941568D+06,  0.30330098752465D+06,  0.18381878031797D+06,&
                    0.11140532140483D+06,  0.67518376608988D+05,  0.40920228247872D+05,  0.24800138332043D+05,&
                    0.15030386867905D+05,  0.91093253744880D+04,  0.55208032572654D+04,  0.33459413680397D+04,&
                    0.20278432533574D+04,  0.12289959111257D+04,  0.74484600674283D+03,  0.45142182226838D+03,&
                    0.27358898319296D+03,  0.16581150496543D+03,  0.10049182119117D+03,  0.60904134055255D+02,&
                    0.36911596397124D+02,  0.22370664483105D+02,  0.13557978474609D+02,  0.82169566512784D+01,&
                    0.49799737280475D+01,  0.30181658957864D+01/)
    coe(1:nfun)=(/  0.20448366819382D+06, -0.19637080922318D+06,  0.42499940884495D+06, -0.20374350933743D+06,&
                    0.26767130729580D+06, -0.56217835599661D+05,  0.13471882558525D+06,  0.14626479585409D+05,&
                    0.77332351375222D+05,  0.36440551001489D+05,  0.54243603585899D+05,  0.39205550455920D+05,&
                    0.41540476793844D+05,  0.32394571968719D+05,  0.26813401276451D+05,  0.16221608407599D+05,&
                    0.70766397443381D+04, -0.37958143811440D+02, -0.11394138360104D+04,  0.15327368860971D+04,&
                    0.35864238249534D+04,  0.14181573846430D+04, -0.78708456904683D+03, -0.84406703750079D+02,&
                    0.55439090446717D+03,  0.15809038323317D+03,  0.70576566101390D+01,  0.34082919654202D+00,&
                   -0.38910596517326D-01,  0.38300558625508D-02/)
  case(065)
    nfun = 31
    alf(1:nfun)=(/  0.10098606126318D+08,  0.61203673492837D+07,  0.37093135450204D+07,  0.22480688151639D+07,&
                    0.13624659485842D+07,  0.82573693853586D+06,  0.50044662941568D+06,  0.30330098752465D+06,&
                    0.18381878031797D+06,  0.11140532140483D+06,  0.67518376608988D+05,  0.40920228247872D+05,&
                    0.24800138332043D+05,  0.15030386867905D+05,  0.91093253744880D+04,  0.55208032572654D+04,&
                    0.33459413680397D+04,  0.20278432533574D+04,  0.12289959111257D+04,  0.74484600674283D+03,&
                    0.45142182226838D+03,  0.27358898319296D+03,  0.16581150496543D+03,  0.10049182119117D+03,&
                    0.60904134055255D+02,  0.36911596397124D+02,  0.22370664483105D+02,  0.13557978474609D+02,&
                    0.82169566512784D+01,  0.49799737280475D+01,  0.30181658957864D+01/)
    coe(1:nfun)=(/  0.12042862954330D+06, -0.22059294483376D+06,  0.57471182675362D+06, -0.32642445057869D+06,&
                    0.38900944550610D+06, -0.11379157133913D+06,  0.18844054389143D+06,  0.26664088642597D+03,&
                    0.99586464475989D+05,  0.36440239348412D+05,  0.65578191436321D+05,  0.43446575545326D+05,&
                    0.49870797623917D+05,  0.39405147541877D+05,  0.36377918589104D+05,  0.26279424764598D+05,&
                    0.17078157249706D+05,  0.63307799094426D+04, -0.10920209481916D+03, -0.12083245353083D+04,&
                    0.19633158894592D+04,  0.37273816917082D+04,  0.12640272010535D+04, -0.88807340338192D+03,&
                    0.22892628345049D+02,  0.57301967898489D+03,  0.13987186008123D+03,  0.53647055491774D+01,&
                    0.27362575358984D+00, -0.31581199072150D-01,  0.30648343190620D-02/)
  case(066)
    nfun = 31
    alf(1:nfun)=(/  0.10098606126318D+08,  0.61203673492837D+07,  0.37093135450204D+07,  0.22480688151639D+07,&
                    0.13624659485842D+07,  0.82573693853586D+06,  0.50044662941568D+06,  0.30330098752465D+06,&
                    0.18381878031797D+06,  0.11140532140483D+06,  0.67518376608988D+05,  0.40920228247872D+05,&
                    0.24800138332043D+05,  0.15030386867905D+05,  0.91093253744880D+04,  0.55208032572654D+04,&
                    0.33459413680397D+04,  0.20278432533574D+04,  0.12289959111257D+04,  0.74484600674283D+03,&
                    0.45142182226838D+03,  0.27358898319296D+03,  0.16581150496543D+03,  0.10049182119117D+03,&
                    0.60904134055255D+02,  0.36911596397124D+02,  0.22370664483105D+02,  0.13557978474609D+02,&
                    0.82169566512784D+01,  0.49799737280475D+01,  0.30181658957864D+01/)
    coe(1:nfun)=(/  0.11954012044754D+07, -0.18384820925198D+07,  0.21079142987843D+07, -0.14336951446743D+07,&
                    0.11221913924179D+07, -0.54460968292713D+06,  0.45041044602919D+06, -0.13933207652259D+06,&
                    0.18672550627774D+06, -0.47935303293467D+04,  0.95167401479602D+05,  0.32736039689764D+05,&
                    0.60692616908461D+05,  0.37317585905351D+05,  0.40383842498770D+05,  0.25711626830168D+05,&
                    0.17890540323532D+05,  0.55532582091950D+04, -0.17059521572781D+03, -0.12420576068180D+04,&
                    0.24154376746417D+04,  0.38431094357459D+04,  0.10902231864541D+04, -0.97135299382571D+03,&
                    0.13656105693858D+03,  0.58346569598817D+03,  0.12246135474728D+03,  0.41237746260049D+01,&
                    0.20160528395263D+00, -0.22513928018839D-01,  0.21345200065923D-02/)
  case(067)
    nfun = 31
    alf(1:nfun)=(/  0.10098606126318D+08,  0.61203673492837D+07,  0.37093135450204D+07,  0.22480688151639D+07,&
                    0.13624659485842D+07,  0.82573693853586D+06,  0.50044662941568D+06,  0.30330098752465D+06,&
                    0.18381878031797D+06,  0.11140532140483D+06,  0.67518376608988D+05,  0.40920228247872D+05,&
                    0.24800138332043D+05,  0.15030386867905D+05,  0.91093253744880D+04,  0.55208032572654D+04,&
                    0.33459413680397D+04,  0.20278432533574D+04,  0.12289959111257D+04,  0.74484600674283D+03,&
                    0.45142182226838D+03,  0.27358898319296D+03,  0.16581150496543D+03,  0.10049182119117D+03,&
                    0.60904134055255D+02,  0.36911596397124D+02,  0.22370664483105D+02,  0.13557978474609D+02,&
                    0.82169566512784D+01,  0.49799737280475D+01,  0.30181658957864D+01/)
    coe(1:nfun)=(/  0.22241908075161D+07, -0.33456458595400D+07,  0.35247142877951D+07, -0.24469033084450D+07,&
                    0.17933945668840D+07, -0.93600391687536D+06,  0.69010242810750D+06, -0.26539921581340D+06,&
                    0.26684431647968D+06, -0.41582747905046D+05,  0.12268728806522D+06,  0.23489641043007D+05,&
                    0.70923746029437D+05,  0.35690546930920D+05,  0.44180070154641D+05,  0.25233601307261D+05,&
                    0.18587037676683D+05,  0.47842535890420D+04, -0.24020464046299D+03, -0.12273481636271D+04,&
                    0.28806473540353D+04,  0.39346310045942D+04,  0.89684589202746D+03, -0.10345325818255D+04,&
                    0.25461675659411D+03,  0.58614994368817D+03,  0.10605832760002D+03,  0.32366891920374D+01,&
                    0.12958802027720D+00, -0.12529614498646D-01,  0.11299434575989D-02/)
  case(068)
    nfun = 31
    alf(1:nfun)=(/  0.10098606126318D+08,  0.61203673492837D+07,  0.37093135450204D+07,  0.22480688151639D+07,&
                    0.13624659485842D+07,  0.82573693853586D+06,  0.50044662941568D+06,  0.30330098752465D+06,&
                    0.18381878031797D+06,  0.11140532140483D+06,  0.67518376608988D+05,  0.40920228247872D+05,&
                    0.24800138332043D+05,  0.15030386867905D+05,  0.91093253744880D+04,  0.55208032572654D+04,&
                    0.33459413680397D+04,  0.20278432533574D+04,  0.12289959111257D+04,  0.74484600674283D+03,&
                    0.45142182226838D+03,  0.27358898319296D+03,  0.16581150496543D+03,  0.10049182119117D+03,&
                    0.60904134055255D+02,  0.36911596397124D+02,  0.22370664483105D+02,  0.13557978474609D+02,&
                    0.82169566512784D+01,  0.49799737280475D+01,  0.18291914519917D+01/)
    coe(1:nfun)=(/  0.30843283212572D+07, -0.45854836144391D+07,  0.46920789546874D+07, -0.32746714792163D+07,&
                    0.23456084580650D+07, -0.12543004181681D+07,  0.88815556019118D+06, -0.36690387140800D+06,&
                    0.33374780030461D+06, -0.70439997816928D+05,  0.14620995392120D+06,  0.16783275527449D+05,&
                    0.79968911636926D+05,  0.34851602170803D+05,  0.47577658414983D+05,  0.24944460848954D+05,&
                    0.19108386505995D+05,  0.40587146538019D+04, -0.33326190378625D+03, -0.11518210322230D+04,&
                    0.33497802025042D+04,  0.40025793111392D+04,  0.68470109453370D+03, -0.10749732983824D+04,&
                    0.37431859852869D+03,  0.58167304519280D+03,  0.90884592128310D+02,  0.26200403530971D+01,&
                    0.59164096872806D-01, -0.20348668811394D-02,  0.21651087644070D-04/)
  case(069)
    nfun = 30
    alf(1:nfun)=(/  0.10098606126318D+08,  0.61203673492837D+07,  0.37093135450204D+07,  0.22480688151639D+07,&
                    0.13624659485842D+07,  0.82573693853586D+06,  0.50044662941568D+06,  0.30330098752465D+06,&
                    0.18381878031797D+06,  0.11140532140483D+06,  0.67518376608988D+05,  0.40920228247872D+05,&
                    0.24800138332043D+05,  0.15030386867905D+05,  0.91093253744880D+04,  0.55208032572654D+04,&
                    0.33459413680397D+04,  0.20278432533574D+04,  0.12289959111257D+04,  0.74484600674283D+03,&
                    0.45142182226838D+03,  0.27358898319296D+03,  0.16581150496543D+03,  0.10049182119117D+03,&
                    0.60904134055255D+02,  0.36911596397124D+02,  0.22370664483105D+02,  0.13557978474609D+02,&
                    0.82169566512784D+01,  0.49799737280475D+01/)
    coe(1:nfun)=(/  0.33666760717545D+07, -0.52522169783478D+07,  0.54626858153687D+07, -0.38721858096523D+07,&
                    0.27710750919628D+07, -0.15025744696379D+07,  0.10488930748816D+07, -0.44716281793284D+06,&
                    0.38963336509633D+06, -0.92644676714897D+05,  0.16648963343841D+06,  0.12208420921564D+05,&
                    0.88064258291230D+05,  0.34666762885913D+05,  0.50648302652173D+05,  0.24797722846325D+05,&
                    0.19477345119961D+05,  0.33655267083924D+04, -0.43894239679631D+03, -0.10181345052551D+04,&
                    0.38219750267641D+04,  0.40429580781518D+04,  0.45747421914863D+03, -0.10918161726448D+04,&
                    0.49377054522939D+03,  0.57044445445146D+03,  0.77195403915955D+02,  0.21463585753781D+01,&
                    0.16311594890112D-01,  0.16179339054077D-02/)
  case(070)
    nfun = 30
    alf(1:nfun)=(/  0.10098606126318D+08,  0.61203673492837D+07,  0.37093135450204D+07,  0.22480688151639D+07,&
                    0.13624659485842D+07,  0.82573693853586D+06,  0.50044662941568D+06,  0.30330098752465D+06,&
                    0.18381878031797D+06,  0.11140532140483D+06,  0.67518376608988D+05,  0.40920228247872D+05,&
                    0.24800138332043D+05,  0.15030386867905D+05,  0.91093253744880D+04,  0.55208032572654D+04,&
                    0.33459413680397D+04,  0.20278432533574D+04,  0.12289959111257D+04,  0.74484600674283D+03,&
                    0.45142182226838D+03,  0.27358898319296D+03,  0.16581150496543D+03,  0.10049182119117D+03,&
                    0.60904134055255D+02,  0.36911596397124D+02,  0.22370664483105D+02,  0.13557978474609D+02,&
                    0.82169566512784D+01,  0.49799737280475D+01/)
    coe(1:nfun)=(/  0.33948238073254D+07, -0.55368527391739D+07,  0.58852039139214D+07, -0.42155574496899D+07,&
                    0.30345356791763D+07, -0.16518801116056D+07,  0.11531306881471D+07, -0.49433674414873D+06,&
                    0.42764167559469D+06, -0.10427861613953D+06,  0.18134358637375D+06,  0.10996426322520D+05,&
                    0.94517722235680D+05,  0.35521460999399D+05,  0.53168192043465D+05,  0.24912023634862D+05,&
                    0.19624680357439D+05,  0.27454318130165D+04, -0.57482703674445D+03, -0.81341756767081D+03,&
                    0.42867856895807D+04,  0.40575799214228D+04,  0.21617860902663D+03, -0.10828514585437D+04,&
                    0.60992089529517D+03,  0.55358109704360D+03,  0.64947545954754D+02,  0.18123503006964D+01,&
                   -0.19921444604279D-01,  0.45315032404574D-02/)
  case(071)
    nfun = 30
    alf(1:nfun)=(/  0.10098606126318D+08,  0.61203673492837D+07,  0.37093135450204D+07,  0.22480688151639D+07,&
                    0.13624659485842D+07,  0.82573693853586D+06,  0.50044662941568D+06,  0.30330098752465D+06,&
                    0.18381878031797D+06,  0.11140532140483D+06,  0.67518376608988D+05,  0.40920228247872D+05,&
                    0.24800138332043D+05,  0.15030386867905D+05,  0.91093253744880D+04,  0.55208032572654D+04,&
                    0.33459413680397D+04,  0.20278432533574D+04,  0.12289959111257D+04,  0.74484600674283D+03,&
                    0.45142182226838D+03,  0.27358898319296D+03,  0.16581150496543D+03,  0.10049182119117D+03,&
                    0.60904134055255D+02,  0.36911596397124D+02,  0.22370664483105D+02,  0.13557978474609D+02,&
                    0.82169566512784D+01,  0.49799737280475D+01/)
    coe(1:nfun)=(/  0.32966458585186D+07, -0.55002606056824D+07,  0.59596595930004D+07, -0.42777299726429D+07,&
                    0.31095896093397D+07, -0.16827840799856D+07,  0.11887485762172D+07, -0.50112708742261D+06,&
                    0.44362069313681D+06, -0.10293836989093D+06,  0.18944669689816D+06,  0.13897405278504D+05,&
                    0.98920502502248D+05,  0.37637280316472D+05,  0.55010202346664D+05,  0.25349871897465D+05,&
                    0.19513555685755D+05,  0.22212609949205D+04, -0.74822597591020D+03, -0.53083439489570D+03,&
                    0.47364381321017D+04,  0.40457411371756D+04, -0.35412143812835D+02, -0.10476582324910D+04,&
                    0.72020319177261D+03,  0.53204363875022D+03,  0.54200670049368D+02,  0.15699930021829D+01,&
                   -0.49819714069145D-01,  0.68595627782599D-02/)
end select

return
end

!-----------------------------------------------------------------------------------------------------------------------------------
!---  EDF data library: ncore = 36, iz = 37- 54
!-----------------------------------------------------------------------------------------------------------------------------------
subroutine EDFLIB36(iz,nfun,alf,coe)
implicit real(kind=8) (a-h,o-z)
real(kind=8) :: alf(*),coe(*)

select case(iz)
  case(037)
    nfun = 35
    alf(1:nfun)=(/  0.37093135450204D+07,  0.22480688151639D+07,  0.13624659485842D+07,  0.82573693853586D+06,&
                    0.50044662941568D+06,  0.30330098752465D+06,  0.18381878031797D+06,  0.11140532140483D+06,&
                    0.67518376608988D+05,  0.40920228247872D+05,  0.24800138332043D+05,  0.15030386867905D+05,&
                    0.91093253744880D+04,  0.55208032572654D+04,  0.33459413680397D+04,  0.20278432533574D+04,&
                    0.12289959111257D+04,  0.74484600674283D+03,  0.45142182226838D+03,  0.27358898319296D+03,&
                    0.16581150496543D+03,  0.10049182119117D+03,  0.60904134055255D+02,  0.36911596397124D+02,&
                    0.22370664483105D+02,  0.13557978474609D+02,  0.82169566512784D+01,  0.49799737280475D+01,&
                    0.30181658957864D+01,  0.18291914519917D+01,  0.11086008799950D+01,  0.67187932120909D+00,&
                    0.40719958861157D+00,  0.24678762946156D+00,  0.14956826027973D+00/)
    coe(1:nfun)=(/  0.27077586262417D+06, -0.33064618082237D+06,  0.28457284845591D+06, -0.18760102149153D+06,&
                    0.12298773087013D+06, -0.66553849404693D+05,  0.44070093890011D+05, -0.18932204411596D+05,&
                    0.16632627427906D+05, -0.29690986310914D+04,  0.82339743432999D+04,  0.24068985592015D+04,&
                    0.60333036276605D+04,  0.43465410869913D+04,  0.53319447200662D+04,  0.43022613178478D+04,&
                    0.35683468431944D+04,  0.18706471225088D+04,  0.50671973296197D+03, -0.23398237783919D+03,&
                   -0.28893899860166D+02,  0.40451936584525D+03,  0.43505400566210D+03,  0.50080025022209D+02,&
                   -0.81032431745271D+02,  0.10993888552649D+02,  0.45030766158317D+02,  0.17640034460872D+02,&
                   -0.38995812869393D+00, -0.71154321790669D+00,  0.70059522603984D+00,  0.50719404419924D+00,&
                    0.12203231160850D+00,  0.98109730258891D-02,  0.23838060267378D-03/)
  case(038)
    nfun = 34
    alf(1:nfun)=(/  0.37093135450204D+07,  0.22480688151639D+07,  0.13624659485842D+07,  0.82573693853586D+06,&
                    0.50044662941568D+06,  0.30330098752465D+06,  0.18381878031797D+06,  0.11140532140483D+06,&
                    0.67518376608988D+05,  0.40920228247872D+05,  0.24800138332043D+05,  0.15030386867905D+05,&
                    0.91093253744880D+04,  0.55208032572654D+04,  0.33459413680397D+04,  0.20278432533574D+04,&
                    0.12289959111257D+04,  0.74484600674283D+03,  0.45142182226838D+03,  0.27358898319296D+03,&
                    0.16581150496543D+03,  0.10049182119117D+03,  0.60904134055255D+02,  0.36911596397124D+02,&
                    0.22370664483105D+02,  0.13557978474609D+02,  0.82169566512784D+01,  0.49799737280475D+01,&
                    0.30181658957864D+01,  0.18291914519917D+01,  0.11086008799950D+01,  0.67187932120909D+00,&
                    0.40719958861157D+00,  0.24678762946156D+00/)
    coe(1:nfun)=(/  0.92649907872891D+06, -0.87589745788622D+06,  0.59962019295359D+06, -0.32763803241444D+06,&
                    0.18455209425664D+06, -0.91616930719733D+05,  0.56088254238725D+05, -0.23764359386325D+05,&
                    0.19788855124354D+05, -0.37982345926464D+04,  0.94562749771699D+04,  0.25727964225486D+04,&
                    0.67548997899033D+04,  0.47510518121924D+04,  0.58258535754979D+04,  0.45876089541223D+04,&
                    0.37217378371977D+04,  0.18284090170455D+04,  0.41117833795713D+03, -0.27745102447714D+03,&
                    0.29651485678856D+02,  0.48728383648780D+03,  0.43968243284016D+03,  0.79327590124230D+01,&
                   -0.84173038762368D+02,  0.25557946114983D+02,  0.50790894433203D+02,  0.14665719744797D+02,&
                   -0.21918690852633D+01, -0.29161497423445D+00,  0.10877346653892D+01,  0.49312038084393D+00,&
                    0.62003716233178D-01,  0.27884337559323D-02/)
  case(039)
    nfun = 35
    alf(1:nfun)=(/  0.37093135450204D+07,  0.22480688151639D+07,  0.13624659485842D+07,  0.82573693853586D+06,&
                    0.50044662941568D+06,  0.30330098752465D+06,  0.18381878031797D+06,  0.11140532140483D+06,&
                    0.67518376608988D+05,  0.40920228247872D+05,  0.24800138332043D+05,  0.15030386867905D+05,&
                    0.91093253744880D+04,  0.55208032572654D+04,  0.33459413680397D+04,  0.20278432533574D+04,&
                    0.12289959111257D+04,  0.74484600674283D+03,  0.45142182226838D+03,  0.27358898319296D+03,&
                    0.16581150496543D+03,  0.10049182119117D+03,  0.60904134055255D+02,  0.36911596397124D+02,&
                    0.22370664483105D+02,  0.13557978474609D+02,  0.82169566512784D+01,  0.49799737280475D+01,&
                    0.30181658957864D+01,  0.18291914519917D+01,  0.11086008799950D+01,  0.67187932120909D+00,&
                    0.40719958861157D+00,  0.24678762946156D+00,  0.14956826027973D+00/)
    coe(1:nfun)=(/  0.25728126156855D+06, -0.38745504814267D+06,  0.37423646884704D+06, -0.26719903008819D+06,&
                    0.18179971286809D+06, -0.10239776848239D+06,  0.67274214279413D+05, -0.30725115121722D+05,&
                    0.24998406528667D+05, -0.59947718696743D+04,  0.11603211252179D+05,  0.22564978235886D+04,&
                    0.78190546926912D+04,  0.50116043770555D+04,  0.64342350162892D+04,  0.48176847149425D+04,&
                    0.38902636376332D+04,  0.17512551815939D+04,  0.31999383338646D+03, -0.31583992015236D+03,&
                    0.10472413858279D+03,  0.56887317555350D+03,  0.43231892294167D+03, -0.35612535521172D+02,&
                   -0.81392563568255D+02,  0.41310924159778D+02,  0.54945680027943D+02,  0.10475927376668D+02,&
                   -0.32689051121644D+01,  0.42883488974820D+00,  0.12894006854594D+01,  0.39059573052200D+00,&
                    0.36784151663369D-01,  0.11436879108420D-02,  0.24768633012373D-04/)
  case(040)
    nfun = 34
    alf(1:nfun)=(/  0.37093135450204D+07,  0.22480688151639D+07,  0.13624659485842D+07,  0.82573693853586D+06,&
                    0.50044662941568D+06,  0.30330098752465D+06,  0.18381878031797D+06,  0.11140532140483D+06,&
                    0.67518376608988D+05,  0.40920228247872D+05,  0.24800138332043D+05,  0.15030386867905D+05,&
                    0.91093253744880D+04,  0.55208032572654D+04,  0.33459413680397D+04,  0.20278432533574D+04,&
                    0.12289959111257D+04,  0.74484600674283D+03,  0.45142182226838D+03,  0.27358898319296D+03,&
                    0.16581150496543D+03,  0.10049182119117D+03,  0.60904134055255D+02,  0.36911596397124D+02,&
                    0.22370664483105D+02,  0.13557978474609D+02,  0.82169566512784D+01,  0.49799737280475D+01,&
                    0.30181658957864D+01,  0.18291914519917D+01,  0.11086008799950D+01,  0.67187932120909D+00,&
                    0.40719958861157D+00,  0.24678762946156D+00/)
    coe(1:nfun)=(/  0.27731098589611D+06, -0.30810168117189D+06,  0.25593613228035D+06, -0.15830188579631D+06,&
                    0.10529835763955D+06, -0.52497974881530D+05,  0.38451864125729D+05, -0.13091201321661D+05,&
                    0.15978134553969D+05,  0.99899189323187D+01,  0.92267331489995D+04,  0.45359765989184D+04,&
                    0.75104919776209D+04,  0.60837568045594D+04,  0.66268943395382D+04,  0.52843299065363D+04,&
                    0.39093985412112D+04,  0.17301787046223D+04,  0.18403241433558D+03, -0.31942452246623D+03,&
                    0.18037084722718D+03,  0.65639284801277D+03,  0.40824465171796D+03, -0.76359313882349D+02,&
                   -0.74760830409539D+02,  0.58961492357881D+02,  0.56818049994264D+02,  0.56084338278681D+01,&
                   -0.37031039921779D+01,  0.12849077994181D+01,  0.13507387987492D+01,  0.28957981514841D+00,&
                    0.18866312577668D-01,  0.57930302104614D-03/)
  case(041)
    nfun = 35
    alf(1:nfun)=(/  0.61203673492837D+07,  0.37093135450204D+07,  0.22480688151639D+07,  0.13624659485842D+07,&
                    0.82573693853586D+06,  0.50044662941568D+06,  0.30330098752465D+06,  0.18381878031797D+06,&
                    0.11140532140483D+06,  0.67518376608988D+05,  0.40920228247872D+05,  0.24800138332043D+05,&
                    0.15030386867905D+05,  0.91093253744880D+04,  0.55208032572654D+04,  0.33459413680397D+04,&
                    0.20278432533574D+04,  0.12289959111257D+04,  0.74484600674283D+03,  0.45142182226838D+03,&
                    0.27358898319296D+03,  0.16581150496543D+03,  0.10049182119117D+03,  0.60904134055255D+02,&
                    0.36911596397124D+02,  0.22370664483105D+02,  0.13557978474609D+02,  0.82169566512784D+01,&
                    0.49799737280475D+01,  0.30181658957864D+01,  0.18291914519917D+01,  0.11086008799950D+01,&
                    0.67187932120909D+00,  0.40719958861157D+00,  0.24678762946156D+00/)
    coe(1:nfun)=(/  0.23617705263424D+06, -0.21541048860073D+06,  0.15755066280413D+06, -0.80418247838020D+05,&
                    0.56795489122152D+05, -0.22672618194818D+05,  0.22724761995196D+05, -0.32453903350830D+04,&
                    0.11376630678743D+05,  0.33813820275962D+04,  0.80349155210406D+04,  0.58332988883480D+04,&
                    0.74453801980056D+04,  0.69254172655381D+04,  0.73591112311119D+04,  0.67386545778238D+04,&
                    0.57985469062147D+04,  0.38809923984646D+04,  0.17077038311608D+04,  0.37883206660103D+02,&
                   -0.30468358836719D+03,  0.26653369486031D+03,  0.74102839529230D+03,  0.37036141469965D+03,&
                   -0.11393170323910D+03, -0.63113276114451D+02,  0.77284704571961D+02,  0.55998927064202D+02,&
                    0.55592386184071D+00, -0.32767950391021D+01,  0.20406586328425D+01,  0.12682528215308D+01,&
                    0.20940871291316D+00,  0.11216055708978D-01,  0.31291511099196D-03/)
  case(042)
    nfun = 35
    alf(1:nfun)=(/  0.61203673492837D+07,  0.37093135450204D+07,  0.22480688151639D+07,  0.13624659485842D+07,&
                    0.82573693853586D+06,  0.50044662941568D+06,  0.30330098752465D+06,  0.18381878031797D+06,&
                    0.11140532140483D+06,  0.67518376608988D+05,  0.40920228247872D+05,  0.24800138332043D+05,&
                    0.15030386867905D+05,  0.91093253744880D+04,  0.55208032572654D+04,  0.33459413680397D+04,&
                    0.20278432533574D+04,  0.12289959111257D+04,  0.74484600674283D+03,  0.45142182226838D+03,&
                    0.27358898319296D+03,  0.16581150496543D+03,  0.10049182119117D+03,  0.60904134055255D+02,&
                    0.36911596397124D+02,  0.22370664483105D+02,  0.13557978474609D+02,  0.82169566512784D+01,&
                    0.49799737280475D+01,  0.30181658957864D+01,  0.18291914519917D+01,  0.11086008799950D+01,&
                    0.67187932120909D+00,  0.40719958861157D+00,  0.24678762946156D+00/)
    coe(1:nfun)=(/  0.53804440444899D+06, -0.78618726617908D+06,  0.73785120222092D+06, -0.52544495274878D+06,&
                    0.35393447820950D+06, -0.20380341488624D+06,  0.13030424700725D+06, -0.63643939594269D+05,&
                    0.46620142427385D+05, -0.15187929869577D+05,  0.19502378996000D+05,  0.60634588082135D+03,&
                    0.11457740429759D+05,  0.57965196616743D+04,  0.89902134962371D+04,  0.66815672643576D+04,&
                    0.64071283451207D+04,  0.37759510871619D+04,  0.17000691955033D+04, -0.12538130243524D+03,&
                   -0.26529963236718D+03,  0.35984763307235D+03,  0.82202847187872D+03,  0.31863544925816D+03,&
                   -0.14659718856700D+03, -0.47031786386882D+02,  0.95971055880806D+02,  0.52221128918709D+02,&
                   -0.41086913896581D+01, -0.22300074509000D+01,  0.26340254823526D+01,  0.11327124524232D+01,&
                    0.14450087454561D+00,  0.61137242282072D-02,  0.15470668912565D-03/)
  case(043)
    nfun = 37
    alf(1:nfun)=(/  0.16662700108425D+08,  0.10098606126318D+08,  0.61203673492837D+07,  0.37093135450204D+07,&
                    0.22480688151639D+07,  0.13624659485842D+07,  0.82573693853586D+06,  0.50044662941568D+06,&
                    0.30330098752465D+06,  0.18381878031797D+06,  0.11140532140483D+06,  0.67518376608988D+05,&
                    0.40920228247872D+05,  0.24800138332043D+05,  0.15030386867905D+05,  0.91093253744880D+04,&
                    0.55208032572654D+04,  0.33459413680397D+04,  0.20278432533574D+04,  0.12289959111257D+04,&
                    0.74484600674283D+03,  0.45142182226838D+03,  0.27358898319296D+03,  0.16581150496543D+03,&
                    0.10049182119117D+03,  0.60904134055255D+02,  0.36911596397124D+02,  0.22370664483105D+02,&
                    0.13557978474609D+02,  0.82169566512784D+01,  0.49799737280475D+01,  0.30181658957864D+01,&
                    0.18291914519917D+01,  0.11086008799950D+01,  0.67187932120909D+00,  0.40719958861157D+00,&
                    0.24678762946156D+00/)
    coe(1:nfun)=(/  0.79504818817902D+06, -0.18622638873844D+07,  0.25671970963726D+07, -0.24117302803936D+07,&
                    0.18430900877409D+07, -0.12113281358118D+07,  0.76105883407855D+06, -0.43665001673901D+06,&
                    0.26350728897440D+06, -0.13709806995678D+06,  0.88905225922376D+05, -0.37472850218832D+05,&
                    0.33116085527442D+05, -0.57181659519337D+04,  0.16166955900285D+05,  0.43516297998279D+04,&
                    0.10850791982215D+05,  0.65247373169009D+04,  0.70678541932232D+04,  0.36161389736433D+04,&
                    0.16944255599619D+04, -0.29550722889893D+03, -0.20519503895429D+03,  0.46285363229370D+03,&
                    0.89497694465129D+03,  0.25541088447491D+03, -0.17417004010770D+03, -0.26160970519410D+02,&
                    0.11418967912377D+03,  0.45525082862509D+02, -0.80176259636553D+01, -0.75526228653939D+00,&
                    0.30980494176604D+01,  0.96455932488306D+00,  0.87262166902462D-01,  0.25600492627316D-02,&
                    0.53962617291514D-04/)
  case(044)
    nfun = 37
    alf(1:nfun)=(/  0.16662700108425D+08,  0.10098606126318D+08,  0.61203673492837D+07,  0.37093135450204D+07,&
                    0.22480688151639D+07,  0.13624659485842D+07,  0.82573693853586D+06,  0.50044662941568D+06,&
                    0.30330098752465D+06,  0.18381878031797D+06,  0.11140532140483D+06,  0.67518376608988D+05,&
                    0.40920228247872D+05,  0.24800138332043D+05,  0.15030386867905D+05,  0.91093253744880D+04,&
                    0.55208032572654D+04,  0.33459413680397D+04,  0.20278432533574D+04,  0.12289959111257D+04,&
                    0.74484600674283D+03,  0.45142182226838D+03,  0.27358898319296D+03,  0.16581150496543D+03,&
                    0.10049182119117D+03,  0.60904134055255D+02,  0.36911596397124D+02,  0.22370664483105D+02,&
                    0.13557978474609D+02,  0.82169566512784D+01,  0.49799737280475D+01,  0.30181658957864D+01,&
                    0.18291914519917D+01,  0.11086008799950D+01,  0.67187932120909D+00,  0.40719958861157D+00,&
                    0.24678762946156D+00/)
    coe(1:nfun)=(/  0.85857494994259D+06, -0.20964944354401D+07,  0.28977874217587D+07, -0.27179086246185D+07,&
                    0.20728815259247D+07, -0.13603367866344D+07,  0.85389789714980D+06, -0.48961842936337D+06,&
                    0.29543573516214D+06, -0.15374442053765D+06,  0.99666972710609D+05, -0.42135704797611D+05,&
                    0.37071323105864D+05, -0.66040944222361D+04,  0.17959897220446D+05,  0.45876911803931D+04,&
                    0.11836086144656D+05,  0.68739426355474D+04,  0.74417071200325D+04,  0.35853883511537D+04,&
                    0.15900483413942D+04, -0.41343514801256D+03, -0.15331285708377D+03,  0.58997286911934D+03,&
                    0.94639780230589D+03,  0.18703305582457D+03, -0.19633800095204D+03,  0.69984131220735D+00,&
                    0.12897415994191D+03,  0.36527909274986D+02, -0.10253000969007D+02,  0.95585173070333D+00,&
                    0.31337389050810D+01,  0.80291820282471D+00,  0.66652670249198D-01,  0.19076800205067D-02,&
                    0.35706322647489D-04/)
  case(045)
    nfun = 36
    alf(1:nfun)=(/  0.16662700108425D+08,  0.10098606126318D+08,  0.61203673492837D+07,  0.37093135450204D+07,&
                    0.22480688151639D+07,  0.13624659485842D+07,  0.82573693853586D+06,  0.50044662941568D+06,&
                    0.30330098752465D+06,  0.18381878031797D+06,  0.11140532140483D+06,  0.67518376608988D+05,&
                    0.40920228247872D+05,  0.24800138332043D+05,  0.15030386867905D+05,  0.91093253744880D+04,&
                    0.55208032572654D+04,  0.33459413680397D+04,  0.20278432533574D+04,  0.12289959111257D+04,&
                    0.74484600674283D+03,  0.45142182226838D+03,  0.27358898319296D+03,  0.16581150496543D+03,&
                    0.10049182119117D+03,  0.60904134055255D+02,  0.36911596397124D+02,  0.22370664483105D+02,&
                    0.13557978474609D+02,  0.82169566512784D+01,  0.49799737280475D+01,  0.30181658957864D+01,&
                    0.18291914519917D+01,  0.11086008799950D+01,  0.67187932120909D+00,  0.40719958861157D+00/)
    coe(1:nfun)=(/  0.14667856303692D+07, -0.25701331477242D+07,  0.32033957781677D+07, -0.28918694925747D+07,&
                    0.21797189306469D+07, -0.14207722418222D+07,  0.89119940344095D+06, -0.50926161563444D+06,&
                    0.30832996339965D+06, -0.15946475776160D+06,  0.10445476890063D+06, -0.43358667923093D+05,&
                    0.39239990170628D+05, -0.64013470185772D+04,  0.19246097557463D+05,  0.51780841838866D+04,&
                    0.12678742952410D+05,  0.73249144617431D+04,  0.77487347638747D+04,  0.35614330264393D+04,&
                    0.14545439021187D+04, -0.51403473698488D+03, -0.89471258678357D+02,  0.72981393922737D+03,&
                    0.98158566341884D+03,  0.11251806949603D+03, -0.21260254638465D+03,  0.32435617419031D+02,&
                    0.14086434209921D+03,  0.25773863946993D+02, -0.11271643113024D+02,  0.26176238700555D+01,&
                    0.31112035035825D+01,  0.65106187340899D+00,  0.43626122034802D-01,  0.12576688360886D-02/)
  case(046)
    nfun = 34
    alf(1:nfun)=(/  0.61203673492837D+07,  0.37093135450204D+07,  0.22480688151639D+07,  0.13624659485842D+07,&
                    0.82573693853586D+06,  0.50044662941568D+06,  0.30330098752465D+06,  0.18381878031797D+06,&
                    0.11140532140483D+06,  0.67518376608988D+05,  0.40920228247872D+05,  0.24800138332043D+05,&
                    0.15030386867905D+05,  0.91093253744880D+04,  0.55208032572654D+04,  0.33459413680397D+04,&
                    0.20278432533574D+04,  0.12289959111257D+04,  0.74484600674283D+03,  0.45142182226838D+03,&
                    0.27358898319296D+03,  0.16581150496543D+03,  0.10049182119117D+03,  0.60904134055255D+02,&
                    0.36911596397124D+02,  0.22370664483105D+02,  0.13557978474609D+02,  0.82169566512784D+01,&
                    0.49799737280475D+01,  0.30181658957864D+01,  0.18291914519917D+01,  0.11086008799950D+01,&
                    0.67187932120909D+00,  0.40719958861157D+00/)
    coe(1:nfun)=(/  0.85109702604437D+06, -0.13328861568794D+07,  0.12955130082769D+07, -0.94088126135302D+06,&
                    0.63366516190004D+06, -0.36659724950910D+06,  0.23153931299746D+06, -0.11531205337036D+06,&
                    0.81703240539730D+05, -0.29173330837399D+05,  0.32980298763782D+05, -0.14164756093472D+04,&
                    0.17975788000219D+05,  0.72618735110350D+04,  0.12744577536128D+05,  0.82254624311011D+04,&
                    0.77922511797245D+04,  0.36503784820712D+04,  0.12309125636370D+04, -0.56225912952406D+03,&
                   -0.30454277123674D+02,  0.88935950704364D+03,  0.99184247914444D+03,  0.37189938371957D+02,&
                   -0.22263966098016D+03,  0.69131367862856D+02,  0.14770549242923D+03,  0.14423551865452D+02,&
                   -0.10768498395041D+02,  0.40069673328171D+01,  0.29116245981132D+01,  0.53258045651506D+00,&
                    0.33720213141066D-01,  0.10436100660867D-02/)
  case(047)
    nfun = 34
    alf(1:nfun)=(/  0.61203673492837D+07,  0.37093135450204D+07,  0.22480688151639D+07,  0.13624659485842D+07,&
                    0.82573693853586D+06,  0.50044662941568D+06,  0.30330098752465D+06,  0.18381878031797D+06,&
                    0.11140532140483D+06,  0.67518376608988D+05,  0.40920228247872D+05,  0.24800138332043D+05,&
                    0.15030386867905D+05,  0.91093253744880D+04,  0.55208032572654D+04,  0.33459413680397D+04,&
                    0.20278432533574D+04,  0.12289959111257D+04,  0.74484600674283D+03,  0.45142182226838D+03,&
                    0.27358898319296D+03,  0.16581150496543D+03,  0.10049182119117D+03,  0.60904134055255D+02,&
                    0.36911596397124D+02,  0.22370664483105D+02,  0.13557978474609D+02,  0.82169566512784D+01,&
                    0.49799737280475D+01,  0.30181658957864D+01,  0.18291914519917D+01,  0.11086008799950D+01,&
                    0.67187932120909D+00,  0.40719958861157D+00/)
    coe(1:nfun)=(/  0.27808739611626D+06, -0.50030139870358D+06,  0.54425404320812D+06, -0.39812855852509D+06,&
                    0.28773971115398D+06, -0.15660207007337D+06,  0.11168632007825D+06, -0.45283945228159D+05,&
                    0.44173135064244D+05, -0.64777814322710D+04,  0.22076973620310D+05,  0.62718403555676D+04,&
                    0.15303459243178D+05,  0.10202786254454D+05,  0.12385683052696D+05,  0.93798012197092D+04,&
                    0.76768728665141D+04,  0.37937752227671D+04,  0.95185011316603D+03, -0.57361858869740D+03,&
                    0.33609606941231D+02,  0.10623112325198D+04,  0.98016833969796D+03, -0.39047269868643D+02,&
                   -0.22658267443379D+03,  0.11027877138027D+03,  0.14980390109309D+03,  0.28557166138498D+01,&
                   -0.94120262059157D+01,  0.52803998108093D+01,  0.27467267919300D+01,  0.40490873184675D+00,&
                    0.18424103584614D-01,  0.49251890186142D-03/)
  case(048)
    nfun = 35
    alf(1:nfun)=(/  0.10098606126318D+08,  0.61203673492837D+07,  0.37093135450204D+07,  0.22480688151639D+07,&
                    0.13624659485842D+07,  0.82573693853586D+06,  0.50044662941568D+06,  0.30330098752465D+06,&
                    0.18381878031797D+06,  0.11140532140483D+06,  0.67518376608988D+05,  0.40920228247872D+05,&
                    0.24800138332043D+05,  0.15030386867905D+05,  0.91093253744880D+04,  0.55208032572654D+04,&
                    0.33459413680397D+04,  0.20278432533574D+04,  0.12289959111257D+04,  0.74484600674283D+03,&
                    0.45142182226838D+03,  0.27358898319296D+03,  0.16581150496543D+03,  0.10049182119117D+03,&
                    0.60904134055255D+02,  0.36911596397124D+02,  0.22370664483105D+02,  0.13557978474609D+02,&
                    0.82169566512784D+01,  0.49799737280475D+01,  0.30181658957864D+01,  0.18291914519917D+01,&
                    0.11086008799950D+01,  0.67187932120909D+00,  0.40719958861157D+00/)
    coe(1:nfun)=(/  0.20723767589092D+06, -0.50073387179184D+06,  0.60146030729580D+06, -0.45946961466980D+06,&
                    0.33970903871679D+06, -0.18770076331949D+06,  0.13359768172288D+06, -0.55155184870362D+05,&
                    0.51935597464204D+05, -0.85954147986472D+04,  0.24942497449636D+05,  0.64197771558017D+04,&
                    0.16740153657757D+05,  0.11181111142352D+05,  0.14032600543674D+05,  0.11581182217801D+05,&
                    0.10795281651820D+05,  0.73956972931023D+04,  0.39925284468404D+04,  0.61979473757138D+03,&
                   -0.54654640607216D+03,  0.10419321578657D+03,  0.12446713806033D+04,  0.94517876151267D+03,&
                   -0.11425746964861D+03, -0.22283953312618D+03,  0.15467677994641D+03,  0.14611436954102D+03,&
                   -0.79840574140817D+01, -0.71473691893530D+01,  0.63527482213113D+01,  0.25071106992422D+01,&
                    0.28659159927699D+00,  0.91712125247909D-02,  0.25972547560826D-03/)
  case(049)
    nfun = 33
    alf(1:nfun)=(/  0.37093135450204D+07,  0.22480688151639D+07,  0.13624659485842D+07,  0.82573693853586D+06,&
                    0.50044662941568D+06,  0.30330098752465D+06,  0.18381878031797D+06,  0.11140532140483D+06,&
                    0.67518376608988D+05,  0.40920228247872D+05,  0.24800138332043D+05,  0.15030386867905D+05,&
                    0.91093253744880D+04,  0.55208032572654D+04,  0.33459413680397D+04,  0.20278432533574D+04,&
                    0.12289959111257D+04,  0.74484600674283D+03,  0.45142182226838D+03,  0.27358898319296D+03,&
                    0.16581150496543D+03,  0.10049182119117D+03,  0.60904134055255D+02,  0.36911596397124D+02,&
                    0.22370664483105D+02,  0.13557978474609D+02,  0.82169566512784D+01,  0.49799737280475D+01,&
                    0.30181658957864D+01,  0.18291914519917D+01,  0.11086008799950D+01,  0.67187932120909D+00,&
                    0.40719958861157D+00/)
    coe(1:nfun)=(/  0.46737790673828D+06, -0.75657556835938D+06,  0.77757056445312D+06, -0.55958898730469D+06,&
                    0.39499356835938D+06, -0.21667210571289D+06,  0.15021514672852D+06, -0.63116319824219D+05,&
                    0.57982104431152D+05, -0.10194164703369D+05,  0.27958154571533D+05,  0.67020888824463D+04,&
                    0.18164451549530D+05,  0.10645818866730D+05,  0.12297163972855D+05,  0.70413363785744D+04,&
                    0.41931468486786D+04,  0.26724159932137D+03, -0.49618758428097D+03,  0.19155874738097D+03,&
                    0.14275862115622D+04,  0.88952290199697D+03, -0.18825263899192D+03, -0.20964269383252D+03,&
                    0.20039486679249D+03,  0.13663777318085D+03, -0.17422047591535D+02, -0.41784158352530D+01,&
                    0.71738918084156D+01,  0.22084398991428D+01,  0.18593244530712D+00,  0.41442089141128D-02,&
                    0.10089310382000D-03/)
  case(050)
    nfun = 34
    alf(1:nfun)=(/  0.37093135450204D+07,  0.22480688151639D+07,  0.13624659485842D+07,  0.82573693853586D+06,&
                    0.50044662941568D+06,  0.30330098752465D+06,  0.18381878031797D+06,  0.11140532140483D+06,&
                    0.67518376608988D+05,  0.40920228247872D+05,  0.24800138332043D+05,  0.15030386867905D+05,&
                    0.91093253744880D+04,  0.55208032572654D+04,  0.33459413680397D+04,  0.20278432533574D+04,&
                    0.12289959111257D+04,  0.74484600674283D+03,  0.45142182226838D+03,  0.27358898319296D+03,&
                    0.16581150496543D+03,  0.10049182119117D+03,  0.60904134055255D+02,  0.36911596397124D+02,&
                    0.22370664483105D+02,  0.13557978474609D+02,  0.82169566512784D+01,  0.49799737280475D+01,&
                    0.30181658957864D+01,  0.18291914519917D+01,  0.11086008799950D+01,  0.67187932120909D+00,&
                    0.40719958861157D+00,  0.24678762946156D+00/)
    coe(1:nfun)=(/  0.73453497900903D+06, -0.12427433110824D+07,  0.12698178782244D+07, -0.93301907185614D+06,&
                    0.64533849174190D+06, -0.36732213868955D+06,  0.24103263337827D+06, -0.11270696169488D+06,&
                    0.88200593419649D+05, -0.24973941654649D+05,  0.38227849900644D+05,  0.28915483794585D+04,&
                    0.21989091161217D+05,  0.99354817128326D+04,  0.13676591076292D+05,  0.67279525701463D+04,&
                    0.43319183171050D+04, -0.68685069468687D+02, -0.43976988362319D+03,  0.30642260657174D+03,&
                    0.16016455400414D+04,  0.81667611259704D+03, -0.26104368439541D+03, -0.18530530403158D+03,&
                    0.24519450083396D+03,  0.12186582371745D+03, -0.25035090211602D+02, -0.68891104375564D+00,&
                    0.77256234663126D+01,  0.18582798357590D+01,  0.10725410903373D+00,  0.26889627415265D-02,&
                   -0.99577704543385D-04,  0.13652534295527D-04/)
  case(051)
    nfun = 32
    alf(1:nfun)=(/  0.37093135450204D+07,  0.22480688151639D+07,  0.13624659485842D+07,  0.82573693853586D+06,&
                    0.50044662941568D+06,  0.30330098752465D+06,  0.18381878031797D+06,  0.11140532140483D+06,&
                    0.67518376608988D+05,  0.40920228247872D+05,  0.24800138332043D+05,  0.15030386867905D+05,&
                    0.91093253744880D+04,  0.55208032572654D+04,  0.33459413680397D+04,  0.20278432533574D+04,&
                    0.12289959111257D+04,  0.74484600674283D+03,  0.45142182226838D+03,  0.27358898319296D+03,&
                    0.16581150496543D+03,  0.10049182119117D+03,  0.60904134055255D+02,  0.36911596397124D+02,&
                    0.22370664483105D+02,  0.13557978474609D+02,  0.82169566512784D+01,  0.49799737280475D+01,&
                    0.30181658957864D+01,  0.18291914519917D+01,  0.11086008799950D+01,  0.67187932120909D+00/)
    coe(1:nfun)=(/  0.90050444987774D+06, -0.16042680882754D+07,  0.16595352906055D+07, -0.12372038044400D+07,&
                    0.85321264137936D+06, -0.49284708515453D+06,  0.31767647877097D+06, -0.15410168789756D+06,&
                    0.11400796005934D+06, -0.37140191525847D+05,  0.47184085492373D+05, -0.77093468405306D+02,&
                    0.25426818268538D+05,  0.94858307783641D+04,  0.14911770322522D+05,  0.64644846915295D+04,&
                    0.44024682508078D+04, -0.38133840012969D+03, -0.37782342224184D+03,  0.44979566624365D+03,&
                    0.17625846987613D+04,  0.72755566470826D+03, -0.33093057916058D+03, -0.14924360524134D+03,&
                    0.28713474703873D+03,  0.10234350625454D+03, -0.30378633379197D+02,  0.30786400348184D+01,&
                    0.79941432966882D+01,  0.14722546786141D+01,  0.58139028281801D-01,  0.13607922061674D-02/)
  case(052)
    nfun = 32
    alf(1:nfun)=(/  0.37093135450204D+07,  0.22480688151639D+07,  0.13624659485842D+07,  0.82573693853586D+06,&
                    0.50044662941568D+06,  0.30330098752465D+06,  0.18381878031797D+06,  0.11140532140483D+06,&
                    0.67518376608988D+05,  0.40920228247872D+05,  0.24800138332043D+05,  0.15030386867905D+05,&
                    0.91093253744880D+04,  0.55208032572654D+04,  0.33459413680397D+04,  0.20278432533574D+04,&
                    0.12289959111257D+04,  0.74484600674283D+03,  0.45142182226838D+03,  0.27358898319296D+03,&
                    0.16581150496543D+03,  0.10049182119117D+03,  0.60904134055255D+02,  0.36911596397124D+02,&
                    0.22370664483105D+02,  0.13557978474609D+02,  0.82169566512784D+01,  0.49799737280475D+01,&
                    0.30181658957864D+01,  0.18291914519917D+01,  0.11086008799950D+01,  0.67187932120909D+00/)
    coe(1:nfun)=(/  0.95991930753756D+06, -0.17106774377370D+07,  0.17733465215569D+07, -0.13211654404602D+07,&
                    0.91271299571919D+06, -0.52616506906652D+06,  0.34020880550730D+06, -0.16422626531476D+06,&
                    0.12233428370944D+06, -0.39339734388664D+05,  0.50722433960199D+05,  0.84310034714639D+02,&
                    0.27208361892421D+05,  0.99956648127921D+04,  0.15613554525295D+05,  0.64617435843665D+04,&
                    0.42882283886019D+04, -0.60315622495092D+03, -0.34475295009965D+03,  0.64128302891925D+03,&
                    0.18954353872049D+04,  0.62948987214564D+03, -0.39949052624980D+03, -0.99403022367558D+02,&
                    0.32327616975655D+03,  0.79421856046356D+02, -0.33421317844070D+02,  0.69998322144475D+01,&
                    0.79225183577022D+01,  0.11012294941659D+01,  0.29305626599239D-01,  0.89838059099812D-03/)
  case(053)
    nfun = 32
    alf(1:nfun)=(/  0.37093135450204D+07,  0.22480688151639D+07,  0.13624659485842D+07,  0.82573693853586D+06,&
                    0.50044662941568D+06,  0.30330098752465D+06,  0.18381878031797D+06,  0.11140532140483D+06,&
                    0.67518376608988D+05,  0.40920228247872D+05,  0.24800138332043D+05,  0.15030386867905D+05,&
                    0.91093253744880D+04,  0.55208032572654D+04,  0.33459413680397D+04,  0.20278432533574D+04,&
                    0.12289959111257D+04,  0.74484600674283D+03,  0.45142182226838D+03,  0.27358898319296D+03,&
                    0.16581150496543D+03,  0.10049182119117D+03,  0.60904134055255D+02,  0.36911596397124D+02,&
                    0.22370664483105D+02,  0.13557978474609D+02,  0.82169566512784D+01,  0.49799737280475D+01,&
                    0.30181658957864D+01,  0.18291914519917D+01,  0.11086008799950D+01,  0.67187932120909D+00/)
    coe(1:nfun)=(/  0.92564516406870D+06, -0.15646529796638D+07,  0.16066216774139D+07, -0.11781792839274D+07,&
                    0.81845772218513D+06, -0.46360472919607D+06,  0.30639296574855D+06, -0.14174006111336D+06,&
                    0.11242465144932D+06, -0.31137502063870D+05,  0.48608799815811D+05,  0.35092622741535D+04,&
                    0.27259470829021D+05,  0.11505322744854D+05,  0.15754329723420D+05,  0.67325073050936D+04,&
                    0.39820525416033D+04, -0.72739335927623D+03, -0.34078720424464D+03,  0.88064549099037D+03,&
                    0.19963535990305D+04,  0.52399216090280D+03, -0.46487598745825D+03, -0.35935382721309D+02,&
                    0.35169365122342D+03,  0.54233975308913D+02, -0.34080536691918D+02,  0.10855885210130D+02,&
                    0.75225078763012D+01,  0.77574799142366D+00,  0.15079326633412D-01,  0.54291489013458D-03/)
  case(054)
    nfun = 32
    alf(1:nfun)=(/  0.37093135450204D+07,  0.22480688151639D+07,  0.13624659485842D+07,  0.82573693853586D+06,&
                    0.50044662941568D+06,  0.30330098752465D+06,  0.18381878031797D+06,  0.11140532140483D+06,&
                    0.67518376608988D+05,  0.40920228247872D+05,  0.24800138332043D+05,  0.15030386867905D+05,&
                    0.91093253744880D+04,  0.55208032572654D+04,  0.33459413680397D+04,  0.20278432533574D+04,&
                    0.12289959111257D+04,  0.74484600674283D+03,  0.45142182226838D+03,  0.27358898319296D+03,&
                    0.16581150496543D+03,  0.10049182119117D+03,  0.60904134055255D+02,  0.36911596397124D+02,&
                    0.22370664483105D+02,  0.13557978474609D+02,  0.82169566512784D+01,  0.49799737280475D+01,&
                    0.30181658957864D+01,  0.18291914519917D+01,  0.11086008799950D+01,  0.67187932120909D+00/)
    coe(1:nfun)=(/  0.77610551109266D+06, -0.11962147882929D+07,  0.12141180792212D+07, -0.86067196099424D+06,&
                    0.60854018950105D+06, -0.32970997320497D+06,  0.23098293856323D+06, -0.95197506422579D+05,&
                    0.89149309559464D+05, -0.15265987406835D+05,  0.42374008099452D+05,  0.93472906444296D+04,&
                    0.26051522727702D+05,  0.13749785885006D+05,  0.15476333035351D+05,  0.71951629402833D+04,&
                    0.35290112572953D+04, -0.77517245531199D+03, -0.35136143434251D+03,  0.11593569604343D+04,&
                    0.20660208499922D+04,  0.41053602970058D+03, -0.52399921123888D+03,  0.39820232336133D+02,&
                    0.37117974154065D+03,  0.27817154504863D+02, -0.32357707840240D+02,  0.14413792133812D+02,&
                    0.68593533510051D+01,  0.51091760557574D+00,  0.84849813703727D-02,  0.95030849283972D-04/)
end select

return
end

!-----------------------------------------------------------------------------------------------------------------------------------
!---  EDF data library: ncore = 46, iz = 47- 58
!-----------------------------------------------------------------------------------------------------------------------------------
subroutine EDFLIB46(iz,nfun,alf,coe)
implicit real(kind=8) (a-h,o-z)
real(kind=8) :: alf(*),coe(*)

select case(iz)
  case(047)
    nfun = 37
    alf(1:nfun)=(/  0.10098606126318D+08,  0.61203673492837D+07,  0.37093135450204D+07,  0.22480688151639D+07,&
                    0.13624659485842D+07,  0.82573693853586D+06,  0.50044662941568D+06,  0.30330098752465D+06,&
                    0.18381878031797D+06,  0.11140532140483D+06,  0.67518376608988D+05,  0.40920228247872D+05,&
                    0.24800138332043D+05,  0.15030386867905D+05,  0.91093253744880D+04,  0.55208032572654D+04,&
                    0.33459413680397D+04,  0.20278432533574D+04,  0.12289959111257D+04,  0.74484600674283D+03,&
                    0.45142182226838D+03,  0.27358898319296D+03,  0.16581150496543D+03,  0.10049182119117D+03,&
                    0.60904134055255D+02,  0.36911596397124D+02,  0.22370664483105D+02,  0.13557978474609D+02,&
                    0.82169566512784D+01,  0.49799737280475D+01,  0.30181658957864D+01,  0.18291914519917D+01,&
                    0.11086008799950D+01,  0.67187932120909D+00,  0.40719958861157D+00,  0.24678762946156D+00,&
                    0.14956826027973D+00/)
    coe(1:nfun)=(/  0.30406267419481D+06,  0.50105134602833D+06, -0.82834230846024D+06,  0.81196983172989D+06,&
                   -0.56327074417591D+06,  0.38432596620870D+06, -0.21044598364210D+06,  0.14176444014072D+06,&
                   -0.61908108202875D+05,  0.53402952685863D+05, -0.11584541631505D+05,  0.24909150668047D+05,&
                    0.47033845606223D+04,  0.16172959707405D+05,  0.97210803331230D+04,  0.12652690561268D+05,&
                    0.92318313972652D+04,  0.77589951217663D+04,  0.37483237199294D+04,  0.97746430982801D+03,&
                   -0.58731024600007D+03,  0.41776410518563D+02,  0.10565774060377D+04,  0.97806628359915D+03,&
                   -0.47158702157511D+02, -0.22514833946804D+03,  0.11713486746715D+03,  0.15855080201365D+03,&
                    0.11035035178211D+01, -0.14830345935583D+02,  0.47914901926085D+01,  0.40566634048022D+01,&
                    0.12742020643147D+01,  0.32907283487835D+00,  0.75969169803351D-01,  0.10343338656675D-01,&
                    0.70976990136340D-03/)
  case(048)
    nfun = 37
    alf(1:nfun)=(/  0.10098606126318D+08,  0.61203673492837D+07,  0.37093135450204D+07,  0.22480688151639D+07,&
                    0.13624659485842D+07,  0.82573693853586D+06,  0.50044662941568D+06,  0.30330098752465D+06,&
                    0.18381878031797D+06,  0.11140532140483D+06,  0.67518376608988D+05,  0.40920228247872D+05,&
                    0.24800138332043D+05,  0.15030386867905D+05,  0.91093253744880D+04,  0.55208032572654D+04,&
                    0.33459413680397D+04,  0.20278432533574D+04,  0.12289959111257D+04,  0.74484600674283D+03,&
                    0.45142182226838D+03,  0.27358898319296D+03,  0.16581150496543D+03,  0.10049182119117D+03,&
                    0.60904134055255D+02,  0.36911596397124D+02,  0.22370664483105D+02,  0.13557978474609D+02,&
                    0.82169566512784D+01,  0.49799737280475D+01,  0.30181658957864D+01,  0.18291914519917D+01,&
                    0.11086008799950D+01,  0.67187932120909D+00,  0.40719958861157D+00,  0.24678762946156D+00,&
                    0.14956826027973D+00/)
    coe(1:nfun)=(/  0.17090792623997D+05, -0.24041817702293D+05,  0.76281860701561D+05, -0.37439419784546D+05,&
                    0.53623053943634D+05, -0.10050508671522D+05,  0.28744656727433D+05,  0.50833431558609D+04,&
                    0.17878118442357D+05,  0.10497694052309D+05,  0.14293352037206D+05,  0.12343583728120D+05,&
                    0.13450355287608D+05,  0.13006580068089D+05,  0.13020234930616D+05,  0.12142475323352D+05,&
                    0.10484157408324D+05,  0.75682414985092D+04,  0.38970470792648D+04,  0.67319814532012D+03,&
                   -0.57542339082723D+03,  0.12075161329767D+03,  0.12337204855297D+04,  0.94390075492622D+03,&
                   -0.12489833994008D+03, -0.21934159656643D+03,  0.16384066480750D+03,  0.15639996516307D+03,&
                   -0.12424621786338D+02, -0.13186247572665D+02,  0.65556820603080D+01,  0.42300158502481D+01,&
                    0.12253537460157D+01,  0.28325546799633D+00,  0.46581227081770D-01,  0.35931179773776D-02,&
                    0.94065586908165D-04/)
  case(049)
    nfun = 35
    alf(1:nfun)=(/  0.37093135450204D+07,  0.22480688151639D+07,  0.13624659485842D+07,  0.82573693853586D+06,&
                    0.50044662941568D+06,  0.30330098752465D+06,  0.18381878031797D+06,  0.11140532140483D+06,&
                    0.67518376608988D+05,  0.40920228247872D+05,  0.24800138332043D+05,  0.15030386867905D+05,&
                    0.91093253744880D+04,  0.55208032572654D+04,  0.33459413680397D+04,  0.20278432533574D+04,&
                    0.12289959111257D+04,  0.74484600674283D+03,  0.45142182226838D+03,  0.27358898319296D+03,&
                    0.16581150496543D+03,  0.10049182119117D+03,  0.60904134055255D+02,  0.36911596397124D+02,&
                    0.22370664483105D+02,  0.13557978474609D+02,  0.82169566512784D+01,  0.49799737280475D+01,&
                    0.30181658957864D+01,  0.18291914519917D+01,  0.11086008799950D+01,  0.67187932120909D+00,&
                    0.40719958861157D+00,  0.24678762946156D+00,  0.14956826027973D+00/)
    coe(1:nfun)=(/  0.37697817944026D+06, -0.58066974837160D+06,  0.59780517374849D+06, -0.42065213759267D+06,&
                    0.30253376165223D+06, -0.15986875074786D+06,  0.11686939014792D+06, -0.44022068656921D+05,&
                    0.47204505149066D+05, -0.41583574815169D+04,  0.24593340886887D+05,  0.85732563972026D+04,&
                    0.17125441859149D+05,  0.11222293826904D+05,  0.11977530654403D+05,  0.72186104487481D+04,&
                    0.40951620370450D+04,  0.32208501468884D+03, -0.52558052929793D+03,  0.20828841710706D+03,&
                    0.14157952115448D+04,  0.88611050695106D+03, -0.19979027320956D+03, -0.20444142846181D+03,&
                    0.21299785238681D+03,  0.14757634546317D+03, -0.25053084277617D+02, -0.10431166875877D+02,&
                    0.82440528300157D+01,  0.43320487330058D+01,  0.11394717522632D+01,  0.21416901991037D+00,&
                    0.22046876965653D-01,  0.82357320188198D-03,  0.12382076860940D-04/)
  case(050)
    nfun = 34
    alf(1:nfun)=(/  0.37093135450204D+07,  0.22480688151639D+07,  0.13624659485842D+07,  0.82573693853586D+06,&
                    0.50044662941568D+06,  0.30330098752465D+06,  0.18381878031797D+06,  0.11140532140483D+06,&
                    0.67518376608988D+05,  0.40920228247872D+05,  0.24800138332043D+05,  0.15030386867905D+05,&
                    0.91093253744880D+04,  0.55208032572654D+04,  0.33459413680397D+04,  0.20278432533574D+04,&
                    0.12289959111257D+04,  0.74484600674283D+03,  0.45142182226838D+03,  0.27358898319296D+03,&
                    0.16581150496543D+03,  0.10049182119117D+03,  0.60904134055255D+02,  0.36911596397124D+02,&
                    0.22370664483105D+02,  0.13557978474609D+02,  0.82169566512784D+01,  0.49799737280475D+01,&
                    0.30181658957864D+01,  0.18291914519917D+01,  0.11086008799950D+01,  0.67187932120909D+00,&
                    0.40719958861157D+00,  0.24678762946156D+00/)
    coe(1:nfun)=(/  0.71158599854112D+06, -0.11575765638728D+07,  0.11692553833346D+07, -0.84959739576483D+06,&
                    0.58801420003891D+06, -0.33144111706793D+06,  0.21977426690215D+06, -0.10046456513953D+06,&
                    0.81271102807492D+05, -0.21086160256416D+05,  0.36058663107559D+05,  0.40985118164942D+04,&
                    0.21318777732857D+05,  0.10307388150090D+05,  0.13470501308633D+05,  0.68421910332935D+04,&
                    0.42690830852282D+04, -0.33310664050747D+02, -0.45801676340261D+03,  0.31674319101556D+03,&
                    0.15924458332704D+04,  0.80870214662529D+03, -0.27196097836710D+03, -0.17857533109232D+03,&
                    0.26225233665440D+03,  0.13218755384298D+03, -0.36090870843335D+02, -0.66542592387357D+01,&
                    0.97971346351437D+01,  0.43328091531263D+01,  0.99037482511955D+00,  0.14051961911095D+00,&
                    0.87399830025419D-02,  0.22450661267692D-03/)
  case(051)
    nfun = 34
    alf(1:nfun)=(/  0.37093135450204D+07,  0.22480688151639D+07,  0.13624659485842D+07,  0.82573693853586D+06,&
                    0.50044662941568D+06,  0.30330098752465D+06,  0.18381878031797D+06,  0.11140532140483D+06,&
                    0.67518376608988D+05,  0.40920228247872D+05,  0.24800138332043D+05,  0.15030386867905D+05,&
                    0.91093253744880D+04,  0.55208032572654D+04,  0.33459413680397D+04,  0.20278432533574D+04,&
                    0.12289959111257D+04,  0.74484600674283D+03,  0.45142182226838D+03,  0.27358898319296D+03,&
                    0.16581150496543D+03,  0.10049182119117D+03,  0.60904134055255D+02,  0.36911596397124D+02,&
                    0.22370664483105D+02,  0.13557978474609D+02,  0.82169566512784D+01,  0.49799737280475D+01,&
                    0.30181658957864D+01,  0.18291914519917D+01,  0.11086008799950D+01,  0.67187932120909D+00,&
                    0.40719958861157D+00,  0.24678762946156D+00/)
    coe(1:nfun)=(/  0.96422525245118D+06, -0.16414266897655D+07,  0.16685145683131D+07, -0.12319281151097D+07,&
                    0.84582487083173D+06, -0.48688644633138D+06,  0.31375987547934D+06, -0.15171001646405D+06,&
                    0.11261660791826D+06, -0.36346013584048D+05,  0.46737506027669D+05,  0.17261425090209D+03,&
                    0.25287989434807D+05,  0.95627847290449D+04,  0.14869411379485D+05,  0.64878146574022D+04,&
                    0.43902980148445D+04, -0.37405236596428D+03, -0.38009128569224D+03,  0.45090622207362D+03,&
                    0.17572403849131D+04,  0.71380460205125D+03, -0.34029127868122D+03, -0.14069420691359D+03,&
                    0.30936294105049D+03,  0.11058973707579D+03, -0.44823362545722D+02, -0.20857788265382D+01,&
                    0.11165979492285D+02,  0.41916077604732D+01,  0.79565244879404D+00,  0.82399427888575D-01,&
                    0.32083936717733D-02,  0.75798289391305D-04/)
  case(052)
    nfun = 34
    alf(1:nfun)=(/  0.37093135450204D+07,  0.22480688151639D+07,  0.13624659485842D+07,  0.82573693853586D+06,&
                    0.50044662941568D+06,  0.30330098752465D+06,  0.18381878031797D+06,  0.11140532140483D+06,&
                    0.67518376608988D+05,  0.40920228247872D+05,  0.24800138332043D+05,  0.15030386867905D+05,&
                    0.91093253744880D+04,  0.55208032572654D+04,  0.33459413680397D+04,  0.20278432533574D+04,&
                    0.12289959111257D+04,  0.74484600674283D+03,  0.45142182226838D+03,  0.27358898319296D+03,&
                    0.16581150496543D+03,  0.10049182119117D+03,  0.60904134055255D+02,  0.36911596397124D+02,&
                    0.22370664483105D+02,  0.13557978474609D+02,  0.82169566512784D+01,  0.49799737280475D+01,&
                    0.30181658957864D+01,  0.18291914519917D+01,  0.11086008799950D+01,  0.67187932120909D+00,&
                    0.40719958861157D+00,  0.24678762946156D+00/)
    coe(1:nfun)=(/  0.11413356309443D+07, -0.19534449109325D+07,  0.19846288806791D+07, -0.14689522022839D+07,&
                    0.10061402366076D+07, -0.58175764938760D+06,  0.37231390320253D+06, -0.18242111318812D+06,&
                    0.13255181773676D+06, -0.45043018194616D+05,  0.53897134216182D+05, -0.16795431943648D+04,&
                    0.28187716349546D+05,  0.94520590996724D+04,  0.15915539706878D+05,  0.62940163057584D+04,&
                    0.43822607412405D+04, -0.65470917479787D+03, -0.31400230722700D+03,  0.62365155448223D+03,&
                    0.18988216236921D+04,  0.60681275731813D+03, -0.40522505053036D+03, -0.89176949351848D+02,&
                    0.35147186125677D+03,  0.83786555314086D+02, -0.50808826238815D+02,  0.30616642255384D+01,&
                    0.12282794946753D+02,  0.38859261745391D+01,  0.59207530364205D+00,  0.44960580612845D-01,&
                    0.11240640181462D-02,  0.35886369020854D-04/)
  case(053)
    nfun = 34
    alf(1:nfun)=(/  0.37093135450204D+07,  0.22480688151639D+07,  0.13624659485842D+07,  0.82573693853586D+06,&
                    0.50044662941568D+06,  0.30330098752465D+06,  0.18381878031797D+06,  0.11140532140483D+06,&
                    0.67518376608988D+05,  0.40920228247872D+05,  0.24800138332043D+05,  0.15030386867905D+05,&
                    0.91093253744880D+04,  0.55208032572654D+04,  0.33459413680397D+04,  0.20278432533574D+04,&
                    0.12289959111257D+04,  0.74484600674283D+03,  0.45142182226838D+03,  0.27358898319296D+03,&
                    0.16581150496543D+03,  0.10049182119117D+03,  0.60904134055255D+02,  0.36911596397124D+02,&
                    0.22370664483105D+02,  0.13557978474609D+02,  0.82169566512784D+01,  0.49799737280475D+01,&
                    0.30181658957864D+01,  0.18291914519917D+01,  0.11086008799950D+01,  0.67187932120909D+00,&
                    0.40719958861157D+00,  0.24678762946156D+00/)
    coe(1:nfun)=(/  0.11494463117437D+07, -0.19567358249698D+07,  0.19928142864432D+07, -0.14705535896211D+07,&
                    0.10110896707546D+07, -0.58123920450693D+06,  0.37524065664929D+06, -0.18108916514239D+06,&
                    0.13461465706907D+06, -0.43557533013061D+05,  0.55531117823537D+05, -0.33993191242218D+03,&
                    0.29397197189286D+05,  0.10318762113662D+05,  0.16413053113572D+05,  0.63669114855251D+04,&
                    0.41860147742351D+04, -0.83974899838259D+03, -0.27600637313037D+03,  0.84366344501945D+03,&
                    0.20082642502061D+04,  0.49216046883620D+03, -0.46648286273073D+03, -0.23084302027626D+02,&
                    0.38578338334034D+03,  0.53161142223500D+02, -0.53841132328069D+02,  0.86056968808919D+01,&
                    0.13050549067129D+02,  0.34350562242101D+01,  0.41213026166878D+00,  0.23786605255573D-01,&
                    0.33338431048691D-03,  0.25645336234614D-04/)
  case(054)
    nfun = 38
    alf(1:nfun)=(/  0.74850931724558D+08,  0.27493455178901D+08,  0.10098606126318D+08,  0.61203673492837D+07,&
                    0.37093135450204D+07,  0.22480688151639D+07,  0.13624659485842D+07,  0.82573693853586D+06,&
                    0.50044662941568D+06,  0.30330098752465D+06,  0.18381878031797D+06,  0.11140532140483D+06,&
                    0.67518376608988D+05,  0.40920228247872D+05,  0.24800138332043D+05,  0.15030386867905D+05,&
                    0.91093253744880D+04,  0.55208032572654D+04,  0.33459413680397D+04,  0.20278432533574D+04,&
                    0.12289959111257D+04,  0.74484600674283D+03,  0.45142182226838D+03,  0.27358898319296D+03,&
                    0.16581150496543D+03,  0.10049182119117D+03,  0.60904134055255D+02,  0.36911596397124D+02,&
                    0.22370664483105D+02,  0.13557978474609D+02,  0.82169566512784D+01,  0.49799737280475D+01,&
                    0.30181658957864D+01,  0.18291914519917D+01,  0.11086008799950D+01,  0.67187932120909D+00,&
                    0.40719958861157D+00,  0.24678762946156D+00/)
    coe(1:nfun)=(/  0.51416012966709D+07, -0.47090157945156D+07,  0.58705991863766D+07, -0.75929197264366D+07,&
                    0.65044164818163D+07, -0.44858467546544D+07,  0.28756465511847D+07, -0.16675301728678D+07,&
                    0.10044670739232D+07, -0.52974161147332D+06,  0.33512423793334D+06, -0.15068736867216D+06,&
                    0.11918559233947D+06, -0.31684831637897D+05,  0.51402529919460D+05,  0.43653203721978D+04,&
                    0.28806347649094D+05,  0.12224535258596D+05,  0.16321785349603D+05,  0.67264431509071D+04,&
                    0.37901775964878D+04, -0.91891602660343D+03, -0.26900117608608D+03,  0.11121179974689D+04,&
                    0.20807544899215D+04,  0.37246853009765D+03, -0.52264219055136D+03,  0.57375249768073D+02,&
                    0.41011061315933D+03,  0.20147609411568D+02, -0.53855561901130D+02,  0.14346921831894D+02,&
                    0.13395225037585D+02,  0.28880663525450D+01,  0.27191439807928D+00,  0.12347647424178D-01,&
                    0.56083284139419D-04,  0.18941926318128D-04/)
  case(055)
    nfun = 36
    alf(1:nfun)=(/  0.10098606126318D+08,  0.61203673492837D+07,  0.37093135450204D+07,  0.22480688151639D+07,&
                    0.13624659485842D+07,  0.82573693853586D+06,  0.50044662941568D+06,  0.30330098752465D+06,&
                    0.18381878031797D+06,  0.11140532140483D+06,  0.67518376608988D+05,  0.40920228247872D+05,&
                    0.24800138332043D+05,  0.15030386867905D+05,  0.91093253744880D+04,  0.55208032572654D+04,&
                    0.33459413680397D+04,  0.20278432533574D+04,  0.12289959111257D+04,  0.74484600674283D+03,&
                    0.45142182226838D+03,  0.27358898319296D+03,  0.16581150496543D+03,  0.10049182119117D+03,&
                    0.60904134055255D+02,  0.36911596397124D+02,  0.22370664483105D+02,  0.13557978474609D+02,&
                    0.82169566512784D+01,  0.49799737280475D+01,  0.30181658957864D+01,  0.18291914519917D+01,&
                    0.11086008799950D+01,  0.67187932120909D+00,  0.40719958861157D+00,  0.24678762946156D+00/)
    coe(1:nfun)=(/  0.12572353936629D+07, -0.23438529918804D+07,  0.24901152087755D+07, -0.18694542528648D+07,&
                    0.13031146816180D+07, -0.75056286774182D+06,  0.48773095986724D+06, -0.23555892727900D+06,&
                    0.17478089484036D+06, -0.58090983944654D+05,  0.71006247739062D+05, -0.19363480307534D+04,&
                    0.37672589160878D+05,  0.14467391136784D+05,  0.25323106732508D+05,  0.15759809700493D+05,&
                    0.15314657865641D+05,  0.75495308441448D+04,  0.30982022922324D+04, -0.83550057342561D+03,&
                   -0.32145108898025D+03,  0.14435529801089D+04,  0.21036156481524D+04,  0.25526193735592D+03,&
                   -0.57481988372403D+03,  0.15329099557351D+03,  0.42121259439866D+03, -0.12886243411257D+02,&
                   -0.51331925217017D+02,  0.20299133713111D+02,  0.13128769361316D+02,  0.23689094289980D+01,&
                    0.15948420304859D+00,  0.77771916317388D-02, -0.34782094080654D-03,  0.32961199981862D-04/)
  case(056)
    nfun = 36
    alf(1:nfun)=(/  0.10098606126318D+08,  0.61203673492837D+07,  0.37093135450204D+07,  0.22480688151639D+07,&
                    0.13624659485842D+07,  0.82573693853586D+06,  0.50044662941568D+06,  0.30330098752465D+06,&
                    0.18381878031797D+06,  0.11140532140483D+06,  0.67518376608988D+05,  0.40920228247872D+05,&
                    0.24800138332043D+05,  0.15030386867905D+05,  0.91093253744880D+04,  0.55208032572654D+04,&
                    0.33459413680397D+04,  0.20278432533574D+04,  0.12289959111257D+04,  0.74484600674283D+03,&
                    0.45142182226838D+03,  0.27358898319296D+03,  0.16581150496543D+03,  0.10049182119117D+03,&
                    0.60904134055255D+02,  0.36911596397124D+02,  0.22370664483105D+02,  0.13557978474609D+02,&
                    0.82169566512784D+01,  0.49799737280475D+01,  0.30181658957864D+01,  0.18291914519917D+01,&
                    0.11086008799950D+01,  0.67187932120909D+00,  0.40719958861157D+00,  0.24678762946156D+00/)
    coe(1:nfun)=(/  0.49664702849627D+06, -0.92383507060766D+06,  0.10673176076841D+07, -0.77087524039078D+06,&
                    0.58136011107588D+06, -0.30331845302248D+06,  0.23047827929354D+06, -0.83786302664936D+05,&
                    0.93071973518312D+05, -0.85396669808179D+04,  0.46857694911815D+05,  0.14698490720078D+05,&
                    0.31377651508942D+05,  0.20580868596457D+05,  0.24123013109528D+05,  0.18063754167289D+05,&
                    0.14967905269556D+05,  0.79615209129180D+04,  0.25952215264598D+04, -0.85461612215958D+03,&
                   -0.28304436344426D+03,  0.17523039621351D+04,  0.21210545193355D+04,  0.11625165078267D+03,&
                   -0.60588813108147D+03,  0.25437588466704D+03,  0.42273356563374D+03, -0.47233257687303D+02,&
                   -0.44888160125796D+02,  0.25446847396310D+02,  0.12597761966561D+02,  0.18134163847207D+01,&
                    0.95186539772556D-01,  0.35154206357504D-02, -0.28827753521554D-03,  0.28707940849193D-04/)
  case(057)
    nfun = 34
    alf(1:nfun)=(/  0.61203673492837D+07,  0.37093135450204D+07,  0.22480688151639D+07,  0.13624659485842D+07,&
                    0.82573693853586D+06,  0.50044662941568D+06,  0.30330098752465D+06,  0.18381878031797D+06,&
                    0.11140532140483D+06,  0.67518376608988D+05,  0.40920228247872D+05,  0.24800138332043D+05,&
                    0.15030386867905D+05,  0.91093253744880D+04,  0.55208032572654D+04,  0.33459413680397D+04,&
                    0.20278432533574D+04,  0.12289959111257D+04,  0.74484600674283D+03,  0.45142182226838D+03,&
                    0.27358898319296D+03,  0.16581150496543D+03,  0.10049182119117D+03,  0.60904134055255D+02,&
                    0.36911596397124D+02,  0.22370664483105D+02,  0.13557978474609D+02,  0.82169566512784D+01,&
                    0.49799737280475D+01,  0.30181658957864D+01,  0.18291914519917D+01,  0.11086008799950D+01,&
                    0.67187932120909D+00,  0.40719958861157D+00/)
    coe(1:nfun)=(/  0.79543962525797D+06, -0.12167748246374D+07,  0.12757030434999D+07, -0.89005428001451D+06,&
                    0.64932643667698D+06, -0.33841565986109D+06,  0.25176628892732D+06, -0.93623494802833D+05,&
                    0.10061861321032D+06, -0.10459490633458D+05,  0.50063630059317D+05,  0.14824858428806D+05,&
                    0.32527206473418D+05,  0.19760782182490D+05,  0.22149019144532D+05,  0.13608828083458D+05,&
                    0.88845937016155D+04,  0.17742986257130D+04, -0.69375568673643D+03, -0.30612122800341D+03,&
                    0.21200247968407D+04,  0.20835209414813D+04, -0.15751378708999D+02, -0.62832259246393D+03,&
                    0.36656687902328D+03,  0.40915259289953D+03, -0.78784742508930D+02, -0.36082246315463D+02,&
                    0.30057381290832D+02,  0.11696500539109D+02,  0.13075166178218D+01,  0.57316030143397D-01,&
                    0.36160488488690D-03,  0.41283678916315D-04/)
  case(058)
    nfun = 34
    alf(1:nfun)=(/  0.61203673492837D+07,  0.37093135450204D+07,  0.22480688151639D+07,  0.13624659485842D+07,&
                    0.82573693853586D+06,  0.50044662941568D+06,  0.30330098752465D+06,  0.18381878031797D+06,&
                    0.11140532140483D+06,  0.67518376608988D+05,  0.40920228247872D+05,  0.24800138332043D+05,&
                    0.15030386867905D+05,  0.91093253744880D+04,  0.55208032572654D+04,  0.33459413680397D+04,&
                    0.20278432533574D+04,  0.12289959111257D+04,  0.74484600674283D+03,  0.45142182226838D+03,&
                    0.27358898319296D+03,  0.16581150496543D+03,  0.10049182119117D+03,  0.60904134055255D+02,&
                    0.36911596397124D+02,  0.22370664483105D+02,  0.13557978474609D+02,  0.82169566512784D+01,&
                    0.49799737280475D+01,  0.30181658957864D+01,  0.18291914519917D+01,  0.11086008799950D+01,&
                    0.67187932120909D+00,  0.40719958861157D+00/)
    coe(1:nfun)=(/  0.15730382851920D+07, -0.25977333537083D+07,  0.26588353613324D+07, -0.19353197946572D+07,&
                    0.13466157035515D+07, -0.75907687479901D+06,  0.50348553806257D+06, -0.23282235453767D+06,&
                    0.18351723736665D+06, -0.52877845071465D+05,  0.77359511118770D+05,  0.29381058146805D+04,&
                    0.42006033322975D+05,  0.16850402759239D+05,  0.25445152670696D+05,  0.12660918766607D+05,&
                    0.95276940360926D+04,  0.10740107421852D+04, -0.58956571735453D+03, -0.25829828645260D+03,&
                    0.24698135077963D+04,  0.20294248796901D+04, -0.15971608230651D+03, -0.62610106055934D+03,&
                    0.47875478807955D+03,  0.38050200334737D+03, -0.10440271408446D+03, -0.24480264950512D+02,&
                    0.32602242465431D+02,  0.10492805090857D+02,  0.10301268009029D+01,  0.40427555099910D-01,&
                    0.12211908409654D-03,  0.37205766299842D-04/)
end select

return
end

!-----------------------------------------------------------------------------------------------------------------------------------
!---  EDF data library: ncore = 54, iz = 55- 71
!-----------------------------------------------------------------------------------------------------------------------------------
subroutine EDFLIB54(iz,nfun,alf,coe)
implicit real(kind=8) (a-h,o-z)
real(kind=8) :: alf(*),coe(*)

select case(iz)
  case(055)
    nfun = 36
    alf(1:nfun)=(/  0.37093135450204D+07,  0.22480688151639D+07,  0.13624659485842D+07,  0.82573693853586D+06,&
                    0.50044662941568D+06,  0.30330098752465D+06,  0.18381878031797D+06,  0.11140532140483D+06,&
                    0.67518376608988D+05,  0.40920228247872D+05,  0.24800138332043D+05,  0.15030386867905D+05,&
                    0.91093253744880D+04,  0.55208032572654D+04,  0.33459413680397D+04,  0.20278432533574D+04,&
                    0.12289959111257D+04,  0.74484600674283D+03,  0.45142182226838D+03,  0.27358898319296D+03,&
                    0.16581150496543D+03,  0.10049182119117D+03,  0.60904134055255D+02,  0.36911596397124D+02,&
                    0.22370664483105D+02,  0.13557978474609D+02,  0.82169566512784D+01,  0.49799737280475D+01,&
                    0.30181658957864D+01,  0.18291914519917D+01,  0.11086008799950D+01,  0.67187932120909D+00,&
                    0.40719958861157D+00,  0.24678762946156D+00,  0.14956826027973D+00,  0.90647430472564D-01/)
    coe(1:nfun)=(/  0.68093724598444D+06, -0.10171826270530D+07,  0.10435135189707D+07, -0.72556129936695D+06,&
                    0.52416540921271D+06, -0.27420243372071D+06,  0.20256553883910D+06, -0.75546506393462D+05,&
                    0.81378180050969D+05, -0.78726924505830D+04,  0.41054877074458D+05,  0.12595720347257D+05,&
                    0.26389945519742D+05,  0.15186783559075D+05,  0.15648559966153D+05,  0.73721672450879D+04,&
                    0.31941365035200D+04, -0.89954367107002D+03, -0.29390785125858D+03,  0.14388085480925D+04,&
                    0.21318635997728D+04,  0.24432811882471D+03, -0.59199199485057D+03,  0.15822497991515D+03,&
                    0.43990942465519D+03, -0.25721140334123D+02, -0.55674225600687D+02,  0.26862452268827D+02,&
                    0.14273242825726D+02, -0.16800543531019D+00, -0.20444277627320D+00,  0.55121238670345D+00,&
                    0.24497773463501D+00,  0.34800193614403D-01,  0.15178439709764D-02,  0.24238609865668D-04/)
  case(056)
    nfun = 37
    alf(1:nfun)=(/  0.10098606126318D+08,  0.61203673492837D+07,  0.37093135450204D+07,  0.22480688151639D+07,&
                    0.13624659485842D+07,  0.82573693853586D+06,  0.50044662941568D+06,  0.30330098752465D+06,&
                    0.18381878031797D+06,  0.11140532140483D+06,  0.67518376608988D+05,  0.40920228247872D+05,&
                    0.24800138332043D+05,  0.15030386867905D+05,  0.91093253744880D+04,  0.55208032572654D+04,&
                    0.33459413680397D+04,  0.20278432533574D+04,  0.12289959111257D+04,  0.74484600674283D+03,&
                    0.45142182226838D+03,  0.27358898319296D+03,  0.16581150496543D+03,  0.10049182119117D+03,&
                    0.60904134055255D+02,  0.36911596397124D+02,  0.22370664483105D+02,  0.13557978474609D+02,&
                    0.82169566512784D+01,  0.49799737280475D+01,  0.30181658957864D+01,  0.18291914519917D+01,&
                    0.11086008799950D+01,  0.67187932120909D+00,  0.40719958861157D+00,  0.24678762946156D+00,&
                    0.14956826027973D+00/)
    coe(1:nfun)=(/  0.73579727861214D+06, -0.75583848215675D+06,  0.68427818142891D+06, -0.38570762969685D+06,&
                    0.29681692078972D+06, -0.11809635676098D+06,  0.11873951902747D+06, -0.18679304479599D+05,&
                    0.56054300179720D+05,  0.12334712468922D+05,  0.35223517469525D+05,  0.21210507952467D+05,&
                    0.27789011501595D+05,  0.22602719468474D+05,  0.23028703667201D+05,  0.18695109946594D+05,&
                    0.14635616055638D+05,  0.81543530226187D+04,  0.24836572460167D+04, -0.80605722047610D+03,&
                   -0.31846698585729D+03,  0.17880377147095D+04,  0.21324040810006D+04,  0.11199690042759D+03,&
                   -0.63364061381849D+03,  0.26902234234579D+03,  0.44103707734676D+03, -0.64340266567689D+02,&
                   -0.48301874228271D+02,  0.34431867991609D+02,  0.12534945366882D+02, -0.13998934196643D+01,&
                    0.15484660413910D+00,  0.73389056746335D+00,  0.19458276859945D+00,  0.13080798758221D-01,&
                    0.38313831125381D-03/)
  case(057)
    nfun = 36
    alf(1:nfun)=(/  0.61203673492837D+07,  0.37093135450204D+07,  0.22480688151639D+07,  0.13624659485842D+07,&
                    0.82573693853586D+06,  0.50044662941568D+06,  0.30330098752465D+06,  0.18381878031797D+06,&
                    0.11140532140483D+06,  0.67518376608988D+05,  0.40920228247872D+05,  0.24800138332043D+05,&
                    0.15030386867905D+05,  0.91093253744880D+04,  0.55208032572654D+04,  0.33459413680397D+04,&
                    0.20278432533574D+04,  0.12289959111257D+04,  0.74484600674283D+03,  0.45142182226838D+03,&
                    0.27358898319296D+03,  0.16581150496543D+03,  0.10049182119117D+03,  0.60904134055255D+02,&
                    0.36911596397124D+02,  0.22370664483105D+02,  0.13557978474609D+02,  0.82169566512784D+01,&
                    0.49799737280475D+01,  0.30181658957864D+01,  0.18291914519917D+01,  0.11086008799950D+01,&
                    0.67187932120909D+00,  0.40719958861157D+00,  0.24678762946156D+00,  0.14956826027973D+00/)
    coe(1:nfun)=(/  0.14145308321724D+07, -0.23194464139814D+07,  0.23686290975885D+07, -0.17203672064033D+07,&
                    0.11973892430048D+07, -0.67340767241120D+06,  0.44797374746859D+06, -0.20575905567241D+06,&
                    0.16390317715621D+06, -0.45845791673213D+05,  0.69822640322119D+05,  0.38735711641014D+04,&
                    0.38642389860187D+05,  0.16398860364662D+05,  0.24041548910065D+05,  0.12578623192105D+05,&
                    0.94646460547526D+04,  0.14456586314922D+04, -0.52680347858928D+03, -0.40702106298628D+03,&
                    0.21978560499317D+04,  0.20754317782131D+04, -0.13268041355783D+02, -0.66543816472151D+03,&
                    0.39210932256809D+03,  0.42443892258383D+03, -0.99454405053280D+02, -0.37460374854808D+02,&
                    0.40909387690915D+02,  0.98334830787469D+01, -0.21019077339794D+01,  0.65789475699330D+00,&
                    0.74387915135098D+00,  0.14206003857178D+00,  0.75763377312104D-02,  0.19895351247357D-03/)
  case(058)
    nfun = 36
    alf(1:nfun)=(/  0.61203673492837D+07,  0.37093135450204D+07,  0.22480688151639D+07,  0.13624659485842D+07,&
                    0.82573693853586D+06,  0.50044662941568D+06,  0.30330098752465D+06,  0.18381878031797D+06,&
                    0.11140532140483D+06,  0.67518376608988D+05,  0.40920228247872D+05,  0.24800138332043D+05,&
                    0.15030386867905D+05,  0.91093253744880D+04,  0.55208032572654D+04,  0.33459413680397D+04,&
                    0.20278432533574D+04,  0.12289959111257D+04,  0.74484600674283D+03,  0.45142182226838D+03,&
                    0.27358898319296D+03,  0.16581150496543D+03,  0.10049182119117D+03,  0.60904134055255D+02,&
                    0.36911596397124D+02,  0.22370664483105D+02,  0.13557978474609D+02,  0.82169566512784D+01,&
                    0.49799737280475D+01,  0.30181658957864D+01,  0.18291914519917D+01,  0.11086008799950D+01,&
                    0.67187932120909D+00,  0.40719958861157D+00,  0.24678762946156D+00,  0.14956826027973D+00/)
    coe(1:nfun)=(/  0.22567958628640D+07, -0.37823713757248D+07,  0.38208950541945D+07, -0.28128891055589D+07,&
                    0.19241760807102D+07, -0.11114674734995D+07,  0.70970143608618D+06, -0.35060918332171D+06,&
                    0.24997543621945D+06, -0.90030085395724D+05,  0.98105127323687D+05, -0.85578939999864D+04,&
                    0.48426965984572D+05,  0.13321191414321D+05,  0.27432619584018D+05,  0.11578508990961D+05,&
                    0.10136393713934D+05,  0.72742791750445D+03, -0.41371661851637D+03, -0.36307146296575D+03,&
                    0.25545738029768D+04,  0.20186977307071D+04, -0.16066185226163D+03, -0.66397639966196D+03,&
                    0.51121555431178D+03,  0.39168117470644D+03, -0.12729184589798D+03, -0.22644874526420D+02,&
                    0.43310507596078D+02,  0.74240710472036D+01, -0.20680734845935D+01,  0.85560419537953D+00,&
                    0.71058757709798D+00,  0.12321965116943D+00,  0.60365894379526D-02,  0.15303476581319D-03/)
  case(059)
    nfun = 36
    alf(1:nfun)=(/  0.61203673492837D+07,  0.37093135450204D+07,  0.22480688151639D+07,  0.13624659485842D+07,&
                    0.82573693853586D+06,  0.50044662941568D+06,  0.30330098752465D+06,  0.18381878031797D+06,&
                    0.11140532140483D+06,  0.67518376608988D+05,  0.40920228247872D+05,  0.24800138332043D+05,&
                    0.15030386867905D+05,  0.91093253744880D+04,  0.55208032572654D+04,  0.33459413680397D+04,&
                    0.20278432533574D+04,  0.12289959111257D+04,  0.74484600674283D+03,  0.45142182226838D+03,&
                    0.27358898319296D+03,  0.16581150496543D+03,  0.10049182119117D+03,  0.60904134055255D+02,&
                    0.36911596397124D+02,  0.22370664483105D+02,  0.13557978474609D+02,  0.82169566512784D+01,&
                    0.49799737280475D+01,  0.30181658957864D+01,  0.18291914519917D+01,  0.11086008799950D+01,&
                    0.67187932120909D+00,  0.40719958861157D+00,  0.24678762946156D+00,  0.14956826027973D+00/)
    coe(1:nfun)=(/  0.28174877656784D+07, -0.48118996119556D+07,  0.48700455454192D+07, -0.36094907790799D+07,&
                    0.24598920660894D+07, -0.14335674768112D+07,  0.90435850133693D+06, -0.45685457395464D+06,&
                    0.31462292157587D+06, -0.12195477645406D+06,  0.11978934923863D+06, -0.17142126483269D+05,&
                    0.56205687416699D+05,  0.11430345806444D+05,  0.30179169185699D+05,  0.10902639874165D+05,&
                    0.10573635117105D+05,  0.10878926153551D+03, -0.34215135427483D+03, -0.25284689153353D+03,&
                    0.28900206033531D+04,  0.19419744193986D+04, -0.31579176203444D+03, -0.63325893642067D+03,&
                    0.62294725175730D+03,  0.34440269794842D+03, -0.14488813003581D+03, -0.70847420352743D+01,&
                    0.42273958317718D+02,  0.60085527456788D+01, -0.17284115659613D+01,  0.77719664659131D+00,&
                    0.68584575559308D+00,  0.12749109461468D+00,  0.63339854375144D-02,  0.17898753764523D-03/)
  case(060)
    nfun = 36
    alf(1:nfun)=(/  0.61203673492837D+07,  0.37093135450204D+07,  0.22480688151639D+07,  0.13624659485842D+07,&
                    0.82573693853586D+06,  0.50044662941568D+06,  0.30330098752465D+06,  0.18381878031797D+06,&
                    0.11140532140483D+06,  0.67518376608988D+05,  0.40920228247872D+05,  0.24800138332043D+05,&
                    0.15030386867905D+05,  0.91093253744880D+04,  0.55208032572654D+04,  0.33459413680397D+04,&
                    0.20278432533574D+04,  0.12289959111257D+04,  0.74484600674283D+03,  0.45142182226838D+03,&
                    0.27358898319296D+03,  0.16581150496543D+03,  0.10049182119117D+03,  0.60904134055255D+02,&
                    0.36911596397124D+02,  0.22370664483105D+02,  0.13557978474609D+02,  0.82169566512784D+01,&
                    0.49799737280475D+01,  0.30181658957864D+01,  0.18291914519917D+01,  0.11086008799950D+01,&
                    0.67187932120909D+00,  0.40719958861157D+00,  0.24678762946156D+00,  0.14956826027973D+00/)
    coe(1:nfun)=(/  0.33946720824909D+07, -0.58048161262898D+07,  0.58606323122792D+07, -0.43506420796185D+07,&
                    0.29563461866591D+07, -0.17298642306521D+07,  0.10840394337026D+07, -0.55403064970744D+06,&
                    0.37448720110720D+06, -0.15089643006581D+06,  0.14004695461108D+06, -0.24736084358543D+05,&
                    0.63566757694777D+05,  0.98472721583657D+04,  0.32767021832316D+05,  0.10278945428369D+05,&
                    0.10925775432469D+05, -0.49206185295060D+03, -0.26483435968123D+03, -0.10105883326964D+03,&
                    0.32187330568519D+04,  0.18390732476044D+04, -0.47420520862624D+03, -0.58138676097521D+03,&
                    0.73229670606089D+03,  0.28871588341265D+03, -0.15919001849045D+03,  0.82470254540717D+01,&
                    0.42111390911513D+02,  0.41458184468644D+01, -0.15778204904678D+01,  0.91018554047403D+00,&
                    0.65629427869821D+00,  0.11229290991323D+00,  0.52379438758443D-02,  0.14648001828421D-03/)
  case(061)
    nfun = 36
    alf(1:nfun)=(/  0.61203673492837D+07,  0.37093135450204D+07,  0.22480688151639D+07,  0.13624659485842D+07,&
                    0.82573693853586D+06,  0.50044662941568D+06,  0.30330098752465D+06,  0.18381878031797D+06,&
                    0.11140532140483D+06,  0.67518376608988D+05,  0.40920228247872D+05,  0.24800138332043D+05,&
                    0.15030386867905D+05,  0.91093253744880D+04,  0.55208032572654D+04,  0.33459413680397D+04,&
                    0.20278432533574D+04,  0.12289959111257D+04,  0.74484600674283D+03,  0.45142182226838D+03,&
                    0.27358898319296D+03,  0.16581150496543D+03,  0.10049182119117D+03,  0.60904134055255D+02,&
                    0.36911596397124D+02,  0.22370664483105D+02,  0.13557978474609D+02,  0.82169566512784D+01,&
                    0.49799737280475D+01,  0.30181658957864D+01,  0.18291914519917D+01,  0.11086008799950D+01,&
                    0.67187932120909D+00,  0.40719958861157D+00,  0.24678762946156D+00,  0.14956826027973D+00/)
    coe(1:nfun)=(/  0.37326119907084D+07, -0.63618585726771D+07,  0.64175506252532D+07, -0.47602506631808D+07,&
                    0.32343676610179D+07, -0.18918997336771D+07,  0.11855576283228D+07, -0.60604984577179D+06,&
                    0.40923837242031D+06, -0.16546632549140D+06,  0.15254556312962D+06, -0.27817967901982D+05,&
                    0.68554825739324D+05,  0.96538029542342D+04,  0.34589414880803D+05,  0.10038861636452D+05,&
                    0.11007463328666D+05, -0.96893369941949D+03, -0.23555115712481D+03,  0.12399090022629D+03,&
                    0.35168758839650D+04,  0.17185116710855D+04, -0.63697747470953D+03, -0.50170131016603D+03,&
                    0.83129050410038D+03,  0.22476063959724D+03, -0.16705379945961D+03,  0.23290489857327D+02,&
                    0.41064736408374D+02,  0.24407498129267D+01, -0.13703727918536D+01,  0.10291518219520D+01,&
                    0.62568875574610D+00,  0.97664172089488D-01,  0.41888695582770D-02,  0.11355552025205D-03/)
  case(062)
    nfun = 36
    alf(1:nfun)=(/  0.61203673492837D+07,  0.37093135450204D+07,  0.22480688151639D+07,  0.13624659485842D+07,&
                    0.82573693853586D+06,  0.50044662941568D+06,  0.30330098752465D+06,  0.18381878031797D+06,&
                    0.11140532140483D+06,  0.67518376608988D+05,  0.40920228247872D+05,  0.24800138332043D+05,&
                    0.15030386867905D+05,  0.91093253744880D+04,  0.55208032572654D+04,  0.33459413680397D+04,&
                    0.20278432533574D+04,  0.12289959111257D+04,  0.74484600674283D+03,  0.45142182226838D+03,&
                    0.27358898319296D+03,  0.16581150496543D+03,  0.10049182119117D+03,  0.60904134055255D+02,&
                    0.36911596397124D+02,  0.22370664483105D+02,  0.13557978474609D+02,  0.82169566512784D+01,&
                    0.49799737280475D+01,  0.30181658957864D+01,  0.18291914519917D+01,  0.11086008799950D+01,&
                    0.67187932120909D+00,  0.40719958861157D+00,  0.24678762946156D+00,  0.14956826027973D+00/)
    coe(1:nfun)=(/  0.37359439237282D+07, -0.63155952716222D+07,  0.63757479440475D+07, -0.47131755727024D+07,&
                    0.32114975989821D+07, -0.18693058156331D+07,  0.11794797404478D+07, -0.59606043425292D+06,&
                    0.40937211652365D+06, -0.16031918928578D+06,  0.15430908560491D+06, -0.24729139349133D+05,&
                    0.70247565339871D+05,  0.11360774517784D+05,  0.35351378007744D+05,  0.10347667125193D+05,&
                    0.10720843051057D+05, -0.12613486195459D+04, -0.28363844731741D+03,  0.43845439041714D+03,&
                    0.37718646934828D+04,  0.15831869967958D+04, -0.80078063247797D+03, -0.39348847814077D+03,&
                    0.91500621559058D+03,  0.15506752907347D+03, -0.16779021918578D+03,  0.37193029212559D+02,&
                    0.39111952314338D+02,  0.10456766132043D+01, -0.11203387867267D+01,  0.11155922352933D+01,&
                    0.59284188228372D+00,  0.86209881201157D-01,  0.34865326199327D-02,  0.94960047372750D-04/)
  case(063)
    nfun = 36
    alf(1:nfun)=(/  0.61203673492837D+07,  0.37093135450204D+07,  0.22480688151639D+07,  0.13624659485842D+07,&
                    0.82573693853586D+06,  0.50044662941568D+06,  0.30330098752465D+06,  0.18381878031797D+06,&
                    0.11140532140483D+06,  0.67518376608988D+05,  0.40920228247872D+05,  0.24800138332043D+05,&
                    0.15030386867905D+05,  0.91093253744880D+04,  0.55208032572654D+04,  0.33459413680397D+04,&
                    0.20278432533574D+04,  0.12289959111257D+04,  0.74484600674283D+03,  0.45142182226838D+03,&
                    0.27358898319296D+03,  0.16581150496543D+03,  0.10049182119117D+03,  0.60904134055255D+02,&
                    0.36911596397124D+02,  0.22370664483105D+02,  0.13557978474609D+02,  0.82169566512784D+01,&
                    0.49799737280475D+01,  0.30181658957864D+01,  0.18291914519917D+01,  0.11086008799950D+01,&
                    0.67187932120909D+00,  0.40719958861157D+00,  0.24678762946156D+00,  0.14956826027973D+00/)
    coe(1:nfun)=(/  0.34825885285528D+07, -0.57807350809884D+07,  0.58407452662296D+07, -0.42858183640561D+07,&
                    0.29369424948664D+07, -0.16915135672309D+07,  0.10829264281925D+07, -0.53380547086465D+06,&
                    0.38044666620627D+06, -0.13856480728981D+06,  0.14708973284409D+06, -0.16437552046157D+05,&
                    0.69189502774455D+05,  0.14651721474918D+05,  0.35241756029816D+05,  0.11077299917114D+05,&
                    0.10150711948731D+05, -0.14201562690992D+04, -0.37652218310093D+03,  0.82899708079966D+03,&
                    0.39809930535842D+04,  0.14374181141807D+04, -0.96633552076308D+03, -0.25489467901617D+03,&
                    0.97928233547093D+03,  0.83018154757925D+02, -0.16248286015523D+03,  0.49768699093268D+02,&
                    0.36544263337453D+02, -0.10048646773157D+00, -0.84965510488189D+00,  0.11816763483901D+01,&
                    0.56014714823267D+00,  0.76204979932918D-01,  0.29357166319173D-02,  0.79042839265409D-04/)
  case(064)
    nfun = 36
    alf(1:nfun)=(/  0.61203673492837D+07,  0.37093135450204D+07,  0.22480688151639D+07,  0.13624659485842D+07,&
                    0.82573693853586D+06,  0.50044662941568D+06,  0.30330098752465D+06,  0.18381878031797D+06,&
                    0.11140532140483D+06,  0.67518376608988D+05,  0.40920228247872D+05,  0.24800138332043D+05,&
                    0.15030386867905D+05,  0.91093253744880D+04,  0.55208032572654D+04,  0.33459413680397D+04,&
                    0.20278432533574D+04,  0.12289959111257D+04,  0.74484600674283D+03,  0.45142182226838D+03,&
                    0.27358898319296D+03,  0.16581150496543D+03,  0.10049182119117D+03,  0.60904134055255D+02,&
                    0.36911596397124D+02,  0.22370664483105D+02,  0.13557978474609D+02,  0.82169566512784D+01,&
                    0.49799737280475D+01,  0.30181658957864D+01,  0.18291914519917D+01,  0.11086008799950D+01,&
                    0.67187932120909D+00,  0.40719958861157D+00,  0.24678762946156D+00,  0.14956826027973D+00/)
    coe(1:nfun)=(/  0.29967038548045D+07, -0.47418542123928D+07,  0.47757157882442D+07, -0.34407845950108D+07,&
                    0.23831676560094D+07, -0.13406511694955D+07,  0.88515253570771D+06, -0.41294643199182D+06,&
                    0.31882733526215D+06, -0.98112756515011D+05,  0.12972094749356D+06, -0.22816938790567D+04,&
                    0.65008207821283D+05,  0.19744274002621D+05,  0.34118030139864D+05,  0.12321963818325D+05,&
                    0.92303822079618D+04, -0.13994104392516D+04, -0.53654408844147D+03,  0.13049040070110D+04,&
                    0.41438562625658D+04,  0.12777116651248D+04, -0.11292384344240D+04, -0.91566858961312D+02,&
                    0.10299203942251D+04,  0.80256521185929D+01, -0.15470265686204D+03,  0.63267512782133D+02,&
                    0.33998853138387D+02, -0.19263601137557D+01, -0.43794077711698D+00,  0.14228492790275D+01,&
                    0.49799455441880D+00,  0.49603093307539D-01,  0.15340769358687D-02,  0.36034675424901D-04/)
  case(065)
    nfun = 36
    alf(1:nfun)=(/  0.61203673492837D+07,  0.37093135450204D+07,  0.22480688151639D+07,  0.13624659485842D+07,&
                    0.82573693853586D+06,  0.50044662941568D+06,  0.30330098752465D+06,  0.18381878031797D+06,&
                    0.11140532140483D+06,  0.67518376608988D+05,  0.40920228247872D+05,  0.24800138332043D+05,&
                    0.15030386867905D+05,  0.91093253744880D+04,  0.55208032572654D+04,  0.33459413680397D+04,&
                    0.20278432533574D+04,  0.12289959111257D+04,  0.74484600674283D+03,  0.45142182226838D+03,&
                    0.27358898319296D+03,  0.16581150496543D+03,  0.10049182119117D+03,  0.60904134055255D+02,&
                    0.36911596397124D+02,  0.22370664483105D+02,  0.13557978474609D+02,  0.82169566512784D+01,&
                    0.49799737280475D+01,  0.30181658957864D+01,  0.18291914519917D+01,  0.11086008799950D+01,&
                    0.67187932120909D+00,  0.40719958861157D+00,  0.24678762946156D+00,  0.14956826027973D+00/)
    coe(1:nfun)=(/  0.22834486566999D+07, -0.34103705008984D+07,  0.34645795564461D+07, -0.24260035173559D+07,&
                    0.17243355684905D+07, -0.92700019348896D+06,  0.65196161535490D+06, -0.27155129199469D+06,&
                    0.24617000818786D+06, -0.51153737863958D+05,  0.10903012194221D+06,  0.13921821744725D+05,&
                    0.59822906998996D+05,  0.25443070212597D+05,  0.32638856382747D+05,  0.13702458506165D+05,&
                    0.81758761448127D+04, -0.13127012898590D+04, -0.69237467551761D+03,  0.18238396560295D+04,&
                    0.42641926233840D+04,  0.10964373722261D+04, -0.12682142410077D+04,  0.94162404525690D+02,&
                    0.10460098977165D+04, -0.60091438762772D+02, -0.13576709987839D+03,  0.69735528080320D+02,&
                    0.30360058991599D+02, -0.17314385558544D+01, -0.28980558148066D+00,  0.12646427981744D+01,&
                    0.49654150937999D+00,  0.59827518450092D-01,  0.21351891305814D-02,  0.54300665247808D-04/)
  case(066)
    nfun = 38
    alf(1:nfun)=(/  0.16662700108425D+08,  0.10098606126318D+08,  0.61203673492837D+07,  0.37093135450204D+07,&
                    0.22480688151639D+07,  0.13624659485842D+07,  0.82573693853586D+06,  0.50044662941568D+06,&
                    0.30330098752465D+06,  0.18381878031797D+06,  0.11140532140483D+06,  0.67518376608988D+05,&
                    0.40920228247872D+05,  0.24800138332043D+05,  0.15030386867905D+05,  0.91093253744880D+04,&
                    0.55208032572654D+04,  0.33459413680397D+04,  0.20278432533574D+04,  0.12289959111257D+04,&
                    0.74484600674283D+03,  0.45142182226838D+03,  0.27358898319296D+03,  0.16581150496543D+03,&
                    0.10049182119117D+03,  0.60904134055255D+02,  0.36911596397124D+02,  0.22370664483105D+02,&
                    0.13557978474609D+02,  0.82169566512784D+01,  0.49799737280475D+01,  0.30181658957864D+01,&
                    0.18291914519917D+01,  0.11086008799950D+01,  0.67187932120909D+00,  0.40719958861157D+00,&
                    0.24678762946156D+00,  0.14956826027973D+00/)
    coe(1:nfun)=(/  0.30875898577642D+07, -0.47890043138275D+07,  0.49302196043358D+07, -0.35027497879610D+07,&
                    0.24888209353952D+07, -0.13555522642670D+07,  0.93659012663364D+06, -0.40501221265125D+06,&
                    0.34712516865754D+06, -0.85800091769814D+05,  0.14810036243400D+06,  0.10705929483145D+05,&
                    0.80254037042871D+05,  0.34872667688861D+05,  0.52129322005505D+05,  0.32591101165980D+05,&
                    0.30335180259521D+05,  0.15477903213572D+05,  0.68436427833668D+04, -0.10812101483488D+04,&
                   -0.88694480135548D+03,  0.24117723845278D+04,  0.43352695577296D+04,  0.90308835012755D+03,&
                   -0.13964846898043D+04,  0.29918226067273D+03,  0.10453368915221D+04, -0.12617667280395D+03,&
                   -0.11624663288368D+03,  0.77003772979264D+02,  0.27006858477773D+02, -0.22479588810238D+01,&
                   -0.20024466142388D-01,  0.12878453440009D+01,  0.46552595996984D+00,  0.53333737828122D-01,&
                    0.17942972985816D-02,  0.50072549108382D-04/)
  case(067)
    nfun = 39
    alf(1:nfun)=(/  0.27493455178901D+08,  0.16662700108425D+08,  0.10098606126318D+08,  0.61203673492837D+07,&
                    0.37093135450204D+07,  0.22480688151639D+07,  0.13624659485842D+07,  0.82573693853586D+06,&
                    0.50044662941568D+06,  0.30330098752465D+06,  0.18381878031797D+06,  0.11140532140483D+06,&
                    0.67518376608988D+05,  0.40920228247872D+05,  0.24800138332043D+05,  0.15030386867905D+05,&
                    0.91093253744880D+04,  0.55208032572654D+04,  0.33459413680397D+04,  0.20278432533574D+04,&
                    0.12289959111257D+04,  0.74484600674283D+03,  0.45142182226838D+03,  0.27358898319296D+03,&
                    0.16581150496543D+03,  0.10049182119117D+03,  0.60904134055255D+02,  0.36911596397124D+02,&
                    0.22370664483105D+02,  0.13557978474609D+02,  0.82169566512784D+01,  0.49799737280475D+01,&
                    0.30181658957864D+01,  0.18291914519917D+01,  0.11086008799950D+01,  0.67187932120909D+00,&
                    0.40719958861157D+00,  0.24678762946156D+00,  0.14956826027973D+00/)
    coe(1:nfun)=(/  0.20263807075977D+07, -0.28535135726013D+07,  0.31007482913723D+07, -0.20715078206921D+07,&
                    0.15843188205290D+07, -0.76773226039505D+06,  0.62502003787231D+06, -0.20120326072860D+06,&
                    0.25487028351653D+06, -0.13772657502532D+05,  0.12517305092597D+06,  0.38650326858759D+05,&
                    0.78621820737913D+05,  0.48646541768678D+05,  0.57551778348450D+05,  0.43632510567846D+05,&
                    0.40224229925441D+05,  0.27737308538760D+05,  0.17348312842587D+05,  0.54013476254176D+04,&
                   -0.79192430408974D+03, -0.10665269918871D+04,  0.30359692606303D+04,  0.43657730774804D+04,&
                    0.69102599350436D+03, -0.15009504198808D+04,  0.51774984476241D+03,  0.10222003878164D+04,&
                   -0.18625858952684D+03, -0.93752333712490D+02,  0.82430793477728D+02,  0.23652731375141D+02,&
                   -0.26067471830177D+01,  0.24220644416442D+00,  0.12987634376021D+01,  0.43629163288793D+00,&
                    0.47475086266040D-01,  0.15410218170189D-02,  0.43308322793978D-04/)
  case(068)
    nfun = 37
    alf(1:nfun)=(/  0.10098606126318D+08,  0.61203673492837D+07,  0.37093135450204D+07,  0.22480688151639D+07,&
                    0.13624659485842D+07,  0.82573693853586D+06,  0.50044662941568D+06,  0.30330098752465D+06,&
                    0.18381878031797D+06,  0.11140532140483D+06,  0.67518376608988D+05,  0.40920228247872D+05,&
                    0.24800138332043D+05,  0.15030386867905D+05,  0.91093253744880D+04,  0.55208032572654D+04,&
                    0.33459413680397D+04,  0.20278432533574D+04,  0.12289959111257D+04,  0.74484600674283D+03,&
                    0.45142182226838D+03,  0.27358898319296D+03,  0.16581150496543D+03,  0.10049182119117D+03,&
                    0.60904134055255D+02,  0.36911596397124D+02,  0.22370664483105D+02,  0.13557978474609D+02,&
                    0.82169566512784D+01,  0.49799737280475D+01,  0.30181658957864D+01,  0.18291914519917D+01,&
                    0.11086008799950D+01,  0.67187932120909D+00,  0.40719958861157D+00,  0.24678762946156D+00,&
                    0.14956826027973D+00/)
    coe(1:nfun)=(/  0.31632162610130D+07, -0.46487217214832D+07,  0.47299366127176D+07, -0.32901719949188D+07,&
                    0.23535223495545D+07, -0.12565456782017D+07,  0.89017195189309D+06, -0.36693245434475D+06,&
                    0.33463555326229D+06, -0.70107788410246D+05,  0.14679575546221D+06,  0.17148908339381D+05,&
                    0.80417555516288D+05,  0.35157233888099D+05,  0.47886084519254D+05,  0.25090012864963D+05,&
                    0.19174118785162D+05,  0.39285038499164D+04, -0.48465192728175D+03, -0.12044308477691D+04,&
                    0.36788985745786D+04,  0.43601877671368D+04,  0.45867428271462D+03, -0.15755737142273D+04,&
                    0.74421480070754D+03,  0.97740020057216D+03, -0.23872266773241D+03, -0.69219066047505D+02,&
                    0.86068866207624D+02,  0.20381915017211D+02, -0.28194832876710D+01,  0.49109554116274D+00,&
                    0.12989865583482D+01,  0.40856688261540D+00,  0.42373944077607D-01,  0.13665934315448D-02,&
                    0.35253666518464D-04/)
  case(069)
    nfun = 37
    alf(1:nfun)=(/  0.10098606126318D+08,  0.61203673492837D+07,  0.37093135450204D+07,  0.22480688151639D+07,&
                    0.13624659485842D+07,  0.82573693853586D+06,  0.50044662941568D+06,  0.30330098752465D+06,&
                    0.18381878031797D+06,  0.11140532140483D+06,  0.67518376608988D+05,  0.40920228247872D+05,&
                    0.24800138332043D+05,  0.15030386867905D+05,  0.91093253744880D+04,  0.55208032572654D+04,&
                    0.33459413680397D+04,  0.20278432533574D+04,  0.12289959111257D+04,  0.74484600674283D+03,&
                    0.45142182226838D+03,  0.27358898319296D+03,  0.16581150496543D+03,  0.10049182119117D+03,&
                    0.60904134055255D+02,  0.36911596397124D+02,  0.22370664483105D+02,  0.13557978474609D+02,&
                    0.82169566512784D+01,  0.49799737280475D+01,  0.30181658957864D+01,  0.18291914519917D+01,&
                    0.11086008799950D+01,  0.67187932120909D+00,  0.40719958861157D+00,  0.24678762946156D+00,&
                    0.14956826027973D+00/)
    coe(1:nfun)=(/  0.50867534593182D+07, -0.79919281089201D+07,  0.80634988687801D+07, -0.57956840336409D+07,&
                    0.40258287050285D+07, -0.22622363403454D+07,  0.14931793999026D+07, -0.69945584793085D+06,&
                    0.53275468863285D+06, -0.17180204561338D+06,  0.21143632514578D+06, -0.12029354507796D+05,&
                    0.10222527387051D+06,  0.27400467862658D+05,  0.55184742157908D+05,  0.22605041985189D+05,&
                    0.20834291620176D+05,  0.24940085271092D+04, -0.19314654909854D+03, -0.12780896776195D+04,&
                    0.43253646315884D+04,  0.43221325596854D+04,  0.20548276603727D+03, -0.16156621327323D+04,&
                    0.97329448857361D+03,  0.91282334497650D+03, -0.28296632501548D+03, -0.43460475163555D+02,&
                    0.88188668077056D+02,  0.17234100407764D+02, -0.29249894895509D+01,  0.72622738570777D+00,&
                    0.12929715006109D+01,  0.38233166048525D+00,  0.37588087249166D-01,  0.11912504921918D-02,&
                    0.28127670987144D-04/)
  case(070)
    nfun = 36
    alf(1:nfun)=(/  0.10098606126318D+08,  0.61203673492837D+07,  0.37093135450204D+07,  0.22480688151639D+07,&
                    0.13624659485842D+07,  0.82573693853586D+06,  0.50044662941568D+06,  0.30330098752465D+06,&
                    0.18381878031797D+06,  0.11140532140483D+06,  0.67518376608988D+05,  0.40920228247872D+05,&
                    0.24800138332043D+05,  0.15030386867905D+05,  0.91093253744880D+04,  0.55208032572654D+04,&
                    0.33459413680397D+04,  0.20278432533574D+04,  0.12289959111257D+04,  0.74484600674283D+03,&
                    0.45142182226838D+03,  0.27358898319296D+03,  0.16581150496543D+03,  0.10049182119117D+03,&
                    0.60904134055255D+02,  0.36911596397124D+02,  0.22370664483105D+02,  0.13557978474609D+02,&
                    0.82169566512784D+01,  0.49799737280475D+01,  0.30181658957864D+01,  0.18291914519917D+01,&
                    0.11086008799950D+01,  0.67187932120909D+00,  0.40719958861157D+00,  0.24678762946156D+00/)
    coe(1:nfun)=(/  0.80039699074917D+07, -0.11799080629162D+08,  0.11385320901871D+08, -0.80844209638672D+07,&
                    0.54903720327730D+07, -0.31149225789752D+07,  0.20003648230324D+07, -0.97364976521945D+06,&
                    0.69799625022769D+06, -0.25424175427872D+06,  0.26574103340256D+06, -0.35081627969265D+05,&
                    0.12085162158348D+06,  0.21542528441317D+05,  0.61454846520331D+05,  0.20645967779182D+05,&
                    0.22123333256833D+05,  0.12146063349294D+04,  0.23066149628954D+02, -0.12505150072853D+04,&
                    0.49508288998420D+04,  0.42602003890577D+04, -0.71291233913886D+02, -0.16144610891224D+04,&
                    0.11976590109162D+04,  0.83082089046247D+03, -0.31812291852616D+03, -0.17142008564597D+02,&
                    0.88729585255425D+02,  0.14316163635597D+02, -0.29417112647081D+01,  0.95297853722343D+00,&
                    0.12752989032811D+01,  0.35909434835273D+00,  0.32897010416811D-01,  0.12121694181570D-02/)
  case(071)
    nfun = 36
    alf(1:nfun)=(/  0.10098606126318D+08,  0.61203673492837D+07,  0.37093135450204D+07,  0.22480688151639D+07,&
                    0.13624659485842D+07,  0.82573693853586D+06,  0.50044662941568D+06,  0.30330098752465D+06,&
                    0.18381878031797D+06,  0.11140532140483D+06,  0.67518376608988D+05,  0.40920228247872D+05,&
                    0.24800138332043D+05,  0.15030386867905D+05,  0.91093253744880D+04,  0.55208032572654D+04,&
                    0.33459413680397D+04,  0.20278432533574D+04,  0.12289959111257D+04,  0.74484600674283D+03,&
                    0.45142182226838D+03,  0.27358898319296D+03,  0.16581150496543D+03,  0.10049182119117D+03,&
                    0.60904134055255D+02,  0.36911596397124D+02,  0.22370664483105D+02,  0.13557978474609D+02,&
                    0.82169566512784D+01,  0.49799737280475D+01,  0.30181658957864D+01,  0.18291914519917D+01,&
                    0.11086008799950D+01,  0.67187932120909D+00,  0.40719958861157D+00,  0.24678762946156D+00/)
    coe(1:nfun)=(/  0.87980333493872D+07, -0.13862492640722D+08,  0.13727883347004D+08, -0.99505942965956D+07,&
                    0.67824447043924D+07, -0.38993088991337D+07,  0.24802130123174D+07, -0.12353092142050D+07,&
                    0.85812396276289D+06, -0.33359433532071D+06,  0.31893254599574D+06, -0.57187597587839D+05,&
                    0.13920460976180D+06,  0.15962667699359D+05,  0.67596816181837D+05,  0.18707181614516D+05,&
                    0.23319908974897D+05, -0.63324677587021D+02,  0.25225934164575D+03, -0.11677175899751D+04,&
                    0.55811856718273D+04,  0.41600463915619D+04, -0.36219531717882D+03, -0.15803548141489D+04,&
                    0.14218173864590D+04,  0.73458581844427D+03, -0.34874720175415D+03,  0.96585579163948D+01,&
                    0.90326373022370D+02,  0.10616062648524D+02, -0.32490198790251D+01,  0.14529874252181D+01,&
                    0.13473238692060D+01,  0.29206017469087D+00,  0.19930940791255D-01,  0.64830275175298D-03/)
end select

return
end

!-----------------------------------------------------------------------------------------------------------------------------------
!---  EDF data library: ncore = 60, iz = 72-103
!-----------------------------------------------------------------------------------------------------------------------------------
subroutine EDFLIB60(iz,nfun,alf,coe)
implicit real(kind=8) (a-h,o-z)
real(kind=8) :: alf(*),coe(*)

select case(iz)
  case(072)
    nfun = 35
    alf(1:nfun)=(/  0.10098606126318D+08,  0.61203673492837D+07,  0.37093135450204D+07,  0.22480688151639D+07,&
                    0.13624659485842D+07,  0.82573693853586D+06,  0.50044662941568D+06,  0.30330098752465D+06,&
                    0.18381878031797D+06,  0.11140532140483D+06,  0.67518376608988D+05,  0.40920228247872D+05,&
                    0.24800138332043D+05,  0.15030386867905D+05,  0.91093253744880D+04,  0.55208032572654D+04,&
                    0.33459413680397D+04,  0.20278432533574D+04,  0.12289959111257D+04,  0.74484600674283D+03,&
                    0.45142182226838D+03,  0.27358898319296D+03,  0.16581150496543D+03,  0.10049182119117D+03,&
                    0.60904134055255D+02,  0.36911596397124D+02,  0.22370664483105D+02,  0.13557978474609D+02,&
                    0.82169566512784D+01,  0.49799737280475D+01,  0.30181658957864D+01,  0.18291914519917D+01,&
                    0.11086008799950D+01,  0.67187932120909D+00,  0.40719958861157D+00/)
    coe(1:nfun)=(/  0.59411224264193D+07, -0.12241585186169D+08,  0.13452425414347D+08, -0.10273650157994D+08,&
                    0.72023489278779D+07, -0.42024565945535D+07,  0.26949279551289D+07, -0.13501975179210D+07,&
                    0.93778304507196D+06, -0.36802969984281D+06,  0.34759864823377D+06, -0.65452499012530D+05,&
                    0.14998365428064D+06,  0.14612211949032D+05,  0.71322679707577D+05,  0.18000534093462D+05,&
                    0.23731949197371D+05, -0.92772053364851D+03,  0.31218676428334D+03, -0.91421474817814D+03,&
                    0.60744072475656D+04,  0.40074735968898D+04, -0.62212540954803D+03, -0.14302104348168D+04,&
                    0.15424863053392D+04,  0.60538760924831D+03, -0.33600482402465D+03,  0.18402225310732D+02,&
                    0.87331143857414D+02,  0.24714225157649D+02,  0.51754906766367D+01,  0.10237890719528D+01,&
                    0.23500303384921D+00,  0.15709083098635D-01,  0.25425324509936D-02/)
  case(073)
    nfun = 35
    alf(1:nfun)=(/  0.10098606126318D+08,  0.61203673492837D+07,  0.37093135450204D+07,  0.22480688151639D+07,&
                    0.13624659485842D+07,  0.82573693853586D+06,  0.50044662941568D+06,  0.30330098752465D+06,&
                    0.18381878031797D+06,  0.11140532140483D+06,  0.67518376608988D+05,  0.40920228247872D+05,&
                    0.24800138332043D+05,  0.15030386867905D+05,  0.91093253744880D+04,  0.55208032572654D+04,&
                    0.33459413680397D+04,  0.20278432533574D+04,  0.12289959111257D+04,  0.74484600674283D+03,&
                    0.45142182226838D+03,  0.27358898319296D+03,  0.16581150496543D+03,  0.10049182119117D+03,&
                    0.60904134055255D+02,  0.36911596397124D+02,  0.22370664483105D+02,  0.13557978474609D+02,&
                    0.82169566512784D+01,  0.49799737280475D+01,  0.30181658957864D+01,  0.18291914519917D+01,&
                    0.11086008799950D+01,  0.67187932120909D+00,  0.40719958861157D+00/)
    coe(1:nfun)=(/  0.52356129652815D+07, -0.11339273825469D+08,  0.12760363540397D+08, -0.97846560698814D+07,&
                    0.69298611460009D+07, -0.40217755890279D+07,  0.26107014894376D+07, -0.12866192704974D+07,&
                    0.91600204323047D+06, -0.34413453870419D+06,  0.34423730953634D+06, -0.55356904998124D+05,&
                    0.15091549283732D+06,  0.18934181136336D+05,  0.71984823928462D+05,  0.18976039925183D+05,&
                    0.23141922024848D+05, -0.12973488185946D+04,  0.89274686631281D+02, -0.45003458954790D+03,&
                    0.65311101671763D+04,  0.39056258234330D+04, -0.95688965322808D+03, -0.12931787823000D+04,&
                    0.17150321872730D+04,  0.49568693762145D+03, -0.35069154583519D+03,  0.43125067975460D+02,&
                    0.88542000209515D+02,  0.23771169877203D+02,  0.47130718012726D+01,  0.91379881807006D+00,&
                    0.15666833638907D+00,  0.80440752910249D-02,  0.11214405064219D-02/)
  case(074)
    nfun = 35
    alf(1:nfun)=(/  0.10098606126318D+08,  0.61203673492837D+07,  0.37093135450204D+07,  0.22480688151639D+07,&
                    0.13624659485842D+07,  0.82573693853586D+06,  0.50044662941568D+06,  0.30330098752465D+06,&
                    0.18381878031797D+06,  0.11140532140483D+06,  0.67518376608988D+05,  0.40920228247872D+05,&
                    0.24800138332043D+05,  0.15030386867905D+05,  0.91093253744880D+04,  0.55208032572654D+04,&
                    0.33459413680397D+04,  0.20278432533574D+04,  0.12289959111257D+04,  0.74484600674283D+03,&
                    0.45142182226838D+03,  0.27358898319296D+03,  0.16581150496543D+03,  0.10049182119117D+03,&
                    0.60904134055255D+02,  0.36911596397124D+02,  0.22370664483105D+02,  0.13557978474609D+02,&
                    0.82169566512784D+01,  0.49799737280475D+01,  0.30181658957864D+01,  0.18291914519917D+01,&
                    0.11086008799950D+01,  0.67187932120909D+00,  0.40719958861157D+00/)
    coe(1:nfun)=(/  0.62957190925303D+07, -0.11650455689798D+08,  0.12573245513553D+08, -0.93716487391796D+07,&
                    0.66114908216505D+07, -0.37747995149045D+07,  0.24783508786967D+07, -0.11904631281369D+07,&
                    0.87568606270832D+06, -0.30872628970429D+06,  0.33489547401315D+06, -0.41501692334205D+05,&
                    0.14998960568444D+06,  0.24403914917205D+05,  0.72007133518293D+05,  0.20236379760307D+05,&
                    0.22304575149194D+05, -0.15669671379875D+04, -0.15660021987616D+03,  0.96835621565697D+02,&
                    0.69585277775721D+04,  0.37679899563249D+04, -0.12934074219567D+04, -0.11154429113499D+04,&
                    0.18686665883112D+04,  0.37688997942926D+03, -0.35654212920221D+03,  0.67158830881553D+02,&
                    0.89268008792840D+02,  0.22661731854513D+02,  0.42495342858803D+01,  0.77353594733376D+00,&
                    0.96470429250076D-01,  0.46860515386662D-02,  0.44866815607847D-03/)
  case(075)
    nfun = 35
    alf(1:nfun)=(/  0.10098606126318D+08,  0.61203673492837D+07,  0.37093135450204D+07,  0.22480688151639D+07,&
                    0.13624659485842D+07,  0.82573693853586D+06,  0.50044662941568D+06,  0.30330098752465D+06,&
                    0.18381878031797D+06,  0.11140532140483D+06,  0.67518376608988D+05,  0.40920228247872D+05,&
                    0.24800138332043D+05,  0.15030386867905D+05,  0.91093253744880D+04,  0.55208032572654D+04,&
                    0.33459413680397D+04,  0.20278432533574D+04,  0.12289959111257D+04,  0.74484600674283D+03,&
                    0.45142182226838D+03,  0.27358898319296D+03,  0.16581150496543D+03,  0.10049182119117D+03,&
                    0.60904134055255D+02,  0.36911596397124D+02,  0.22370664483105D+02,  0.13557978474609D+02,&
                    0.82169566512784D+01,  0.49799737280475D+01,  0.30181658957864D+01,  0.18291914519917D+01,&
                    0.11086008799950D+01,  0.67187932120909D+00,  0.40719958861157D+00/)
    coe(1:nfun)=(/  0.70636817969351D+07, -0.11545589996164D+08,  0.12019470492002D+08, -0.86949384564972D+07,&
                    0.61275385212522D+07, -0.34268140364203D+07,  0.22890956336595D+07, -0.10607277490450D+07,&
                    0.81748393959868D+06, -0.26256228619453D+06,  0.32014963014512D+06, -0.24252902290992D+05,&
                    0.14742923230844D+06,  0.30885893704645D+05,  0.71458555428559D+05,  0.21736954300882D+05,&
                    0.21242225922102D+05, -0.17433944262888D+04, -0.41685192728054D+03,  0.72368323218165D+03,&
                    0.73521759862097D+04,  0.35939388380680D+04, -0.16267184995616D+04, -0.89734161465336D+03,&
                    0.19993145800508D+04,  0.25184579157181D+03, -0.35395610902515D+03,  0.90415792483766D+02,&
                    0.89409741429381D+02,  0.21390035692235D+02,  0.37851831165228D+01,  0.62606504377970D+00,&
                    0.55413184568721D-01,  0.30024304036225D-02,  0.10814822823457D-03/)
  case(076)
    nfun = 34
    alf(1:nfun)=(/  0.10098606126318D+08,  0.61203673492837D+07,  0.37093135450204D+07,  0.22480688151639D+07,&
                    0.13624659485842D+07,  0.82573693853586D+06,  0.50044662941568D+06,  0.30330098752465D+06,&
                    0.18381878031797D+06,  0.11140532140483D+06,  0.67518376608988D+05,  0.40920228247872D+05,&
                    0.24800138332043D+05,  0.15030386867905D+05,  0.91093253744880D+04,  0.55208032572654D+04,&
                    0.33459413680397D+04,  0.20278432533574D+04,  0.12289959111257D+04,  0.74484600674283D+03,&
                    0.45142182226838D+03,  0.27358898319296D+03,  0.16581150496543D+03,  0.10049182119117D+03,&
                    0.60904134055255D+02,  0.36911596397124D+02,  0.22370664483105D+02,  0.13557978474609D+02,&
                    0.82169566512784D+01,  0.49799737280475D+01,  0.30181658957864D+01,  0.18291914519917D+01,&
                    0.11086008799950D+01,  0.67187932120909D+00/)
    coe(1:nfun)=(/  0.65240876647873D+07, -0.10147351197273D+08,  0.10569909485947D+08, -0.75001406095467D+07,&
                    0.53580090906496D+07, -0.29219421279826D+07,  0.20154870915775D+07, -0.88351347087765D+06,&
                    0.73414142583442D+06, -0.20173995409292D+06,  0.29788346530026D+06, -0.24326730510741D+04,&
                    0.14257747719185D+06,  0.38737351920374D+05,  0.70134174948696D+05,  0.23581881406806D+05,&
                    0.19899325749502D+05, -0.17934204900372D+04, -0.70265161576867D+03,  0.14375525980184D+04,&
                    0.77018204497669D+04,  0.33863444877139D+04, -0.19528948952956D+04, -0.63984418856046D+03,&
                    0.21035694347750D+04,  0.12320768702361D+03, -0.34322062622055D+03,  0.11262715684519D+03,&
                    0.88952308789152D+02,  0.19981613597554D+02,  0.33326749789653D+01,  0.48244455153252D+00,&
                    0.29774770756745D-01,  0.17474591318809D-02/)
  case(077)
    nfun = 34
    alf(1:nfun)=(/  0.10098606126318D+08,  0.61203673492837D+07,  0.37093135450204D+07,  0.22480688151639D+07,&
                    0.13624659485842D+07,  0.82573693853586D+06,  0.50044662941568D+06,  0.30330098752465D+06,&
                    0.18381878031797D+06,  0.11140532140483D+06,  0.67518376608988D+05,  0.40920228247872D+05,&
                    0.24800138332043D+05,  0.15030386867905D+05,  0.91093253744880D+04,  0.55208032572654D+04,&
                    0.33459413680397D+04,  0.20278432533574D+04,  0.12289959111257D+04,  0.74484600674283D+03,&
                    0.45142182226838D+03,  0.27358898319296D+03,  0.16581150496543D+03,  0.10049182119117D+03,&
                    0.60904134055255D+02,  0.36911596397124D+02,  0.22370664483105D+02,  0.13557978474609D+02,&
                    0.82169566512784D+01,  0.49799737280475D+01,  0.30181658957864D+01,  0.18291914519917D+01,&
                    0.11086008799950D+01,  0.67187932120909D+00/)
    coe(1:nfun)=(/  0.53867530632906D+07, -0.80963073995571D+07,  0.86328326964130D+07, -0.59977781171951D+07,&
                    0.44099993670378D+07, -0.23135542748799D+07,  0.16854634470103D+07, -0.67342763278532D+06,&
                    0.63361203311598D+06, -0.13052704820108D+06,  0.27048118506470D+06,  0.22646771773040D+05,&
                    0.13617408130284D+06,  0.47538109176509D+05,  0.68260472608123D+05,  0.25639808135835D+05,&
                    0.18346385564242D+05, -0.17499414701560D+04, -0.99019888926204D+03,  0.22258335758161D+04,&
                    0.80083465476256D+04,  0.31416525639337D+04, -0.22639876390865D+04, -0.34640837422363D+03,&
                    0.21792252758259D+04, -0.64288368928792D+01, -0.32487454376850D+03,  0.13354355787457D+03,&
                    0.87957271070285D+02,  0.18457685472599D+02,  0.28788804473077D+01,  0.35443118284149D+00,&
                    0.16501334765010D-01,  0.98572605494951D-03/)
  case(078)
    nfun = 36
    alf(1:nfun)=(/  0.27493455178901D+08,  0.16662700108425D+08,  0.10098606126318D+08,  0.61203673492837D+07,&
                    0.37093135450204D+07,  0.22480688151639D+07,  0.13624659485842D+07,  0.82573693853586D+06,&
                    0.50044662941568D+06,  0.30330098752465D+06,  0.18381878031797D+06,  0.11140532140483D+06,&
                    0.67518376608988D+05,  0.40920228247872D+05,  0.24800138332043D+05,  0.15030386867905D+05,&
                    0.91093253744880D+04,  0.55208032572654D+04,  0.33459413680397D+04,  0.20278432533574D+04,&
                    0.12289959111257D+04,  0.74484600674283D+03,  0.45142182226838D+03,  0.27358898319296D+03,&
                    0.16581150496543D+03,  0.10049182119117D+03,  0.60904134055255D+02,  0.36911596397124D+02,&
                    0.22370664483105D+02,  0.13557978474609D+02,  0.82169566512784D+01,  0.49799737280475D+01,&
                    0.30181658957864D+01,  0.18291914519917D+01,  0.11086008799950D+01,  0.67187932120909D+00/)
    coe(1:nfun)=(/  0.83664482896576D+07, -0.15822936282318D+08,  0.17682768643356D+08, -0.13104575024651D+08,&
                    0.94333513489685D+07, -0.52735007079353D+07,  0.35778957836571D+07, -0.16468531949091D+07,&
                    0.12838814050481D+07, -0.40845112842560D+06,  0.49944939231658D+06, -0.38385020225823D+05,&
                    0.23167868798700D+06,  0.54615424085528D+05,  0.12614813081780D+06,  0.58447964870989D+05,&
                    0.65177689392872D+05,  0.28271280541597D+05,  0.16384799842184D+05, -0.14986315856185D+04,&
                   -0.13353048108811D+04,  0.31187674572882D+04,  0.82479115580000D+04,  0.28715607199058D+04,&
                   -0.25604703397117D+04, -0.16586579327035D+02,  0.22220870327268D+04, -0.13333839193805D+03,&
                   -0.30003805242573D+03,  0.15320294141194D+03,  0.86349933397130D+02,  0.16855074033019D+02,&
                    0.24285728059940D+01,  0.25467883559674D+00,  0.99940269308689D-02,  0.64816324133510D-03/)
  case(079)
    nfun = 36
    alf(1:nfun)=(/  0.27493455178901D+08,  0.16662700108425D+08,  0.10098606126318D+08,  0.61203673492837D+07,&
                    0.37093135450204D+07,  0.22480688151639D+07,  0.13624659485842D+07,  0.82573693853586D+06,&
                    0.50044662941568D+06,  0.30330098752465D+06,  0.18381878031797D+06,  0.11140532140483D+06,&
                    0.67518376608988D+05,  0.40920228247872D+05,  0.24800138332043D+05,  0.15030386867905D+05,&
                    0.91093253744880D+04,  0.55208032572654D+04,  0.33459413680397D+04,  0.20278432533574D+04,&
                    0.12289959111257D+04,  0.74484600674283D+03,  0.45142182226838D+03,  0.27358898319296D+03,&
                    0.16581150496543D+03,  0.10049182119117D+03,  0.60904134055255D+02,  0.36911596397124D+02,&
                    0.22370664483105D+02,  0.13557978474609D+02,  0.82169566512784D+01,  0.49799737280475D+01,&
                    0.30181658957864D+01,  0.18291914519917D+01,  0.11086008799950D+01,  0.67187932120909D+00/)
    coe(1:nfun)=(/  0.42840408493958D+07, -0.72350311061096D+07,  0.88650420517654D+07, -0.61507660274429D+07,&
                    0.48489567998505D+07, -0.23988701037464D+07,  0.19311652570410D+07, -0.66804781383657D+06,&
                    0.76046797362971D+06, -0.90722372559547D+05,  0.34257800605702D+06,  0.65561798961341D+05,&
                    0.18753682728085D+06,  0.89810067912430D+05,  0.11462936848139D+06,  0.70245864922695D+05,&
                    0.61565419947928D+05,  0.31089275896128D+05,  0.14231602093499D+05, -0.11547544057327D+04,&
                   -0.16679484828722D+04,  0.40760222680640D+04,  0.84369091768801D+04,  0.25653127677613D+04,&
                   -0.28298950364599D+04,  0.34194364855748D+03,  0.22329560364980D+04, -0.25628021113381D+03,&
                   -0.26897196021036D+03,  0.17145780468957D+03,  0.84072866139472D+02,  0.15233029294940D+02,&
                    0.20126305528348D+01,  0.17695548470323D+00,  0.56762585247299D-02,  0.37292571571967D-03/)
  case(080)
    nfun = 32
    alf(1:nfun)=(/  0.10098606126318D+08,  0.61203673492837D+07,  0.37093135450204D+07,  0.22480688151639D+07,&
                    0.13624659485842D+07,  0.82573693853586D+06,  0.50044662941568D+06,  0.30330098752465D+06,&
                    0.18381878031797D+06,  0.11140532140483D+06,  0.67518376608988D+05,  0.40920228247872D+05,&
                    0.24800138332043D+05,  0.15030386867905D+05,  0.91093253744880D+04,  0.55208032572654D+04,&
                    0.33459413680397D+04,  0.20278432533574D+04,  0.12289959111257D+04,  0.74484600674283D+03,&
                    0.45142182226838D+03,  0.27358898319296D+03,  0.16581150496543D+03,  0.10049182119117D+03,&
                    0.60904134055255D+02,  0.36911596397124D+02,  0.22370664483105D+02,  0.13557978474609D+02,&
                    0.82169566512784D+01,  0.49799737280475D+01,  0.30181658957864D+01,  0.18291914519917D+01/)
    coe(1:nfun)=(/  0.12085146174961D+08, -0.27998311791740D+08,  0.31642012106499D+08, -0.24649623207382D+08,&
                    0.17287538510220D+08, -0.10237639871637D+08,  0.64620068961926D+07, -0.33469207316666D+07,&
                    0.22073294113183D+07, -0.96284700424361D+06,  0.77563137210441D+06, -0.22625739953190D+06,&
                    0.29843292800999D+06, -0.26284499421805D+05,  0.11799019679652D+06,  0.49790726339817D+03,&
                    0.30521547384743D+05, -0.11051176257920D+05,  0.37556727764290D+04,  0.19010888889767D+04,&
                    0.10352906080250D+05,  0.12227721961010D+04, -0.24967467727778D+04,  0.39812585206237D+03,&
                    0.23950279342742D+04, -0.47164507784382D+03, -0.18446957178019D+03,  0.16865821218042D+03,&
                    0.87123113513147D+02,  0.12458351672208D+02,  0.17018806499950D+01,  0.14347365858086D+00/)
  case(081)
    nfun = 37
    alf(1:nfun)=(/  0.45364201045187D+08,  0.27493455178901D+08,  0.16662700108425D+08,  0.10098606126318D+08,&
                    0.61203673492837D+07,  0.37093135450204D+07,  0.22480688151639D+07,  0.13624659485842D+07,&
                    0.82573693853586D+06,  0.50044662941568D+06,  0.30330098752465D+06,  0.18381878031797D+06,&
                    0.11140532140483D+06,  0.67518376608988D+05,  0.40920228247872D+05,  0.24800138332043D+05,&
                    0.15030386867905D+05,  0.91093253744880D+04,  0.55208032572654D+04,  0.33459413680397D+04,&
                    0.20278432533574D+04,  0.12289959111257D+04,  0.74484600674283D+03,  0.45142182226838D+03,&
                    0.27358898319296D+03,  0.16581150496543D+03,  0.10049182119117D+03,  0.60904134055255D+02,&
                    0.36911596397124D+02,  0.22370664483105D+02,  0.13557978474609D+02,  0.82169566512784D+01,&
                    0.49799737280475D+01,  0.30181658957864D+01,  0.18291914519917D+01,  0.11086008799950D+01,&
                    0.67187932120909D+00/)
    coe(1:nfun)=(/  0.13940761553040D+08, -0.21089801682602D+08,  0.21892104355972D+08, -0.15204194760948D+08,&
                    0.10936210643562D+08, -0.58417120633678D+07,  0.41144278386726D+07, -0.17411510580988D+07,&
                    0.14963720566611D+07, -0.39049051591825D+06,  0.60634426911819D+06, -0.88465438216925D+03,&
                    0.29268757499683D+06,  0.90742912442029D+05,  0.16621797885026D+06,  0.88990823077038D+05,&
                    0.95462360472564D+05,  0.53284662292706D+05,  0.36965869368912D+05,  0.95337753992379D+04,&
                   -0.26633631252125D+03, -0.22276834420492D+04,  0.61380324334323D+04,  0.86607284061820D+04,&
                    0.18441528620041D+04, -0.32604211851663D+04,  0.11236048787464D+04,  0.21561117030978D+04,&
                   -0.48112551346673D+03, -0.19037496896920D+03,  0.20273997820909D+03,  0.77573447457634D+02,&
                    0.12085755043244D+02,  0.13165927228227D+01,  0.74126053450755D-01,  0.19349145982233D-02,&
                    0.13615701789149D-04/)
  case(082)
    nfun = 34
    alf(1:nfun)=(/  0.16662700108425D+08,  0.10098606126318D+08,  0.61203673492837D+07,  0.37093135450204D+07,&
                    0.22480688151639D+07,  0.13624659485842D+07,  0.82573693853586D+06,  0.50044662941568D+06,&
                    0.30330098752465D+06,  0.18381878031797D+06,  0.11140532140483D+06,  0.67518376608988D+05,&
                    0.40920228247872D+05,  0.24800138332043D+05,  0.15030386867905D+05,  0.91093253744880D+04,&
                    0.55208032572654D+04,  0.33459413680397D+04,  0.20278432533574D+04,  0.12289959111257D+04,&
                    0.74484600674283D+03,  0.45142182226838D+03,  0.27358898319296D+03,  0.16581150496543D+03,&
                    0.10049182119117D+03,  0.60904134055255D+02,  0.36911596397124D+02,  0.22370664483105D+02,&
                    0.13557978474609D+02,  0.82169566512784D+01,  0.49799737280475D+01,  0.30181658957864D+01,&
                    0.18291914519917D+01,  0.11086008799950D+01/)
    coe(1:nfun)=(/  0.93624463762550D+07, -0.12350832531097D+08,  0.12818314485962D+08, -0.84910529691582D+07,&
                    0.63220996692276D+07, -0.31549485163612D+07,  0.24058853019094D+07, -0.88551050441885D+06,&
                    0.91919762904644D+06, -0.15267205394936D+06,  0.39739968735278D+06,  0.48119032981873D+05,&
                    0.20199535377097D+06,  0.77826234437421D+05,  0.10725468151520D+06,  0.49478522653293D+05,&
                    0.39547086762447D+05,  0.72507399228048D+04,  0.15052096988261D+03, -0.23743695211783D+04,&
                    0.71947794376966D+04,  0.87065425034547D+04,  0.14243300195243D+04, -0.34056066397767D+04,&
                    0.15332520863684D+04,  0.20703036481991D+04, -0.57979064315119D+03, -0.14422971692352D+03,&
                    0.21543178780552D+03,  0.73560956859887D+02,  0.10599956722047D+02,  0.10290422370647D+01,&
                    0.45531935675172D-01,  0.96439955223104D-03/)
  case(083)
    nfun = 34
    alf(1:nfun)=(/  0.16662700108425D+08,  0.10098606126318D+08,  0.61203673492837D+07,  0.37093135450204D+07,&
                    0.22480688151639D+07,  0.13624659485842D+07,  0.82573693853586D+06,  0.50044662941568D+06,&
                    0.30330098752465D+06,  0.18381878031797D+06,  0.11140532140483D+06,  0.67518376608988D+05,&
                    0.40920228247872D+05,  0.24800138332043D+05,  0.15030386867905D+05,  0.91093253744880D+04,&
                    0.55208032572654D+04,  0.33459413680397D+04,  0.20278432533574D+04,  0.12289959111257D+04,&
                    0.74484600674283D+03,  0.45142182226838D+03,  0.27358898319296D+03,  0.16581150496543D+03,&
                    0.10049182119117D+03,  0.60904134055255D+02,  0.36911596397124D+02,  0.22370664483105D+02,&
                    0.13557978474609D+02,  0.82169566512784D+01,  0.49799737280475D+01,  0.30181658957864D+01,&
                    0.18291914519917D+01,  0.11086008799950D+01/)
    coe(1:nfun)=(/  0.12307487306065D+08, -0.17285839949699D+08,  0.17811981448078D+08, -0.12185161473831D+08,&
                    0.88320834327831D+07, -0.46287377789345D+07,  0.33167272093382D+07, -0.13669749620824D+07,&
                    0.12228879504075D+07, -0.29670013247311D+06,  0.49856837678206D+06,  0.83382019343376D+04,&
                    0.23662382143179D+06,  0.67543187896848D+05,  0.11859011153357D+06,  0.45847784493845D+05,&
                    0.41899266022608D+05,  0.50177962405663D+04,  0.54990531005245D+03, -0.24214112139712D+04,&
                    0.82553947954464D+04,  0.87077545818117D+04,  0.96627554889768D+03, -0.34956686777726D+04,&
                    0.19465163677725D+04,  0.19545682621495D+04, -0.66721019153654D+03, -0.94509091642038D+02,&
                    0.22599714282112D+03,  0.69131850128721D+02,  0.92070996414050D+01,  0.78345699052238D+00,&
                    0.26899258403148D-01,  0.44304106635451D-03/)
  case(084)
    nfun = 35
    alf(1:nfun)=(/  0.16662700108425D+08,  0.10098606126318D+08,  0.61203673492837D+07,  0.37093135450204D+07,&
                    0.22480688151639D+07,  0.13624659485842D+07,  0.82573693853586D+06,  0.50044662941568D+06,&
                    0.30330098752465D+06,  0.18381878031797D+06,  0.11140532140483D+06,  0.67518376608988D+05,&
                    0.40920228247872D+05,  0.24800138332043D+05,  0.15030386867905D+05,  0.91093253744880D+04,&
                    0.55208032572654D+04,  0.33459413680397D+04,  0.20278432533574D+04,  0.12289959111257D+04,&
                    0.74484600674283D+03,  0.45142182226838D+03,  0.27358898319296D+03,  0.16581150496543D+03,&
                    0.10049182119117D+03,  0.60904134055255D+02,  0.36911596397124D+02,  0.22370664483105D+02,&
                    0.13557978474609D+02,  0.82169566512784D+01,  0.49799737280475D+01,  0.30181658957864D+01,&
                    0.18291914519917D+01,  0.11086008799950D+01,  0.67187932120909D+00/)
    coe(1:nfun)=(/  0.14561408302056D+08, -0.20934761041311D+08,  0.21532092150621D+08, -0.14898802832880D+08,&
                    0.10700213726064D+08, -0.57045912430835D+07,  0.39980212249174D+07, -0.17141835948133D+07,&
                    0.14530892384387D+07, -0.39803874217212D+06,  0.57713692285019D+06, -0.18080236756921D+05,&
                    0.26428151948091D+06,  0.61369893046789D+05,  0.12766334932609D+06,  0.43380617303953D+05,&
                    0.43474968366863D+05,  0.31363665326876D+04,  0.77749121332425D+03, -0.22818128836801D+04,&
                    0.92700331280463D+04,  0.86829407116276D+04,  0.46317198308225D+03, -0.35209846501044D+04,&
                    0.23543654633680D+04,  0.18119134618390D+04, -0.74268483480549D+03, -0.41714211521368D+02,&
                    0.23423168574503D+03,  0.64414168524127D+02,  0.79144047201282D+01,  0.57923186260632D+00,&
                    0.15715511103679D-01,  0.78959655446909D-04,  0.14130118493938D-04/)
  case(085)
    nfun = 35
    alf(1:nfun)=(/  0.16662700108425D+08,  0.10098606126318D+08,  0.61203673492837D+07,  0.37093135450204D+07,&
                    0.22480688151639D+07,  0.13624659485842D+07,  0.82573693853586D+06,  0.50044662941568D+06,&
                    0.30330098752465D+06,  0.18381878031797D+06,  0.11140532140483D+06,  0.67518376608988D+05,&
                    0.40920228247872D+05,  0.24800138332043D+05,  0.15030386867905D+05,  0.91093253744880D+04,&
                    0.55208032572654D+04,  0.33459413680397D+04,  0.20278432533574D+04,  0.12289959111257D+04,&
                    0.74484600674283D+03,  0.45142182226838D+03,  0.27358898319296D+03,  0.16581150496543D+03,&
                    0.10049182119117D+03,  0.60904134055255D+02,  0.36911596397124D+02,  0.22370664483105D+02,&
                    0.13557978474609D+02,  0.82169566512784D+01,  0.49799737280475D+01,  0.30181658957864D+01,&
                    0.18291914519917D+01,  0.11086008799950D+01,  0.67187932120909D+00/)
    coe(1:nfun)=(/  0.16420382455072D+08, -0.23883501773355D+08,  0.24581996622534D+08, -0.17099035048225D+08,&
                    0.12237884202115D+08, -0.65737966593337D+07,  0.45626599364622D+07, -0.19917588865643D+07,&
                    0.16462719489870D+07, -0.47686809632379D+06,  0.64431114010090D+06, -0.37355429858938D+05,&
                    0.28846215494895D+06,  0.57325816220004D+05,  0.13557527925682D+06,  0.41452012015106D+05,&
                    0.44614423152763D+05,  0.14261510454803D+04,  0.93771055169683D+03, -0.20053959495761D+04,&
                    0.10258703865892D+05,  0.86156291782869D+04, -0.73672963866207D+02, -0.34811813735861D+04,&
                    0.27518583555233D+04,  0.16437820308896D+04, -0.80478273102689D+03,  0.13206531171554D+02,&
                    0.24013539589084D+03,  0.59485513339809D+02,  0.67388530727731D+01,  0.41156767540696D+00,&
                    0.10151235382779D-01, -0.30145468066678D-03,  0.42519149628717D-04/)
  case(086)
    nfun = 35
    alf(1:nfun)=(/  0.16662700108425D+08,  0.10098606126318D+08,  0.61203673492837D+07,  0.37093135450204D+07,&
                    0.22480688151639D+07,  0.13624659485842D+07,  0.82573693853586D+06,  0.50044662941568D+06,&
                    0.30330098752465D+06,  0.18381878031797D+06,  0.11140532140483D+06,  0.67518376608988D+05,&
                    0.40920228247872D+05,  0.24800138332043D+05,  0.15030386867905D+05,  0.91093253744880D+04,&
                    0.55208032572654D+04,  0.33459413680397D+04,  0.20278432533574D+04,  0.12289959111257D+04,&
                    0.74484600674283D+03,  0.45142182226838D+03,  0.27358898319296D+03,  0.16581150496543D+03,&
                    0.10049182119117D+03,  0.60904134055255D+02,  0.36911596397124D+02,  0.22370664483105D+02,&
                    0.13557978474609D+02,  0.82169566512784D+01,  0.49799737280475D+01,  0.30181658957864D+01,&
                    0.18291914519917D+01,  0.11086008799950D+01,  0.67187932120909D+00/)
    coe(1:nfun)=(/  0.17578143136864D+08, -0.25559698910959D+08,  0.26380580005800D+08, -0.18336752839072D+08,&
                    0.13143338194248D+08, -0.70497769895296D+07,  0.49008939556653D+07, -0.21369863023406D+07,&
                    0.17670949048914D+07, -0.51339195280868D+06,  0.68903214813265D+06, -0.43333120998055D+05,&
                    0.30571305936735D+06,  0.57317469958540D+05,  0.14124404811702D+06,  0.40651098106833D+05,&
                    0.44988888370074D+05,  0.74858904403169D+02,  0.93690892235492D+03, -0.15368360112727D+04,&
                    0.11185264949067D+05,  0.85182660128108D+04, -0.64648911973508D+03, -0.33696566525384D+04,&
                    0.31305721623576D+04,  0.14537648821440D+04, -0.85310558814132D+03,  0.69621272945860D+02,&
                    0.24362775508962D+03,  0.54466469356661D+02,  0.56791502754925D+01,  0.27832898798999D+00,&
                    0.80348088448545D-02, -0.67769446528132D-03,  0.75295055732560D-04/)
  case(087)
    nfun = 33
    alf(1:nfun)=(/  0.16662700108425D+08,  0.10098606126318D+08,  0.61203673492837D+07,  0.37093135450204D+07,&
                    0.22480688151639D+07,  0.13624659485842D+07,  0.82573693853586D+06,  0.50044662941568D+06,&
                    0.30330098752465D+06,  0.18381878031797D+06,  0.11140532140483D+06,  0.67518376608988D+05,&
                    0.40920228247872D+05,  0.24800138332043D+05,  0.15030386867905D+05,  0.91093253744880D+04,&
                    0.55208032572654D+04,  0.33459413680397D+04,  0.20278432533574D+04,  0.12289959111257D+04,&
                    0.74484600674283D+03,  0.45142182226838D+03,  0.27358898319296D+03,  0.16581150496543D+03,&
                    0.10049182119117D+03,  0.60904134055255D+02,  0.36911596397124D+02,  0.22370664483105D+02,&
                    0.13557978474609D+02,  0.82169566512784D+01,  0.49799737280475D+01,  0.30181658957864D+01,&
                    0.18291914519917D+01/)
    coe(1:nfun)=(/  0.17916860730469D+08, -0.26118711789062D+08,  0.27073099757812D+08, -0.18640672054688D+08,&
                    0.13416926000000D+08, -0.71546569199219D+07,  0.50433822578125D+07, -0.21669067656250D+07,&
                    0.18202393979492D+07, -0.50699608520508D+06,  0.71093000048828D+06, -0.36427361877441D+05,&
                    0.31685555517578D+06,  0.60693795288086D+05,  0.14506628567505D+06,  0.40762000061035D+05,&
                    0.44701159839630D+05, -0.96682412528992D+03,  0.80731646060944D+03, -0.88883934307098D+03,&
                    0.12050272485018D+05,  0.83850796576738D+04, -0.12483375140429D+04, -0.31847222978771D+04,&
                    0.34842131703943D+04,  0.12449682523757D+04, -0.88726006654091D+03,  0.12689815491252D+03,&
                    0.24461816213746D+03,  0.49525894423365D+02,  0.46991211592976D+01,  0.18966312719567D+00,&
                    0.32248954748866D-02/)
  case(088)
    nfun = 33
    alf(1:nfun)=(/  0.16662700108425D+08,  0.10098606126318D+08,  0.61203673492837D+07,  0.37093135450204D+07,&
                    0.22480688151639D+07,  0.13624659485842D+07,  0.82573693853586D+06,  0.50044662941568D+06,&
                    0.30330098752465D+06,  0.18381878031797D+06,  0.11140532140483D+06,  0.67518376608988D+05,&
                    0.40920228247872D+05,  0.24800138332043D+05,  0.15030386867905D+05,  0.91093253744880D+04,&
                    0.55208032572654D+04,  0.33459413680397D+04,  0.20278432533574D+04,  0.12289959111257D+04,&
                    0.74484600674283D+03,  0.45142182226838D+03,  0.27358898319296D+03,  0.16581150496543D+03,&
                    0.10049182119117D+03,  0.60904134055255D+02,  0.36911596397124D+02,  0.22370664483105D+02,&
                    0.13557978474609D+02,  0.82169566512784D+01,  0.49799737280475D+01,  0.30181658957864D+01,&
                    0.18291914519917D+01/)
    coe(1:nfun)=(/  0.17713937593750D+08, -0.25478877414062D+08,  0.26662717656250D+08, -0.18197017085938D+08,&
                    0.13240151574219D+08, -0.69618229414062D+07,  0.50014087792969D+07, -0.20870264453125D+07,&
                    0.18193041357422D+07, -0.47169302148438D+06,  0.71869010205078D+06, -0.20287544372559D+05,&
                    0.32283359942627D+06,  0.67368804473877D+05,  0.14696220347595D+06,  0.41805692565918D+05,&
                    0.43759895736694D+05, -0.17076467361450D+04,  0.56892118644714D+03, -0.72446859359741D+02,&
                    0.12854678524971D+05,  0.82071699130535D+04, -0.18669165517092D+04, -0.29295312270522D+04,&
                    0.38092985918522D+04,  0.10191753273979D+04, -0.90603449850529D+03,  0.18376717964932D+03,&
                    0.24342439056560D+03,  0.44626006852835D+02,  0.38540270260710D+01,  0.11778395549482D+00,&
                    0.24806674782667D-02/)
  case(089)
    nfun = 33
    alf(1:nfun)=(/  0.16662700108425D+08,  0.10098606126318D+08,  0.61203673492837D+07,  0.37093135450204D+07,&
                    0.22480688151639D+07,  0.13624659485842D+07,  0.82573693853586D+06,  0.50044662941568D+06,&
                    0.30330098752465D+06,  0.18381878031797D+06,  0.11140532140483D+06,  0.67518376608988D+05,&
                    0.40920228247872D+05,  0.24800138332043D+05,  0.15030386867905D+05,  0.91093253744880D+04,&
                    0.55208032572654D+04,  0.33459413680397D+04,  0.20278432533574D+04,  0.12289959111257D+04,&
                    0.74484600674283D+03,  0.45142182226838D+03,  0.27358898319296D+03,  0.16581150496543D+03,&
                    0.10049182119117D+03,  0.60904134055255D+02,  0.36911596397124D+02,  0.22370664483105D+02,&
                    0.13557978474609D+02,  0.82169566512784D+01,  0.49799737280475D+01,  0.30181658957864D+01,&
                    0.18291914519917D+01/)
    coe(1:nfun)=(/  0.17267519386719D+08, -0.24196900796875D+08,  0.25573071687500D+08, -0.17195432984375D+08,&
                    0.12694350246094D+08, -0.65325867929688D+07,  0.48264356093750D+07, -0.19275223925781D+07,&
                    0.17762674619141D+07, -0.41012068457031D+06,  0.71270955737305D+06,  0.46151971435547D+04,&
                    0.32444684628296D+06,  0.76559746520996D+05,  0.14748323374939D+06,  0.43438097122192D+05,&
                    0.42349932716370D+05, -0.22369291343689D+04,  0.27580611610413D+03,  0.88832482576370D+03,&
                    0.13602128159523D+05,  0.79778786505461D+04, -0.24928954244852D+04, -0.26050145305693D+04,&
                    0.41004240781814D+04,  0.77992211880535D+03, -0.90939233386703D+03,  0.23948441371601D+03,&
                    0.24015501386765D+03,  0.39857997502899D+02,  0.31223885064537D+01,  0.66636754119827D-01,&
                    0.22625932579103D-02/)
  case(090)
    nfun = 33
    alf(1:nfun)=(/  0.16662700108425D+08,  0.10098606126318D+08,  0.61203673492837D+07,  0.37093135450204D+07,&
                    0.22480688151639D+07,  0.13624659485842D+07,  0.82573693853586D+06,  0.50044662941568D+06,&
                    0.30330098752465D+06,  0.18381878031797D+06,  0.11140532140483D+06,  0.67518376608988D+05,&
                    0.40920228247872D+05,  0.24800138332043D+05,  0.15030386867905D+05,  0.91093253744880D+04,&
                    0.55208032572654D+04,  0.33459413680397D+04,  0.20278432533574D+04,  0.12289959111257D+04,&
                    0.74484600674283D+03,  0.45142182226838D+03,  0.27358898319296D+03,  0.16581150496543D+03,&
                    0.10049182119117D+03,  0.60904134055255D+02,  0.36911596397124D+02,  0.22370664483105D+02,&
                    0.13557978474609D+02,  0.82169566512784D+01,  0.49799737280475D+01,  0.30181658957864D+01,&
                    0.18291914519917D+01/)
    coe(1:nfun)=(/  0.16364175597656D+08, -0.21969496781250D+08,  0.23523397187500D+08, -0.15433942617188D+08,&
                    0.11653295566406D+08, -0.57927650683594D+07,  0.44746080644531D+07, -0.16620422949219D+07,&
                    0.16762109257812D+07, -0.31466123828125D+06,  0.68953100036621D+06,  0.39847056030273D+05,&
                    0.32080962454224D+06,  0.88847411376953D+05,  0.14620357513428D+06,  0.45916034622192D+05,&
                    0.40326974670410D+05, -0.24727066993713D+04, -0.10105715847015D+03,  0.20073038191795D+04,&
                    0.14278220802546D+05,  0.76989918226004D+04, -0.31195779706836D+04, -0.22119650462568D+04,&
                    0.43521803674549D+04,  0.53102651736885D+03, -0.89748691618629D+03,  0.29332427105121D+03,&
                    0.23494549224433D+03,  0.35284235483035D+02,  0.25021564972412D+01,  0.30894539413566D-01,&
                    0.23369061409539D-02/)
  case(091)
    nfun = 33
    alf(1:nfun)=(/  0.16662700108425D+08,  0.10098606126318D+08,  0.61203673492837D+07,  0.37093135450204D+07,&
                    0.22480688151639D+07,  0.13624659485842D+07,  0.82573693853586D+06,  0.50044662941568D+06,&
                    0.30330098752465D+06,  0.18381878031797D+06,  0.11140532140483D+06,  0.67518376608988D+05,&
                    0.40920228247872D+05,  0.24800138332043D+05,  0.15030386867905D+05,  0.91093253744880D+04,&
                    0.55208032572654D+04,  0.33459413680397D+04,  0.20278432533574D+04,  0.12289959111257D+04,&
                    0.74484600674283D+03,  0.45142182226838D+03,  0.27358898319296D+03,  0.16581150496543D+03,&
                    0.10049182119117D+03,  0.60904134055255D+02,  0.36911596397124D+02,  0.22370664483105D+02,&
                    0.13557978474609D+02,  0.82169566512784D+01,  0.49799737280475D+01,  0.30181658957864D+01,&
                    0.18291914519917D+01/)
    coe(1:nfun)=(/  0.15739847921875D+08, -0.19821684039062D+08,  0.21472426804688D+08, -0.13615426265625D+08,&
                    0.10570285816406D+08, -0.50070170957031D+07,  0.40976120468750D+07, -0.13769374921875D+07,&
                    0.15693830351562D+07, -0.21362723437500D+06,  0.66491404968262D+06,  0.76973212463379D+05,&
                    0.31648985971069D+06,  0.10174571160889D+06,  0.14448187242126D+06,  0.48484501190186D+05,&
                    0.38095451766968D+05, -0.26220543022156D+04, -0.44675066471100D+03,  0.32275704641342D+04,&
                    0.14904482821465D+05,  0.73552413082123D+04, -0.37325274707079D+04, -0.17542796936929D+04,&
                    0.45597187339216D+04,  0.27637692364305D+03, -0.86988240077533D+03,  0.34371298571769D+03,&
                    0.22828704690142D+03,  0.30931954990607D+02,  0.19755602108780D+01,  0.10824078890437D-01,&
                    0.24323047628059D-02/)
  case(092)
    nfun = 35
    alf(1:nfun)=(/  0.45364201045187D+08,  0.27493455178901D+08,  0.16662700108425D+08,  0.10098606126318D+08,&
                    0.61203673492837D+07,  0.37093135450204D+07,  0.22480688151639D+07,  0.13624659485842D+07,&
                    0.82573693853586D+06,  0.50044662941568D+06,  0.30330098752465D+06,  0.18381878031797D+06,&
                    0.11140532140483D+06,  0.67518376608988D+05,  0.40920228247872D+05,  0.24800138332043D+05,&
                    0.15030386867905D+05,  0.91093253744880D+04,  0.55208032572654D+04,  0.33459413680397D+04,&
                    0.20278432533574D+04,  0.12289959111257D+04,  0.74484600674283D+03,  0.45142182226838D+03,&
                    0.27358898319296D+03,  0.16581150496543D+03,  0.10049182119117D+03,  0.60904134055255D+02,&
                    0.36911596397124D+02,  0.22370664483105D+02,  0.13557978474609D+02,  0.82169566512784D+01,&
                    0.49799737280475D+01,  0.30181658957864D+01,  0.18291914519917D+01/)
    coe(1:nfun)=(/  0.23530619312836D+08, -0.42389792347839D+08,  0.49882199024048D+08, -0.36151016678192D+08,&
                    0.26965429151947D+08, -0.14289535306908D+08,  0.10157242815910D+08, -0.43197464970360D+07,&
                    0.36912564968567D+07, -0.10400028699608D+07,  0.14260335101986D+07, -0.86257419380903D+05,&
                    0.62611160963666D+06,  0.12306901802719D+06,  0.30766635610202D+06,  0.11722349572170D+06,&
                    0.14124822030775D+06,  0.51707459130950D+05,  0.35354879863419D+05, -0.25105572727285D+04,&
                   -0.84307526393700D+03,  0.45916871290435D+04,  0.15446922067831D+05,  0.69603838850029D+04,&
                   -0.43284668843192D+04, -0.12347843396850D+04,  0.47197271789203D+04,  0.20172028870278D+02,&
                   -0.82865880336743D+03,  0.39142422894822D+03,  0.21987753224141D+03,  0.26896322802510D+02,&
                    0.15462123028233D+01, -0.17635542138805D-02,  0.23558468404588D-02/)
  case(093)
    nfun = 35
    alf(1:nfun)=(/  0.45364201045187D+08,  0.27493455178901D+08,  0.16662700108425D+08,  0.10098606126318D+08,&
                    0.61203673492837D+07,  0.37093135450204D+07,  0.22480688151639D+07,  0.13624659485842D+07,&
                    0.82573693853586D+06,  0.50044662941568D+06,  0.30330098752465D+06,  0.18381878031797D+06,&
                    0.11140532140483D+06,  0.67518376608988D+05,  0.40920228247872D+05,  0.24800138332043D+05,&
                    0.15030386867905D+05,  0.91093253744880D+04,  0.55208032572654D+04,  0.33459413680397D+04,&
                    0.20278432533574D+04,  0.12289959111257D+04,  0.74484600674283D+03,  0.45142182226838D+03,&
                    0.27358898319296D+03,  0.16581150496543D+03,  0.10049182119117D+03,  0.60904134055255D+02,&
                    0.36911596397124D+02,  0.22370664483105D+02,  0.13557978474609D+02,  0.82169566512784D+01,&
                    0.49799737280475D+01,  0.30181658957864D+01,  0.18291914519917D+01/)
    coe(1:nfun)=(/  0.18870041316467D+08, -0.31876938709412D+08,  0.39467488058868D+08, -0.27727387045898D+08,&
                    0.21603200542786D+08, -0.10772392858673D+08,  0.82450781842422D+07, -0.30964967732582D+07,&
                    0.31011220499153D+07, -0.63153398282909D+06,  0.12584649552112D+07,  0.53487870536327D+05,&
                    0.58328547801840D+06,  0.17206891126794D+06,  0.29782906787747D+06,  0.13336361280642D+06,&
                    0.13754947097975D+06,  0.54995599674292D+05,  0.32430561428249D+05, -0.23147452356759D+04,&
                   -0.11828888031412D+04,  0.60402228459865D+04,  0.15930339887054D+05,  0.64972265484917D+04,&
                   -0.48897923737479D+04, -0.66200801906572D+03,  0.48308203744091D+04, -0.23474380823424D+03,&
                   -0.77375950273324D+03,  0.43530781262033D+03,  0.21015079075576D+03,  0.23178199352956D+02,&
                    0.12025049026761D+01, -0.87820372129954D-02,  0.22319759538334D-02/)
  case(094)
    nfun = 35
    alf(1:nfun)=(/  0.45364201045187D+08,  0.27493455178901D+08,  0.16662700108425D+08,  0.10098606126318D+08,&
                    0.61203673492837D+07,  0.37093135450204D+07,  0.22480688151639D+07,  0.13624659485842D+07,&
                    0.82573693853586D+06,  0.50044662941568D+06,  0.30330098752465D+06,  0.18381878031797D+06,&
                    0.11140532140483D+06,  0.67518376608988D+05,  0.40920228247872D+05,  0.24800138332043D+05,&
                    0.15030386867905D+05,  0.91093253744880D+04,  0.55208032572654D+04,  0.33459413680397D+04,&
                    0.20278432533574D+04,  0.12289959111257D+04,  0.74484600674283D+03,  0.45142182226838D+03,&
                    0.27358898319296D+03,  0.16581150496543D+03,  0.10049182119117D+03,  0.60904134055255D+02,&
                    0.36911596397124D+02,  0.22370664483105D+02,  0.13557978474609D+02,  0.82169566512784D+01,&
                    0.49799737280475D+01,  0.30181658957864D+01,  0.18291914519917D+01/)
    coe(1:nfun)=(/  0.14728449969528D+08, -0.21375884279388D+08,  0.28737358904556D+08, -0.18843989123917D+08,&
                    0.15868012397972D+08, -0.69744619226685D+07,  0.61707581033363D+07, -0.17686576161957D+07,&
                    0.24574511070442D+07, -0.18684995796108D+06,  0.10737367428632D+07,  0.20554776318955D+06,&
                    0.53508374103904D+06,  0.22505140542287D+06,  0.28621948895159D+06,  0.15053226200844D+06,&
                    0.13317747023904D+06,  0.58443519203112D+05,  0.29279222135477D+05, -0.19941484770048D+04,&
                   -0.14792667100662D+04,  0.75786886233748D+04,  0.16340195740945D+05,  0.59713719856096D+04,&
                   -0.54096296765454D+04, -0.40170135212655D+02,  0.48894820363256D+04, -0.48378907309519D+03,&
                   -0.70651573037675D+03,  0.47491752189479D+03,  0.19938024282117D+03,  0.19795049206249D+02,&
                    0.92711330415932D+00, -0.11014412643476D-01,  0.19753258786182D-02/)
  case(095)
    nfun = 35
    alf(1:nfun)=(/  0.45364201045187D+08,  0.27493455178901D+08,  0.16662700108425D+08,  0.10098606126318D+08,&
                    0.61203673492837D+07,  0.37093135450204D+07,  0.22480688151639D+07,  0.13624659485842D+07,&
                    0.82573693853586D+06,  0.50044662941568D+06,  0.30330098752465D+06,  0.18381878031797D+06,&
                    0.11140532140483D+06,  0.67518376608988D+05,  0.40920228247872D+05,  0.24800138332043D+05,&
                    0.15030386867905D+05,  0.91093253744880D+04,  0.55208032572654D+04,  0.33459413680397D+04,&
                    0.20278432533574D+04,  0.12289959111257D+04,  0.74484600674283D+03,  0.45142182226838D+03,&
                    0.27358898319296D+03,  0.16581150496543D+03,  0.10049182119117D+03,  0.60904134055255D+02,&
                    0.36911596397124D+02,  0.22370664483105D+02,  0.13557978474609D+02,  0.82169566512784D+01,&
                    0.49799737280475D+01,  0.30181658957864D+01,  0.18291914519917D+01/)
    coe(1:nfun)=(/  0.10627684709229D+08, -0.10870035062836D+08,  0.18133733859833D+08, -0.10086961903656D+08,&
                    0.10298606006500D+08, -0.32779182865067D+07,  0.41770227932358D+07, -0.47565095648003D+06,&
                    0.18431378358269D+07,  0.24636431775951D+06,  0.90031383457136D+06,  0.35403837361526D+06,&
                    0.49072901829565D+06,  0.27697230388021D+06,  0.27564716363120D+06,  0.16725125470218D+06,&
                    0.12888634311484D+06,  0.61648802371204D+05,  0.26115226546951D+05, -0.16621068461221D+04,&
                   -0.16584129277021D+04,  0.91631275687683D+04,  0.16691488319443D+05,  0.53725536747668D+04,&
                   -0.58724575508414D+04,  0.61976269237540D+03,  0.48974156428162D+04, -0.72520144029977D+03,&
                   -0.62756765311577D+03,  0.51002785943090D+03,  0.18765269004994D+03,  0.16788282183120D+02,&
                    0.71121186417349D+00, -0.11719638036084D-01,  0.17203720862540D-02/)
  case(096)
    nfun = 36
    alf(1:nfun)=(/  0.74850931724558D+08,  0.45364201045187D+08,  0.27493455178901D+08,  0.16662700108425D+08,&
                    0.10098606126318D+08,  0.61203673492837D+07,  0.37093135450204D+07,  0.22480688151639D+07,&
                    0.13624659485842D+07,  0.82573693853586D+06,  0.50044662941568D+06,  0.30330098752465D+06,&
                    0.18381878031797D+06,  0.11140532140483D+06,  0.67518376608988D+05,  0.40920228247872D+05,&
                    0.24800138332043D+05,  0.15030386867905D+05,  0.91093253744880D+04,  0.55208032572654D+04,&
                    0.33459413680397D+04,  0.20278432533574D+04,  0.12289959111257D+04,  0.74484600674283D+03,&
                    0.45142182226838D+03,  0.27358898319296D+03,  0.16581150496543D+03,  0.10049182119117D+03,&
                    0.60904134055255D+02,  0.36911596397124D+02,  0.22370664483105D+02,  0.13557978474609D+02,&
                    0.82169566512784D+01,  0.49799737280475D+01,  0.30181658957864D+01,  0.18291914519917D+01/)
    coe(1:nfun)=(/ -0.80322926334381D+07,  0.17955645205048D+08, -0.86868206329956D+07,  0.12282811680908D+08,&
                   -0.36869928980103D+07,  0.58913091564102D+07, -0.12702974369812D+06,  0.24735403774300D+07,&
                    0.67537884598541D+06,  0.13126177687550D+07,  0.64085333218765D+06,  0.75163033686543D+06,&
                    0.49175736042404D+06,  0.45417667186356D+06,  0.32545882637423D+06,  0.26743069498312D+06,&
                    0.18270356249575D+06,  0.12511999882581D+06,  0.64354865730964D+05,  0.23086002890386D+05,&
                   -0.13785237373468D+04, -0.16809905972751D+04,  0.10768788236834D+05,  0.16985109136707D+05,&
                    0.47004334868955D+04, -0.62678069861904D+04,  0.13093343763338D+04,  0.48545486686181D+04,&
                   -0.95593482643878D+03, -0.53800800059192D+03,  0.54028741095661D+03,  0.17520630529152D+03,&
                    0.14173054344126D+02,  0.53599768604909D+00, -0.10414061977031D-01,  0.13660381663669D-02/)
  case(097)
    nfun = 36
    alf(1:nfun)=(/  0.74850931724558D+08,  0.45364201045187D+08,  0.27493455178901D+08,  0.16662700108425D+08,&
                    0.10098606126318D+08,  0.61203673492837D+07,  0.37093135450204D+07,  0.22480688151639D+07,&
                    0.13624659485842D+07,  0.82573693853586D+06,  0.50044662941568D+06,  0.30330098752465D+06,&
                    0.18381878031797D+06,  0.11140532140483D+06,  0.67518376608988D+05,  0.40920228247872D+05,&
                    0.24800138332043D+05,  0.15030386867905D+05,  0.91093253744880D+04,  0.55208032572654D+04,&
                    0.33459413680397D+04,  0.20278432533574D+04,  0.12289959111257D+04,  0.74484600674283D+03,&
                    0.45142182226838D+03,  0.27358898319296D+03,  0.16581150496543D+03,  0.10049182119117D+03,&
                    0.60904134055255D+02,  0.36911596397124D+02,  0.22370664483105D+02,  0.13557978474609D+02,&
                    0.82169566512784D+01,  0.49799737280475D+01,  0.30181658957864D+01,  0.18291914519917D+01/)
    coe(1:nfun)=(/  0.99511586681366D+07, -0.80593298213806D+07,  0.16035928061829D+08, -0.50461659224548D+07,&
                    0.79006345537567D+07, -0.73622349821472D+06,  0.40249040954514D+07,  0.32062923993683D+06,&
                    0.20595954476757D+07,  0.67211803452492D+06,  0.10994405786676D+07,  0.57338375676680D+06,&
                    0.64852277559781D+06,  0.40912530002356D+06,  0.37967202800149D+06,  0.25650614624503D+06,&
                    0.19967386278877D+06,  0.12035275756787D+06,  0.67364835944869D+05,  0.19775514709901D+05,&
                   -0.92316801865213D+03, -0.16491631761035D+04,  0.12444751922666D+05,  0.17185529684978D+05,&
                    0.39741625901751D+04, -0.65969031736811D+04,  0.20273193088924D+04,  0.47561702405809D+04,&
                   -0.11699276625170D+04, -0.44032169978463D+03,  0.56549322436993D+03,  0.16246704502264D+03,&
                    0.11868103126327D+02,  0.39972550389848D+00, -0.74264602176299D-02,  0.95866729132865D-03/)
  case(098)
    nfun = 36
    alf(1:nfun)=(/  0.74850931724558D+08,  0.45364201045187D+08,  0.27493455178901D+08,  0.16662700108425D+08,&
                    0.10098606126318D+08,  0.61203673492837D+07,  0.37093135450204D+07,  0.22480688151639D+07,&
                    0.13624659485842D+07,  0.82573693853586D+06,  0.50044662941568D+06,  0.30330098752465D+06,&
                    0.18381878031797D+06,  0.11140532140483D+06,  0.67518376608988D+05,  0.40920228247872D+05,&
                    0.24800138332043D+05,  0.15030386867905D+05,  0.91093253744880D+04,  0.55208032572654D+04,&
                    0.33459413680397D+04,  0.20278432533574D+04,  0.12289959111257D+04,  0.74484600674283D+03,&
                    0.45142182226838D+03,  0.27358898319296D+03,  0.16581150496543D+03,  0.10049182119117D+03,&
                    0.60904134055255D+02,  0.36911596397124D+02,  0.22370664483105D+02,  0.13557978474609D+02,&
                    0.82169566512784D+01,  0.49799737280475D+01,  0.30181658957864D+01,  0.18291914519917D+01/)
    coe(1:nfun)=(/  0.25126566714401D+08, -0.28739965857605D+08,  0.35457054028259D+08, -0.18304638996368D+08,&
                    0.16878020074265D+08, -0.57659401726685D+07,  0.72759313279495D+07, -0.13021029419708D+07,&
                    0.31587028207245D+07,  0.19950245432854D+06,  0.14718695824680D+07,  0.44752850270605D+06,&
                    0.77914738495326D+06,  0.38050482728422D+06,  0.42577166363358D+06,  0.25043929690757D+06,&
                    0.21397669047529D+06,  0.11682942178367D+06,  0.69473519019600D+05,  0.16827024954494D+05,&
                   -0.61091403147951D+03, -0.13936037383582D+04,  0.14095293986304D+05,  0.17331510977982D+05,&
                    0.31747593521356D+04, -0.68382179116294D+04,  0.27558170666621D+04,  0.46094833778972D+04,&
                   -0.13679544784044D+04, -0.33461983621198D+03,  0.58547335272245D+03,  0.14948960607409D+03,&
                    0.99179685309846D+01,  0.28754466749183D+00, -0.36532167689529D-02,  0.50127396750099D-03/)
  case(099)
    nfun = 37
    alf(1:nfun)=(/  0.74850931724558D+08,  0.45364201045187D+08,  0.27493455178901D+08,  0.16662700108425D+08,&
                    0.10098606126318D+08,  0.61203673492837D+07,  0.37093135450204D+07,  0.22480688151639D+07,&
                    0.13624659485842D+07,  0.82573693853586D+06,  0.50044662941568D+06,  0.30330098752465D+06,&
                    0.18381878031797D+06,  0.11140532140483D+06,  0.67518376608988D+05,  0.40920228247872D+05,&
                    0.24800138332043D+05,  0.15030386867905D+05,  0.91093253744880D+04,  0.55208032572654D+04,&
                    0.33459413680397D+04,  0.20278432533574D+04,  0.12289959111257D+04,  0.74484600674283D+03,&
                    0.45142182226838D+03,  0.27358898319296D+03,  0.16581150496543D+03,  0.10049182119117D+03,&
                    0.60904134055255D+02,  0.36911596397124D+02,  0.22370664483105D+02,  0.13557978474609D+02,&
                    0.82169566512784D+01,  0.49799737280475D+01,  0.30181658957864D+01,  0.18291914519917D+01,&
                    0.11086008799950D+01/)
    coe(1:nfun)=(/  0.34860262473618D+08, -0.44124588437836D+08,  0.51361329304337D+08, -0.29482781633804D+08,&
                    0.24720488574860D+08, -0.10127439846130D+08,  0.10185912177507D+08, -0.27184789986486D+07,&
                    0.41585984853835D+07, -0.20647204604053D+06,  0.18144833289046D+07,  0.34326419503236D+06,&
                    0.90134355593812D+06,  0.35850336089510D+06,  0.46926515456568D+06,  0.24625219853177D+06,&
                    0.22718988161213D+06,  0.11371191859628D+06,  0.71093013209460D+05,  0.14050000486230D+05,&
                   -0.34766169899283D+03, -0.94565358064906D+03,  0.15727281370514D+05,  0.17409341606411D+05,&
                    0.23127945278792D+04, -0.69887007302416D+04,  0.34887894668076D+04,  0.44142782323043D+04,&
                   -0.15464704885653D+04, -0.22278617146356D+03,  0.60021247393140D+03,  0.13653545323929D+03,&
                    0.82818712942385D+01,  0.19124857205986D+00,  0.25450668026110D-02, -0.51831767792976D-03,&
                    0.81197164556457D-04/)
  case(100)
    nfun = 35
    alf(1:nfun)=(/  0.74850931724558D+08,  0.45364201045187D+08,  0.27493455178901D+08,  0.16662700108425D+08,&
                    0.10098606126318D+08,  0.61203673492837D+07,  0.37093135450204D+07,  0.22480688151639D+07,&
                    0.13624659485842D+07,  0.82573693853586D+06,  0.50044662941568D+06,  0.30330098752465D+06,&
                    0.18381878031797D+06,  0.11140532140483D+06,  0.67518376608988D+05,  0.40920228247872D+05,&
                    0.24800138332043D+05,  0.15030386867905D+05,  0.91093253744880D+04,  0.55208032572654D+04,&
                    0.33459413680397D+04,  0.20278432533574D+04,  0.12289959111257D+04,  0.74484600674283D+03,&
                    0.45142182226838D+03,  0.27358898319296D+03,  0.16581150496543D+03,  0.10049182119117D+03,&
                    0.60904134055255D+02,  0.36911596397124D+02,  0.22370664483105D+02,  0.13557978474609D+02,&
                    0.82169566512784D+01,  0.49799737280475D+01,  0.30181658957864D+01/)
    coe(1:nfun)=(/  0.37973256301178D+08, -0.51131359478760D+08,  0.60315527106506D+08, -0.35917405770447D+08,&
                    0.29711158630981D+08, -0.12815309675919D+08,  0.12179915605713D+08, -0.36022506279678D+07,&
                    0.48763570249176D+07, -0.44562583075714D+06,  0.20725888906431D+07,  0.29078555003548D+06,&
                    0.99813420449829D+06,  0.35252678748870D+06,  0.50490584981263D+06,  0.24673748380375D+06,&
                    0.23777074660763D+06,  0.11178744256711D+06,  0.71822988901168D+05,  0.11651341526292D+05,&
                   -0.21223682202771D+03, -0.26348833859526D+03,  0.17311297029583D+05,  0.17423183248197D+05,&
                    0.13900431912357D+04, -0.70392196611261D+04,  0.42149782694437D+04,  0.41740683919152D+04,&
                   -0.17038257726856D+04, -0.10610529084427D+03,  0.60961208666819D+03,  0.12389780159024D+03,&
                    0.68732674558250D+01,  0.13117061914272D+00,  0.21421098675773D-02/)
  case(101)
    nfun = 35
    alf(1:nfun)=(/  0.74850931724558D+08,  0.45364201045187D+08,  0.27493455178901D+08,  0.16662700108425D+08,&
                    0.10098606126318D+08,  0.61203673492837D+07,  0.37093135450204D+07,  0.22480688151639D+07,&
                    0.13624659485842D+07,  0.82573693853586D+06,  0.50044662941568D+06,  0.30330098752465D+06,&
                    0.18381878031797D+06,  0.11140532140483D+06,  0.67518376608988D+05,  0.40920228247872D+05,&
                    0.24800138332043D+05,  0.15030386867905D+05,  0.91093253744880D+04,  0.55208032572654D+04,&
                    0.33459413680397D+04,  0.20278432533574D+04,  0.12289959111257D+04,  0.74484600674283D+03,&
                    0.45142182226838D+03,  0.27358898319296D+03,  0.16581150496543D+03,  0.10049182119117D+03,&
                    0.60904134055255D+02,  0.36911596397124D+02,  0.22370664483105D+02,  0.13557978474609D+02,&
                    0.82169566512784D+01,  0.49799737280475D+01,  0.30181658957864D+01/)
    coe(1:nfun)=(/  0.33978500291718D+08, -0.50719602537781D+08,  0.64043448110046D+08, -0.39115585100586D+08,&
                    0.32857784995209D+08, -0.14396502818405D+08,  0.13577748938873D+08, -0.41281267363586D+07,&
                    0.54120618288345D+07, -0.57014060321999D+06,  0.22750800449047D+07,  0.27504646762896D+06,&
                    0.10783380911007D+07,  0.35782449955988D+06,  0.53540798661888D+06,  0.25044601950073D+06,&
                    0.24647013641429D+06,  0.11066092407595D+06,  0.71842777151197D+05,  0.95574438705146D+04,&
                   -0.16314801665395D+03,  0.64252220418118D+03,  0.18843280493940D+05,  0.17365454694769D+05,&
                    0.41505988993635D+03, -0.69854272593067D+04,  0.49254580436832D+04,  0.38917821447799D+04,&
                   -0.18381259606507D+04,  0.13718948781418D+02,  0.61397172631823D+03,  0.11163763026553D+03,&
                    0.57150989486036D+01,  0.78965031787902D-01,  0.36317758895166D-02/)
  case(102)
    nfun = 35
    alf(1:nfun)=(/  0.74850931724558D+08,  0.45364201045187D+08,  0.27493455178901D+08,  0.16662700108425D+08,&
                    0.10098606126318D+08,  0.61203673492837D+07,  0.37093135450204D+07,  0.22480688151639D+07,&
                    0.13624659485842D+07,  0.82573693853586D+06,  0.50044662941568D+06,  0.30330098752465D+06,&
                    0.18381878031797D+06,  0.11140532140483D+06,  0.67518376608988D+05,  0.40920228247872D+05,&
                    0.24800138332043D+05,  0.15030386867905D+05,  0.91093253744880D+04,  0.55208032572654D+04,&
                    0.33459413680397D+04,  0.20278432533574D+04,  0.12289959111257D+04,  0.74484600674283D+03,&
                    0.45142182226838D+03,  0.27358898319296D+03,  0.16581150496543D+03,  0.10049182119117D+03,&
                    0.60904134055255D+02,  0.36911596397124D+02,  0.22370664483105D+02,  0.13557978474609D+02,&
                    0.82169566512784D+01,  0.49799737280475D+01,  0.30181658957864D+01/)
    coe(1:nfun)=(/  0.26725823121460D+08, -0.45179243890503D+08,  0.63059335915649D+08, -0.38724499976562D+08,&
                    0.33718209539673D+08, -0.14550021319016D+08,  0.14187771417969D+08, -0.41832303378677D+07,&
                    0.57008565288544D+07, -0.54291927205276D+06,  0.24020330542164D+07,  0.30671990808916D+06,&
                    0.11363173201671D+07,  0.37743114880753D+06,  0.55904193689084D+06,  0.25830890107459D+06,&
                    0.25271432267213D+06,  0.11063345849772D+06,  0.70986817571133D+05,  0.78698956124336D+04,&
                   -0.23445955670625D+03,  0.17936106467079D+04,  0.20302699854729D+05,  0.17234503030703D+05,&
                   -0.60304324307758D+03, -0.68243815934777D+04,  0.56114878134424D+04,  0.35706900120584D+04,&
                   -0.19480508601102D+04,  0.13522481601893D+03,  0.61371835871782D+03,  0.99935827767229D+02,&
                    0.47269672308676D+01,  0.28838353731544D-01,  0.43391753649757D-02/)
  case(103)
    nfun = 35
    alf(1:nfun)=(/  0.74850931724558D+08,  0.45364201045187D+08,  0.27493455178901D+08,  0.16662700108425D+08,&
                    0.10098606126318D+08,  0.61203673492837D+07,  0.37093135450204D+07,  0.22480688151639D+07,&
                    0.13624659485842D+07,  0.82573693853586D+06,  0.50044662941568D+06,  0.30330098752465D+06,&
                    0.18381878031797D+06,  0.11140532140483D+06,  0.67518376608988D+05,  0.40920228247872D+05,&
                    0.24800138332043D+05,  0.15030386867905D+05,  0.91093253744880D+04,  0.55208032572654D+04,&
                    0.33459413680397D+04,  0.20278432533574D+04,  0.12289959111257D+04,  0.74484600674283D+03,&
                    0.45142182226838D+03,  0.27358898319296D+03,  0.16581150496543D+03,  0.10049182119117D+03,&
                    0.60904134055255D+02,  0.36911596397124D+02,  0.22370664483105D+02,  0.13557978474609D+02,&
                    0.82169566512784D+01,  0.49799737280475D+01,  0.30181658957864D+01/)
    coe(1:nfun)=(/  0.20131497487000D+08, -0.38430757187622D+08,  0.60248338640686D+08, -0.36589377910400D+08,&
                    0.33441301691864D+08, -0.13945453582214D+08,  0.14384805962845D+08, -0.39714922371521D+07,&
                    0.58590098101845D+07, -0.43099215255165D+06,  0.24916967571907D+07,  0.36591023586512D+06,&
                    0.11817499200153D+07,  0.40680455845833D+06,  0.57807027523494D+06,  0.26893591651559D+06,&
                    0.25738667003131D+06,  0.11108980090770D+06,  0.69643635906488D+05,  0.63766010297984D+04,&
                   -0.29243503176421D+03,  0.31242689048406D+04,  0.21709479314498D+05,  0.17010993343732D+05,&
                   -0.16462288761979D+04, -0.65581065221077D+04,  0.62657421183104D+04,  0.32150250754112D+04,&
                   -0.20325257661397D+04,  0.25710470212247D+03,  0.60858515985510D+03,  0.88860877838331D+02,&
                    0.39538635946064D+01, -0.71142016245176D-02,  0.58671979621749D-02/)
end select

return
end

!-----------------------------------------------------------------------------------------------------------------------------------
!---  EDF data library: ncore = 68, iz = 72- 86
!-----------------------------------------------------------------------------------------------------------------------------------
subroutine EDFLIB68(iz,nfun,alf,coe)
implicit real(kind=8) (a-h,o-z)
real(kind=8) :: alf(*),coe(*)

select case(iz)
  case(072)
    nfun = 36
    alf(1:nfun)=(/  0.10098606126318D+08,  0.61203673492837D+07,  0.37093135450204D+07,  0.22480688151639D+07,&
                    0.13624659485842D+07,  0.82573693853586D+06,  0.50044662941568D+06,  0.30330098752465D+06,&
                    0.18381878031797D+06,  0.11140532140483D+06,  0.67518376608988D+05,  0.40920228247872D+05,&
                    0.24800138332043D+05,  0.15030386867905D+05,  0.91093253744880D+04,  0.55208032572654D+04,&
                    0.33459413680397D+04,  0.20278432533574D+04,  0.12289959111257D+04,  0.74484600674283D+03,&
                    0.45142182226838D+03,  0.27358898319296D+03,  0.16581150496543D+03,  0.10049182119117D+03,&
                    0.60904134055255D+02,  0.36911596397124D+02,  0.22370664483105D+02,  0.13557978474609D+02,&
                    0.82169566512784D+01,  0.49799737280475D+01,  0.30181658957864D+01,  0.18291914519917D+01,&
                    0.11086008799950D+01,  0.67187932120909D+00,  0.40719958861157D+00,  0.24678762946156D+00/)
    coe(1:nfun)=(/  0.10849491830646D+08, -0.17550728293743D+08,  0.17462369864117D+08, -0.12775883503175D+08,&
                    0.86784829464293D+07, -0.50396252694049D+07,  0.31662066443536D+07, -0.16125760039773D+07,&
                    0.10840315400397D+07, -0.44903407227933D+06,  0.39271013731098D+06, -0.90334022816792D+05,&
                    0.16390225703722D+06,  0.69953864809647D+04,  0.75627857370853D+05,  0.15662522119790D+05,&
                    0.25045902952498D+05, -0.16815225013415D+04,  0.69252833313774D+03, -0.11366908953876D+04,&
                    0.62684362846573D+04,  0.39836584579877D+04, -0.63950528460692D+03, -0.15140547919597D+04,&
                    0.16435285390452D+04,  0.60290790118113D+03, -0.38759068679273D+03,  0.38427239203978D+02,&
                    0.10357576101823D+03,  0.15315473766292D+02,  0.75587350911042D+00,  0.29727289992665D+01,&
                    0.15793959443125D+01,  0.24221202787976D+00,  0.15707110427517D-01,  0.23264724043504D-03/)
  case(073)
    nfun = 37
    alf(1:nfun)=(/  0.10098606126318D+08,  0.61203673492837D+07,  0.37093135450204D+07,  0.22480688151639D+07,&
                    0.13624659485842D+07,  0.82573693853586D+06,  0.50044662941568D+06,  0.30330098752465D+06,&
                    0.18381878031797D+06,  0.11140532140483D+06,  0.67518376608988D+05,  0.40920228247872D+05,&
                    0.24800138332043D+05,  0.15030386867905D+05,  0.91093253744880D+04,  0.55208032572654D+04,&
                    0.33459413680397D+04,  0.20278432533574D+04,  0.12289959111257D+04,  0.74484600674283D+03,&
                    0.45142182226838D+03,  0.27358898319296D+03,  0.16581150496543D+03,  0.10049182119117D+03,&
                    0.60904134055255D+02,  0.36911596397124D+02,  0.22370664483105D+02,  0.13557978474609D+02,&
                    0.82169566512784D+01,  0.49799737280475D+01,  0.30181658957864D+01,  0.18291914519917D+01,&
                    0.11086008799950D+01,  0.67187932120909D+00,  0.40719958861157D+00,  0.24678762946156D+00,&
                    0.14956826027973D+00/)
    coe(1:nfun)=(/  0.96781889124274D+07, -0.16596767569809D+08,  0.17003036107986D+08, -0.12586970449180D+08,&
                    0.86439048153508D+07, -0.50190704882476D+07,  0.31801221916720D+07, -0.16066369319714D+07,&
                    0.10952161262560D+07, -0.44372952762966D+06,  0.39976885615879D+06, -0.86033027360067D+05,&
                    0.16806743037410D+06,  0.95313258911800D+04,  0.77286788716404D+05,  0.16087388789847D+05,&
                    0.24759038592651D+05, -0.22240862047484D+04,  0.56215002706554D+03, -0.71911716696853D+03,&
                    0.67635400474936D+04,  0.38643034430204D+04, -0.97915174375170D+03, -0.13778384867657D+04,&
                    0.18329349590227D+04,  0.47898205066096D+03, -0.40220748290319D+03,  0.68644569616110D+02,&
                    0.10393755227251D+03,  0.12126582272892D+02,  0.97161163842145D+00,  0.33586825315593D+01,&
                    0.14347295312035D+01,  0.17951671179161D+00,  0.11619923189105D-01, -0.26213376384110D-03,&
                    0.44725340737628D-04/)
  case(074)
    nfun = 37
    alf(1:nfun)=(/  0.10098606126318D+08,  0.61203673492837D+07,  0.37093135450204D+07,  0.22480688151639D+07,&
                    0.13624659485842D+07,  0.82573693853586D+06,  0.50044662941568D+06,  0.30330098752465D+06,&
                    0.18381878031797D+06,  0.11140532140483D+06,  0.67518376608988D+05,  0.40920228247872D+05,&
                    0.24800138332043D+05,  0.15030386867905D+05,  0.91093253744880D+04,  0.55208032572654D+04,&
                    0.33459413680397D+04,  0.20278432533574D+04,  0.12289959111257D+04,  0.74484600674283D+03,&
                    0.45142182226838D+03,  0.27358898319296D+03,  0.16581150496543D+03,  0.10049182119117D+03,&
                    0.60904134055255D+02,  0.36911596397124D+02,  0.22370664483105D+02,  0.13557978474609D+02,&
                    0.82169566512784D+01,  0.49799737280475D+01,  0.30181658957864D+01,  0.18291914519917D+01,&
                    0.11086008799950D+01,  0.67187932120909D+00,  0.40719958861157D+00,  0.24678762946156D+00,&
                    0.14956826027973D+00/)
    coe(1:nfun)=(/  0.90743669701345D+07, -0.15435353369869D+08,  0.15900554409063D+08, -0.11714787020928D+08,&
                    0.80988324886401D+07, -0.46616888144028D+07,  0.29914850362731D+07, -0.14812521615250D+07,&
                    0.10392731934069D+07, -0.39985469760409D+06,  0.38580778167176D+06, -0.69623327736259D+05,&
                    0.16574411778710D+06,  0.15783857804615D+05,  0.76881961281393D+05,  0.17586570161250D+05,&
                    0.23786657889661D+05, -0.24239766117476D+04,  0.27437325088563D+03, -0.14272728173813D+03,&
                    0.71882686637116D+04,  0.37307902281375D+04, -0.13349457466424D+04, -0.11906419060441D+04,&
                    0.19990516030103D+04,  0.34592775871751D+03, -0.40762076770182D+03,  0.99150621157666D+02,&
                    0.10280856730932D+03,  0.88263619076141D+01,  0.14770478603317D+01,  0.36669575448585D+01,&
                    0.12518955192337D+01,  0.13624476452711D+00,  0.80630620452560D-02, -0.24018369195122D-03,&
                    0.40740168131270D-04/)
  case(075)
    nfun = 37
    alf(1:nfun)=(/  0.10098606126318D+08,  0.61203673492837D+07,  0.37093135450204D+07,  0.22480688151639D+07,&
                    0.13624659485842D+07,  0.82573693853586D+06,  0.50044662941568D+06,  0.30330098752465D+06,&
                    0.18381878031797D+06,  0.11140532140483D+06,  0.67518376608988D+05,  0.40920228247872D+05,&
                    0.24800138332043D+05,  0.15030386867905D+05,  0.91093253744880D+04,  0.55208032572654D+04,&
                    0.33459413680397D+04,  0.20278432533574D+04,  0.12289959111257D+04,  0.74484600674283D+03,&
                    0.45142182226838D+03,  0.27358898319296D+03,  0.16581150496543D+03,  0.10049182119117D+03,&
                    0.60904134055255D+02,  0.36911596397124D+02,  0.22370664483105D+02,  0.13557978474609D+02,&
                    0.82169566512784D+01,  0.49799737280475D+01,  0.30181658957864D+01,  0.18291914519917D+01,&
                    0.11086008799950D+01,  0.67187932120909D+00,  0.40719958861157D+00,  0.24678762946156D+00,&
                    0.14956826027973D+00/)
    coe(1:nfun)=(/  0.86133897677965D+07, -0.14060838821651D+08,  0.14423969786217D+08, -0.10482615917361D+08,&
                    0.72953798840206D+07, -0.41357426461467D+07,  0.27031955668206D+07, -0.12967008404683D+07,&
                    0.95069678696123D+06, -0.33685003223608D+06,  0.36174686635105D+06, -0.47195352097288D+05,&
                    0.16032718584247D+06,  0.23860560255691D+05,  0.75455767257298D+05,  0.19576035244347D+05,&
                    0.22449724043974D+05, -0.24535910372816D+04, -0.70714828808967D+02,  0.53902038059063D+03,&
                    0.75661005817676D+04,  0.35671740692139D+04, -0.16931674055057D+04, -0.95749191844151D+03,&
                    0.21397061492719D+04,  0.20586106406429D+03, -0.40357958453824D+03,  0.12932276380523D+03,&
                    0.10021229416702D+03,  0.55470379773362D+01,  0.22131030519430D+01,  0.38708642795601D+01,&
                    0.10761181011132D+01,  0.10165370331328D+00,  0.47546510025747D-02, -0.11956536790247D-03,&
                    0.23666539895162D-04/)
  case(076)
    nfun = 35
    alf(1:nfun)=(/  0.10098606126318D+08,  0.61203673492837D+07,  0.37093135450204D+07,  0.22480688151639D+07,&
                    0.13624659485842D+07,  0.82573693853586D+06,  0.50044662941568D+06,  0.30330098752465D+06,&
                    0.18381878031797D+06,  0.11140532140483D+06,  0.67518376608988D+05,  0.40920228247872D+05,&
                    0.24800138332043D+05,  0.15030386867905D+05,  0.91093253744880D+04,  0.55208032572654D+04,&
                    0.33459413680397D+04,  0.20278432533574D+04,  0.12289959111257D+04,  0.74484600674283D+03,&
                    0.45142182226838D+03,  0.27358898319296D+03,  0.16581150496543D+03,  0.10049182119117D+03,&
                    0.60904134055255D+02,  0.36911596397124D+02,  0.22370664483105D+02,  0.13557978474609D+02,&
                    0.82169566512784D+01,  0.49799737280475D+01,  0.30181658957864D+01,  0.18291914519917D+01,&
                    0.11086008799950D+01,  0.67187932120909D+00,  0.40719958861157D+00/)
    coe(1:nfun)=(/  0.94189675914249D+07, -0.13174139771420D+08,  0.12794137179562D+08, -0.88521706137390D+07,&
                    0.61419821366568D+07, -0.33604960138416D+07,  0.22607848070016D+07, -0.10191695301409D+07,&
                    0.80975715818274D+06, -0.24337350762045D+06,  0.32120481719226D+06, -0.15152569381028D+05,&
                    0.14980659395292D+06,  0.34874979042903D+05,  0.72383460368935D+05,  0.22393443377579D+05,&
                    0.20563187536804D+05, -0.22081019183090D+04, -0.52404969512019D+03,  0.13555793236180D+04,&
                    0.78750142275298D+04,  0.33828282331108D+04, -0.20540939034183D+04, -0.67690450784867D+03,&
                    0.22505801421984D+04,  0.61702674837223D+02, -0.39067249226384D+03,  0.15927444572203D+03,&
                    0.95898319249116D+02,  0.23798147683242D+01,  0.31704647955087D+01,  0.40068547170905D+01,&
                    0.90176507067525D+00,  0.70193625466985D-01,  0.23170927164600D-02/)
  case(077)
    nfun = 36
    alf(1:nfun)=(/  0.10098606126318D+08,  0.61203673492837D+07,  0.37093135450204D+07,  0.22480688151639D+07,&
                    0.13624659485842D+07,  0.82573693853586D+06,  0.50044662941568D+06,  0.30330098752465D+06,&
                    0.18381878031797D+06,  0.11140532140483D+06,  0.67518376608988D+05,  0.40920228247872D+05,&
                    0.24800138332043D+05,  0.15030386867905D+05,  0.91093253744880D+04,  0.55208032572654D+04,&
                    0.33459413680397D+04,  0.20278432533574D+04,  0.12289959111257D+04,  0.74484600674283D+03,&
                    0.45142182226838D+03,  0.27358898319296D+03,  0.16581150496543D+03,  0.10049182119117D+03,&
                    0.60904134055255D+02,  0.36911596397124D+02,  0.22370664483105D+02,  0.13557978474609D+02,&
                    0.82169566512784D+01,  0.49799737280475D+01,  0.30181658957864D+01,  0.18291914519917D+01,&
                    0.11086008799950D+01,  0.67187932120909D+00,  0.40719958861157D+00,  0.24678762946156D+00/)
    coe(1:nfun)=(/  0.62213234949365D+07, -0.88097406843243D+07,  0.90593598186121D+07, -0.61989052418728D+07,&
                    0.45038631655297D+07, -0.23558507172756D+07,  0.17063485589859D+07, -0.68345223357701D+06,&
                    0.63916309396422D+06, -0.13318682372364D+06,  0.27217903550521D+06,  0.21944025809176D+05,&
                    0.13675371708760D+06,  0.47375540341750D+05,  0.68464440828266D+05,  0.25587365018611D+05,&
                    0.18375055076212D+05, -0.18181281889344D+04, -0.10065761928471D+04,  0.22636475848209D+04,&
                    0.81307427120080D+04,  0.31640242781430D+04, -0.24023009329456D+04, -0.35442773348998D+03,&
                    0.23280065812669D+04, -0.83516007751618D+02, -0.36792886778510D+03,  0.18741117653918D+03,&
                    0.89916872188059D+02, -0.18547967115069D+00,  0.42676470532756D+01,  0.39498755051220D+01,&
                    0.74849772817117D+00,  0.52643311769792D-01,  0.10718178273734D-02,  0.53171859012657D-04/)
  case(078)
    nfun = 38
    alf(1:nfun)=(/  0.27493455178901D+08,  0.16662700108425D+08,  0.10098606126318D+08,  0.61203673492837D+07,&
                    0.37093135450204D+07,  0.22480688151639D+07,  0.13624659485842D+07,  0.82573693853586D+06,&
                    0.50044662941568D+06,  0.30330098752465D+06,  0.18381878031797D+06,  0.11140532140483D+06,&
                    0.67518376608988D+05,  0.40920228247872D+05,  0.24800138332043D+05,  0.15030386867905D+05,&
                    0.91093253744880D+04,  0.55208032572654D+04,  0.33459413680397D+04,  0.20278432533574D+04,&
                    0.12289959111257D+04,  0.74484600674283D+03,  0.45142182226838D+03,  0.27358898319296D+03,&
                    0.16581150496543D+03,  0.10049182119117D+03,  0.60904134055255D+02,  0.36911596397124D+02,&
                    0.22370664483105D+02,  0.13557978474609D+02,  0.82169566512784D+01,  0.49799737280475D+01,&
                    0.30181658957864D+01,  0.18291914519917D+01,  0.11086008799950D+01,  0.67187932120909D+00,&
                    0.40719958861157D+00,  0.24678762946156D+00/)
    coe(1:nfun)=(/  0.88646963621902D+07, -0.13788490723965D+08,  0.14602818190388D+08, -0.10300893068617D+08,&
                    0.74343958814144D+07, -0.39955529405107D+07,  0.28138828710418D+07, -0.12036788204482D+07,&
                    0.10327677565181D+07, -0.26688374943316D+06,  0.42075324984580D+06,  0.57506032762527D+04,&
                    0.20743931101546D+06,  0.68316911261268D+05,  0.11875654454043D+06,  0.62716725096740D+05,&
                    0.62929839297852D+05,  0.29579342923852D+05,  0.15653082402523D+05, -0.11504640692081D+04,&
                   -0.15843065434220D+04,  0.32987402373281D+04,  0.83057288916760D+04,  0.29241602258242D+04,&
                   -0.27379877930512D+04,  0.10342925445708D+02,  0.23665483850044D+04, -0.22483870511776D+03,&
                   -0.33658832954473D+03,  0.21327867851123D+03,  0.82346853559627D+02, -0.18555538481448D+01,&
                    0.53025293867007D+01,  0.37241622388391D+01,  0.63953467925597D+00,  0.42960149890021D-01,&
                    0.79264472207186D-03,  0.45168369386506D-04/)
  case(079)
    nfun = 40
    alf(1:nfun)=(/  0.74850931724558D+08,  0.45364201045187D+08,  0.27493455178901D+08,  0.16662700108425D+08,&
                    0.10098606126318D+08,  0.61203673492837D+07,  0.37093135450204D+07,  0.22480688151639D+07,&
                    0.13624659485842D+07,  0.82573693853586D+06,  0.50044662941568D+06,  0.30330098752465D+06,&
                    0.18381878031797D+06,  0.11140532140483D+06,  0.67518376608988D+05,  0.40920228247872D+05,&
                    0.24800138332043D+05,  0.15030386867905D+05,  0.91093253744880D+04,  0.55208032572654D+04,&
                    0.33459413680397D+04,  0.20278432533574D+04,  0.12289959111257D+04,  0.74484600674283D+03,&
                    0.45142182226838D+03,  0.27358898319296D+03,  0.16581150496543D+03,  0.10049182119117D+03,&
                    0.60904134055255D+02,  0.36911596397124D+02,  0.22370664483105D+02,  0.13557978474609D+02,&
                    0.82169566512784D+01,  0.49799737280475D+01,  0.30181658957864D+01,  0.18291914519917D+01,&
                    0.11086008799950D+01,  0.67187932120909D+00,  0.40719958861157D+00,  0.24678762946156D+00/)
    coe(1:nfun)=(/  0.10254156551819D+07, -0.49271275138092D+06,  0.18814130939102D+07, -0.39395993682861D+06,&
                    0.12322274251518D+07,  0.15703696044922D+05,  0.66389086048508D+06,  0.20356251223755D+06,&
                    0.39535766182280D+06,  0.21517172609162D+06,  0.26160028995299D+06,  0.18944573143137D+06,&
                    0.18671338361079D+06,  0.15262651703212D+06,  0.13949421910049D+06,  0.11674219404370D+06,&
                    0.99918423899047D+05,  0.78587166887132D+05,  0.57065312896829D+05,  0.33647340712726D+05,&
                    0.12799859653397D+05, -0.42435743290861D+03, -0.21303546396482D+04,  0.43898060217945D+04,&
                    0.84364188828112D+04,  0.26428033432841D+04, -0.30457339224816D+04,  0.40656490953227D+03,&
                    0.23714809381499D+04, -0.36342071767017D+03, -0.29720762992027D+03,  0.23750664057598D+03,&
                    0.73307995051034D+02, -0.30469711204139D+01,  0.63338653679274D+01,  0.35104161564619D+01,&
                    0.53131506767303D+00,  0.30903826024570D-01,  0.55945931553580D-03,  0.26856007599552D-04/)
  case(080)
    nfun = 36
    alf(1:nfun)=(/  0.10098606126318D+08,  0.61203673492837D+07,  0.37093135450204D+07,  0.22480688151639D+07,&
                    0.13624659485842D+07,  0.82573693853586D+06,  0.50044662941568D+06,  0.30330098752465D+06,&
                    0.18381878031797D+06,  0.11140532140483D+06,  0.67518376608988D+05,  0.40920228247872D+05,&
                    0.24800138332043D+05,  0.15030386867905D+05,  0.91093253744880D+04,  0.55208032572654D+04,&
                    0.33459413680397D+04,  0.20278432533574D+04,  0.12289959111257D+04,  0.74484600674283D+03,&
                    0.45142182226838D+03,  0.27358898319296D+03,  0.16581150496543D+03,  0.10049182119117D+03,&
                    0.60904134055255D+02,  0.36911596397124D+02,  0.22370664483105D+02,  0.13557978474609D+02,&
                    0.82169566512784D+01,  0.49799737280475D+01,  0.30181658957864D+01,  0.18291914519917D+01,&
                    0.11086008799950D+01,  0.67187932120909D+00,  0.40719958861157D+00,  0.24678762946156D+00/)
    coe(1:nfun)=(/  0.10350335084816D+08, -0.18931326938621D+08,  0.20477906428938D+08, -0.15219852038025D+08,&
                    0.10759080310483D+08, -0.61329736711195D+07,  0.40256602848886D+07, -0.19415408332602D+07,&
                    0.14118117104177D+07, -0.51594427130643D+06,  0.52658298766842D+06, -0.87363693749517D+05,&
                    0.22153254262338D+06,  0.16593158708563D+05,  0.94340139991510D+05,  0.13675807285802D+05,&
                    0.23194702121274D+05, -0.70600406778082D+04,  0.14853744420138D+04,  0.32347330294698D+04,&
                    0.98053367548485D+04,  0.15931319205636D+04, -0.29011810403574D+04,  0.58713515257592D+03,&
                    0.24797339679581D+04, -0.57079478974171D+03, -0.21563617459009D+03,  0.24735507528552D+03,&
                    0.65119846125245D+02, -0.30852125681083D+01,  0.66784696799803D+01,  0.36099807705256D+01,&
                    0.32696145157531D+00,  0.38608189498660D-01, -0.28691028271446D-02,  0.32903654857663D-03/)
  case(081)
    nfun = 36
    alf(1:nfun)=(/  0.16662700108425D+08,  0.10098606126318D+08,  0.61203673492837D+07,  0.37093135450204D+07,&
                    0.22480688151639D+07,  0.13624659485842D+07,  0.82573693853586D+06,  0.50044662941568D+06,&
                    0.30330098752465D+06,  0.18381878031797D+06,  0.11140532140483D+06,  0.67518376608988D+05,&
                    0.40920228247872D+05,  0.24800138332043D+05,  0.15030386867905D+05,  0.91093253744880D+04,&
                    0.55208032572654D+04,  0.33459413680397D+04,  0.20278432533574D+04,  0.12289959111257D+04,&
                    0.74484600674283D+03,  0.45142182226838D+03,  0.27358898319296D+03,  0.16581150496543D+03,&
                    0.10049182119117D+03,  0.60904134055255D+02,  0.36911596397124D+02,  0.22370664483105D+02,&
                    0.13557978474609D+02,  0.82169566512784D+01,  0.49799737280475D+01,  0.30181658957864D+01,&
                    0.18291914519917D+01,  0.11086008799950D+01,  0.67187932120909D+00,  0.40719958861157D+00/)
    coe(1:nfun)=(/  0.91321602179470D+07, -0.13439862671307D+08,  0.14251286760674D+08, -0.98639115641594D+07,&
                    0.72264271274300D+07, -0.37861338639364D+07,  0.27388243316417D+07, -0.11141710810950D+07,&
                    0.10194010484095D+07, -0.23295994426465D+06,  0.42289866221684D+06,  0.18591695607841D+05,&
                    0.20665406887808D+06,  0.66811483249985D+05,  0.10798439778086D+06,  0.46477044043023D+05,&
                    0.40805477606196D+05,  0.73743015606282D+04,  0.85394332950981D+03, -0.29066430266225D+04,&
                    0.66108292219033D+04,  0.86078271415344D+04,  0.19271529426840D+04, -0.35329709112038D+04,&
                    0.12648458240964D+04,  0.22792615862339D+04, -0.62473975827665D+03, -0.19357672766324D+03,&
                    0.27770585641013D+03,  0.51427468731062D+02, -0.36070043155619D+01,  0.83828219855654D+01,&
                    0.30231805714263D+01,  0.31886063125978D+00,  0.93907709909473D-02,  0.30667417885422D-03/)
  case(082)
    nfun = 36
    alf(1:nfun)=(/  0.16662700108425D+08,  0.10098606126318D+08,  0.61203673492837D+07,  0.37093135450204D+07,&
                    0.22480688151639D+07,  0.13624659485842D+07,  0.82573693853586D+06,  0.50044662941568D+06,&
                    0.30330098752465D+06,  0.18381878031797D+06,  0.11140532140483D+06,  0.67518376608988D+05,&
                    0.40920228247872D+05,  0.24800138332043D+05,  0.15030386867905D+05,  0.91093253744880D+04,&
                    0.55208032572654D+04,  0.33459413680397D+04,  0.20278432533574D+04,  0.12289959111257D+04,&
                    0.74484600674283D+03,  0.45142182226838D+03,  0.27358898319296D+03,  0.16581150496543D+03,&
                    0.10049182119117D+03,  0.60904134055255D+02,  0.36911596397124D+02,  0.22370664483105D+02,&
                    0.13557978474609D+02,  0.82169566512784D+01,  0.49799737280475D+01,  0.30181658957864D+01,&
                    0.18291914519917D+01,  0.11086008799950D+01,  0.67187932120909D+00,  0.40719958861157D+00/)
    coe(1:nfun)=(/  0.12190433299660D+08, -0.18879807294426D+08,  0.19836528411144D+08, -0.14058709099588D+08,&
                    0.10074542473249D+08, -0.54761044729748D+07,  0.37743428837676D+07, -0.16701162081037D+07,&
                    0.13632202263881D+07, -0.40095070139945D+06,  0.53635398228180D+06, -0.28748915825307D+05,&
                    0.24505113949886D+06,  0.54227224966757D+05,  0.12057806672823D+06,  0.42233734506372D+05,&
                    0.43628092676543D+05,  0.49478270964613D+04,  0.13429370060465D+04, -0.30910126831105D+04,&
                    0.77115482363546D+04,  0.86474168107182D+04,  0.14929731521933D+04, -0.36967733307884D+04,&
                    0.17144345222522D+04,  0.21808327013481D+04, -0.74197124705724D+03, -0.13043357439569D+03,&
                    0.29204499635398D+03,  0.39296528840647D+02, -0.27068097415349D+01,  0.92272174069010D+01,&
                    0.27155259387860D+01,  0.22841308421900D+00,  0.43053914726414D-02,  0.22433356380436D-03/)
  case(083)
    nfun = 36
    alf(1:nfun)=(/  0.16662700108425D+08,  0.10098606126318D+08,  0.61203673492837D+07,  0.37093135450204D+07,&
                    0.22480688151639D+07,  0.13624659485842D+07,  0.82573693853586D+06,  0.50044662941568D+06,&
                    0.30330098752465D+06,  0.18381878031797D+06,  0.11140532140483D+06,  0.67518376608988D+05,&
                    0.40920228247872D+05,  0.24800138332043D+05,  0.15030386867905D+05,  0.91093253744880D+04,&
                    0.55208032572654D+04,  0.33459413680397D+04,  0.20278432533574D+04,  0.12289959111257D+04,&
                    0.74484600674283D+03,  0.45142182226838D+03,  0.27358898319296D+03,  0.16581150496543D+03,&
                    0.10049182119117D+03,  0.60904134055255D+02,  0.36911596397124D+02,  0.22370664483105D+02,&
                    0.13557978474609D+02,  0.82169566512784D+01,  0.49799737280475D+01,  0.30181658957864D+01,&
                    0.18291914519917D+01,  0.11086008799950D+01,  0.67187932120909D+00,  0.40719958861157D+00/)
    coe(1:nfun)=(/  0.15011228528229D+08, -0.23267666559448D+08,  0.24169565340096D+08, -0.17199144567059D+08,&
                    0.12202637302737D+08, -0.67101458246126D+07,  0.45430957684727D+07, -0.20696001793277D+07,&
                    0.16205739147264D+07, -0.51891049126208D+06,  0.62303305567235D+06, -0.60423134676158D+05,&
                    0.27521357848059D+06,  0.46447805566885D+05,  0.13054286441958D+06,  0.39368910952331D+05,&
                    0.45554081185453D+05,  0.29406008257698D+04,  0.16096613803194D+04, -0.30602755709703D+04,&
                    0.87545830262955D+04,  0.86761481782477D+04,  0.99733648432698D+03, -0.37913682230928D+04,&
                    0.21646021793297D+04,  0.20504276531123D+04, -0.84830400762444D+03, -0.60280131045300D+02,&
                    0.30161974342369D+03,  0.26922184297543D+02, -0.10308280417446D+01,  0.98976173982028D+01,&
                    0.23663499736687D+01,  0.15478843433135D+00,  0.19527905692058D-02,  0.15794219627807D-03/)
  case(084)
    nfun = 36
    alf(1:nfun)=(/  0.16662700108425D+08,  0.10098606126318D+08,  0.61203673492837D+07,  0.37093135450204D+07,&
                    0.22480688151639D+07,  0.13624659485842D+07,  0.82573693853586D+06,  0.50044662941568D+06,&
                    0.30330098752465D+06,  0.18381878031797D+06,  0.11140532140483D+06,  0.67518376608988D+05,&
                    0.40920228247872D+05,  0.24800138332043D+05,  0.15030386867905D+05,  0.91093253744880D+04,&
                    0.55208032572654D+04,  0.33459413680397D+04,  0.20278432533574D+04,  0.12289959111257D+04,&
                    0.74484600674283D+03,  0.45142182226838D+03,  0.27358898319296D+03,  0.16581150496543D+03,&
                    0.10049182119117D+03,  0.60904134055255D+02,  0.36911596397124D+02,  0.22370664483105D+02,&
                    0.13557978474609D+02,  0.82169566512784D+01,  0.49799737280475D+01,  0.30181658957864D+01,&
                    0.18291914519917D+01,  0.11086008799950D+01,  0.67187932120909D+00,  0.40719958861157D+00/)
    coe(1:nfun)=(/  0.16397997333303D+08, -0.25257067106773D+08,  0.26203601457033D+08, -0.18613431295637D+08,&
                    0.13207546633096D+08, -0.72559830913258D+07,  0.49135187833433D+07, -0.22387431451924D+07,&
                    0.17503723839532D+07, -0.56395360967988D+06,  0.67028162063771D+06, -0.69377490717441D+05,&
                    0.29321292319959D+06,  0.45660525936544D+05,  0.13664704690173D+06,  0.38554078935241D+05,&
                    0.46210661458674D+05,  0.15576947384281D+04,  0.15530368127106D+04, -0.27564249173177D+04,&
                    0.97059309217421D+04,  0.87034970324940D+04,  0.43807683950773D+03, -0.38087398370102D+04,&
                    0.26069675196153D+04,  0.18900617146104D+04, -0.94205821776703D+03,  0.16155061586525D+02,&
                    0.30581117548401D+03,  0.14832680165470D+02,  0.13324518788866D+01,  0.10355526861373D+02,&
                    0.19885111216730D+01,  0.10142862569458D+00,  0.74903936356918D-03,  0.13186047254354D-03/)
  case(085)
    nfun = 36
    alf(1:nfun)=(/  0.16662700108425D+08,  0.10098606126318D+08,  0.61203673492837D+07,  0.37093135450204D+07,&
                    0.22480688151639D+07,  0.13624659485842D+07,  0.82573693853586D+06,  0.50044662941568D+06,&
                    0.30330098752465D+06,  0.18381878031797D+06,  0.11140532140483D+06,  0.67518376608988D+05,&
                    0.40920228247872D+05,  0.24800138332043D+05,  0.15030386867905D+05,  0.91093253744880D+04,&
                    0.55208032572654D+04,  0.33459413680397D+04,  0.20278432533574D+04,  0.12289959111257D+04,&
                    0.74484600674283D+03,  0.45142182226838D+03,  0.27358898319296D+03,  0.16581150496543D+03,&
                    0.10049182119117D+03,  0.60904134055255D+02,  0.36911596397124D+02,  0.22370664483105D+02,&
                    0.13557978474609D+02,  0.82169566512784D+01,  0.49799737280475D+01,  0.30181658957864D+01,&
                    0.18291914519917D+01,  0.11086008799950D+01,  0.67187932120909D+00,  0.40719958861157D+00/)
    coe(1:nfun)=(/  0.17165787781437D+08, -0.26019361987732D+08,  0.26998635643898D+08, -0.19061880661526D+08,&
                    0.13576878983826D+08, -0.74062136214838D+07,  0.50559378992743D+07, -0.22743250287466D+07,&
                    0.18070341889242D+07, -0.56623694967276D+06,  0.69484777332145D+06, -0.64905577979803D+05,&
                    0.30424842961125D+06,  0.48939005084760D+05,  0.14051611660393D+06,  0.38872960441047D+05,&
                    0.46099725064538D+05,  0.52901908530854D+03,  0.13275587049611D+04, -0.22568386299978D+04,&
                    0.10600892759498D+05,  0.87039602400319D+04, -0.16866151521486D+03, -0.37502824610035D+04,&
                    0.30373059733688D+04,  0.17000155372357D+04, -0.10204183501841D+04,  0.97208091649536D+02,&
                    0.30454709621640D+03,  0.33678649749527D+01,  0.43384204781808D+01,  0.10539515672278D+02,&
                    0.16120495707504D+01,  0.65208424570727D-01,  0.16029725646405D-03,  0.11629173989203D-03/)
  case(086)
    nfun = 36
    alf(1:nfun)=(/  0.16662700108425D+08,  0.10098606126318D+08,  0.61203673492837D+07,  0.37093135450204D+07,&
                    0.22480688151639D+07,  0.13624659485842D+07,  0.82573693853586D+06,  0.50044662941568D+06,&
                    0.30330098752465D+06,  0.18381878031797D+06,  0.11140532140483D+06,  0.67518376608988D+05,&
                    0.40920228247872D+05,  0.24800138332043D+05,  0.15030386867905D+05,  0.91093253744880D+04,&
                    0.55208032572654D+04,  0.33459413680397D+04,  0.20278432533574D+04,  0.12289959111257D+04,&
                    0.74484600674283D+03,  0.45142182226838D+03,  0.27358898319296D+03,  0.16581150496543D+03,&
                    0.10049182119117D+03,  0.60904134055255D+02,  0.36911596397124D+02,  0.22370664483105D+02,&
                    0.13557978474609D+02,  0.82169566512784D+01,  0.49799737280475D+01,  0.30181658957864D+01,&
                    0.18291914519917D+01,  0.11086008799950D+01,  0.67187932120909D+00,  0.40719958861157D+00/)
    coe(1:nfun)=(/  0.16950294586164D+08, -0.25172029687250D+08,  0.26268658251820D+08, -0.18366055820993D+08,&
                    0.13202515847065D+08, -0.70981784236741D+07,  0.49348051432405D+07, -0.21566396982660D+07,&
                    0.17796391010344D+07, -0.51969274056411D+06,  0.69333947723010D+06, -0.45121376759619D+05,&
                    0.30723840379894D+06,  0.56874000897259D+05,  0.14179751103167D+06,  0.40510475231580D+05,&
                    0.45116201812484D+05, -0.81555257462896D+02,  0.90828468511486D+03, -0.15438691136399D+04,&
                    0.11424268317708D+05,  0.86780988945916D+04, -0.81868483355874D+03, -0.36122779079631D+04,&
                    0.34484434587441D+04,  0.14828044113354D+04, -0.10818443363107D+04,  0.18158394123679D+03,&
                    0.29774072869788D+03, -0.70781997972590D+01,  0.78363837489926D+01,  0.10449541431766D+02,&
                    0.12596637571640D+01,  0.40884307609630D-01, -0.44390892380619D-03,  0.98797745832435D-04/)
end select

return
end

!-----------------------------------------------------------------------------------------------------------------------------------
!---  EDF data library: ncore = 78, iz = 79-118
!-----------------------------------------------------------------------------------------------------------------------------------
subroutine EDFLIB78(iz,nfun,alf,coe)
implicit real(kind=8) (a-h,o-z)
real(kind=8) :: alf(*),coe(*)

select case(iz)
  case(079)
    nfun = 39
    alf(1:nfun)=(/  0.27493455178901D+08,  0.16662700108425D+08,  0.10098606126318D+08,  0.61203673492837D+07,&
                    0.37093135450204D+07,  0.22480688151639D+07,  0.13624659485842D+07,  0.82573693853586D+06,&
                    0.50044662941568D+06,  0.30330098752465D+06,  0.18381878031797D+06,  0.11140532140483D+06,&
                    0.67518376608988D+05,  0.40920228247872D+05,  0.24800138332043D+05,  0.15030386867905D+05,&
                    0.91093253744880D+04,  0.55208032572654D+04,  0.33459413680397D+04,  0.20278432533574D+04,&
                    0.12289959111257D+04,  0.74484600674283D+03,  0.45142182226838D+03,  0.27358898319296D+03,&
                    0.16581150496543D+03,  0.10049182119117D+03,  0.60904134055255D+02,  0.36911596397124D+02,&
                    0.22370664483105D+02,  0.13557978474609D+02,  0.82169566512784D+01,  0.49799737280475D+01,&
                    0.30181658957864D+01,  0.18291914519917D+01,  0.11086008799950D+01,  0.67187932120909D+00,&
                    0.40719958861157D+00,  0.24678762946156D+00,  0.14956826027973D+00/)
    coe(1:nfun)=(/  0.41058051472626D+07, -0.41572755796318D+07,  0.48104573323021D+07, -0.26321721750679D+07,&
                    0.23865894236526D+07, -0.84060783538628D+06,  0.10039969813004D+07, -0.13184789190674D+06,&
                    0.45703523727393D+06,  0.80149294602990D+05,  0.24759894973791D+06,  0.11878489443463D+06,&
                    0.15828073377758D+06,  0.10632086512960D+06,  0.10569693640385D+06,  0.75383949687898D+05,&
                    0.58840709657183D+05,  0.32663650684208D+05,  0.13345268630256D+05, -0.72569739234960D+03,&
                   -0.19623696824026D+04,  0.42969070160298D+04,  0.84817893526038D+04,  0.26001963794222D+04,&
                   -0.30382184081896D+04,  0.41911322937057D+03,  0.24030237964187D+04, -0.38705302216842D+03,&
                   -0.30978258786949D+03,  0.24789365603206D+03,  0.82210545603294D+02, -0.80293112549757D+01,&
                    0.31853511643960D+01,  0.41215939698476D+01,  0.14658447849928D+01,  0.44290592099358D+00,&
                    0.11176438174817D+00,  0.18891152915081D-01,  0.17376539876686D-02/)
  case(080)
    nfun = 37
    alf(1:nfun)=(/  0.10098606126318D+08,  0.61203673492837D+07,  0.37093135450204D+07,  0.22480688151639D+07,&
                    0.13624659485842D+07,  0.82573693853586D+06,  0.50044662941568D+06,  0.30330098752465D+06,&
                    0.18381878031797D+06,  0.11140532140483D+06,  0.67518376608988D+05,  0.40920228247872D+05,&
                    0.24800138332043D+05,  0.15030386867905D+05,  0.91093253744880D+04,  0.55208032572654D+04,&
                    0.33459413680397D+04,  0.20278432533574D+04,  0.12289959111257D+04,  0.74484600674283D+03,&
                    0.45142182226838D+03,  0.27358898319296D+03,  0.16581150496543D+03,  0.10049182119117D+03,&
                    0.60904134055255D+02,  0.36911596397124D+02,  0.22370664483105D+02,  0.13557978474609D+02,&
                    0.82169566512784D+01,  0.49799737280475D+01,  0.30181658957864D+01,  0.18291914519917D+01,&
                    0.11086008799950D+01,  0.67187932120909D+00,  0.40719958861157D+00,  0.24678762946156D+00,&
                    0.90647430472564D-01/)
    coe(1:nfun)=(/  0.11069773225557D+08, -0.19360663848068D+08,  0.20592571321478D+08, -0.15171931148400D+08,&
                    0.10683853233201D+08, -0.60708615249193D+07,  0.39845292656361D+07, -0.19163211153635D+07,&
                    0.13971116198113D+07, -0.50754209788404D+06,  0.52185378075903D+06, -0.84716127027666D+05,&
                    0.22005752697237D+06,  0.17414002357901D+05,  0.93883961640516D+05,  0.13929687622607D+05,&
                    0.23053855804062D+05, -0.69804679675148D+04,  0.14420848188903D+04,  0.32590892661380D+04,&
                    0.97834743298942D+04,  0.15841136835200D+04, -0.29144363948589D+04,  0.61811623417871D+03,&
                    0.25070592234762D+04, -0.59749265049936D+03, -0.23053894741913D+03,  0.26262799901485D+03,&
                    0.73811526476301D+02, -0.98276016336849D+01,  0.34433918445391D+01,  0.47429434879606D+01,&
                    0.13972965170470D+01,  0.45204335234156D+00,  0.76759521442952D-01,  0.11650828326430D-01,&
                    0.44143484288238D-04/)
  case(081)
    nfun = 38
    alf(1:nfun)=(/  0.16662700108425D+08,  0.10098606126318D+08,  0.61203673492837D+07,  0.37093135450204D+07,&
                    0.22480688151639D+07,  0.13624659485842D+07,  0.82573693853586D+06,  0.50044662941568D+06,&
                    0.30330098752465D+06,  0.18381878031797D+06,  0.11140532140483D+06,  0.67518376608988D+05,&
                    0.40920228247872D+05,  0.24800138332043D+05,  0.15030386867905D+05,  0.91093253744880D+04,&
                    0.55208032572654D+04,  0.33459413680397D+04,  0.20278432533574D+04,  0.12289959111257D+04,&
                    0.74484600674283D+03,  0.45142182226838D+03,  0.27358898319296D+03,  0.16581150496543D+03,&
                    0.10049182119117D+03,  0.60904134055255D+02,  0.36911596397124D+02,  0.22370664483105D+02,&
                    0.13557978474609D+02,  0.82169566512784D+01,  0.49799737280475D+01,  0.30181658957864D+01,&
                    0.18291914519917D+01,  0.11086008799950D+01,  0.67187932120909D+00,  0.40719958861157D+00,&
                    0.24678762946156D+00,  0.14956826027973D+00/)
    coe(1:nfun)=(/  0.10829594873725D+08, -0.14951140506828D+08,  0.15197454160767D+08, -0.10343128698320D+08,&
                    0.74653392980909D+07, -0.39035691887631D+07,  0.27991463443968D+07, -0.11456356387823D+07,&
                    0.10363040949038D+07, -0.24211463046885D+06,  0.42792044601211D+06,  0.15829767344549D+05,&
                    0.20818078988168D+06,  0.65966736433648D+05,  0.10845307728110D+06,  0.46216707855644D+05,&
                    0.40950896149009D+05,  0.72933628410893D+04,  0.90094238108536D+03, -0.29321270838021D+04,&
                    0.66254924201460D+04,  0.85885317760322D+04,  0.19125370452868D+04, -0.35451509742179D+04,&
                    0.13038865982991D+04,  0.23080052355172D+04, -0.65903760745528D+03, -0.20823819185501D+03,&
                    0.29757213498312D+03,  0.59884407147518D+02, -0.12700424233173D+02,  0.55118228228581D+01,&
                    0.46365039618912D+01,  0.15561907400980D+01,  0.37348569948023D+00,  0.54005111414916D-01,&
                    0.32465741649350D-02,  0.60683262266048D-04/)
  case(082)
    nfun = 37
    alf(1:nfun)=(/  0.16662700108425D+08,  0.10098606126318D+08,  0.61203673492837D+07,  0.37093135450204D+07,&
                    0.22480688151639D+07,  0.13624659485842D+07,  0.82573693853586D+06,  0.50044662941568D+06,&
                    0.30330098752465D+06,  0.18381878031797D+06,  0.11140532140483D+06,  0.67518376608988D+05,&
                    0.40920228247872D+05,  0.24800138332043D+05,  0.15030386867905D+05,  0.91093253744880D+04,&
                    0.55208032572654D+04,  0.33459413680397D+04,  0.20278432533574D+04,  0.12289959111257D+04,&
                    0.74484600674283D+03,  0.45142182226838D+03,  0.27358898319296D+03,  0.16581150496543D+03,&
                    0.10049182119117D+03,  0.60904134055255D+02,  0.36911596397124D+02,  0.22370664483105D+02,&
                    0.13557978474609D+02,  0.82169566512784D+01,  0.49799737280475D+01,  0.30181658957864D+01,&
                    0.18291914519917D+01,  0.11086008799950D+01,  0.67187932120909D+00,  0.40719958861157D+00,&
                    0.24678762946156D+00/)
    coe(1:nfun)=(/  0.14283188802286D+08, -0.23440895863293D+08,  0.24660929475788D+08, -0.17856158634605D+08,&
                    0.12623498681967D+08, -0.70500792780609D+07,  0.47006768843398D+07, -0.22013866446919D+07,&
                    0.16633245239506D+07, -0.56910310673821D+06,  0.63011666606730D+06, -0.80898876499742D+05,&
                    0.27401205432685D+06,  0.38156225272894D+05,  0.12949251921947D+06,  0.37289572703153D+05,&
                    0.46370912132824D+05,  0.34264508586936D+04,  0.21893025356801D+04, -0.35600495726690D+04,&
                    0.79723264676831D+04,  0.84882801593785D+04,  0.15515951411779D+04, -0.37512152808751D+04,&
                    0.17871846197679D+04,  0.21957648829155D+04, -0.77662098086419D+03, -0.14833972414176D+03,&
                    0.31961933141502D+03,  0.45095363321590D+02, -0.13307291750862D+02,  0.65696612215954D+01,&
                    0.50611206754643D+01,  0.14829881444421D+01,  0.31542395712638D+00,  0.26431729248735D-01,&
                    0.14216712710624D-02/)
  case(083)
    nfun = 38
    alf(1:nfun)=(/  0.16662700108425D+08,  0.10098606126318D+08,  0.61203673492837D+07,  0.37093135450204D+07,&
                    0.22480688151639D+07,  0.13624659485842D+07,  0.82573693853586D+06,  0.50044662941568D+06,&
                    0.30330098752465D+06,  0.18381878031797D+06,  0.11140532140483D+06,  0.67518376608988D+05,&
                    0.40920228247872D+05,  0.24800138332043D+05,  0.15030386867905D+05,  0.91093253744880D+04,&
                    0.55208032572654D+04,  0.33459413680397D+04,  0.20278432533574D+04,  0.12289959111257D+04,&
                    0.74484600674283D+03,  0.45142182226838D+03,  0.27358898319296D+03,  0.16581150496543D+03,&
                    0.10049182119117D+03,  0.60904134055255D+02,  0.36911596397124D+02,  0.22370664483105D+02,&
                    0.13557978474609D+02,  0.82169566512784D+01,  0.49799737280475D+01,  0.30181658957864D+01,&
                    0.18291914519917D+01,  0.11086008799950D+01,  0.67187932120909D+00,  0.40719958861157D+00,&
                    0.24678762946156D+00,  0.14956826027973D+00/)
    coe(1:nfun)=(/  0.17547336968006D+08, -0.27107925360920D+08,  0.27728561286453D+08, -0.19795819479651D+08,&
                    0.13881798728553D+08, -0.77238237631650D+07,  0.51328989869030D+07, -0.24054500566236D+07,&
                    0.18096182236112D+07, -0.62459120861465D+06,  0.68189785607633D+06, -0.93139800786912D+05,&
                    0.29337753840909D+06,  0.36369638073143D+05,  0.13613366154913D+06,  0.36267455987315D+05,&
                    0.47275711443042D+05,  0.19851542459903D+04,  0.21428838611073D+04, -0.33558664172604D+04,&
                    0.89190200151470D+04,  0.85668827072752D+04,  0.10233755141653D+04, -0.38283544346772D+04,&
                    0.22390801663806D+04,  0.20684892592017D+04, -0.89426912566069D+03, -0.74199513885816D+02,&
                    0.33368111877386D+03,  0.31171514411098D+02, -0.14219706752718D+02,  0.83998685892788D+01,&
                    0.51521612113462D+01,  0.14491095259865D+01,  0.21804379769364D+00,  0.15523667252619D-01,&
                    0.25958449451124D-03,  0.13838526949694D-04/)
  case(084)
    nfun = 38
    alf(1:nfun)=(/  0.16662700108425D+08,  0.10098606126318D+08,  0.61203673492837D+07,  0.37093135450204D+07,&
                    0.22480688151639D+07,  0.13624659485842D+07,  0.82573693853586D+06,  0.50044662941568D+06,&
                    0.30330098752465D+06,  0.18381878031797D+06,  0.11140532140483D+06,  0.67518376608988D+05,&
                    0.40920228247872D+05,  0.24800138332043D+05,  0.15030386867905D+05,  0.91093253744880D+04,&
                    0.55208032572654D+04,  0.33459413680397D+04,  0.20278432533574D+04,  0.12289959111257D+04,&
                    0.74484600674283D+03,  0.45142182226838D+03,  0.27358898319296D+03,  0.16581150496543D+03,&
                    0.10049182119117D+03,  0.60904134055255D+02,  0.36911596397124D+02,  0.22370664483105D+02,&
                    0.13557978474609D+02,  0.82169566512784D+01,  0.49799737280475D+01,  0.30181658957864D+01,&
                    0.18291914519917D+01,  0.11086008799950D+01,  0.67187932120909D+00,  0.40719958861157D+00,&
                    0.24678762946156D+00,  0.14956826027973D+00/)
    coe(1:nfun)=(/  0.19248351993299D+08, -0.30025363664125D+08,  0.30816698997468D+08, -0.22069331770235D+08,&
                    0.15472639997480D+08, -0.86348166188145D+07,  0.57192047539001D+07, -0.26987540387563D+07,&
                    0.20096497255266D+07, -0.70902214325860D+06,  0.75111869693227D+06, -0.11431863559733D+06,&
                    0.31816665682045D+06,  0.31814052653965D+05,  0.14432831804256D+06,  0.34293044134313D+05,&
                    0.48575681853568D+05,  0.24528174615512D+03,  0.22848512338791D+04, -0.31623635297333D+04,&
                    0.99312786158961D+04,  0.85563554076737D+04,  0.47999734749804D+03, -0.38543954793619D+04,&
                    0.26991965790686D+04,  0.19009178283278D+04, -0.99469288626902D+03,  0.46400493165265D+01,&
                    0.34425418242154D+03,  0.15410528607815D+02, -0.13546188146301D+02,  0.99903064977476D+01,&
                    0.53079070042796D+01,  0.12933496656349D+01,  0.14731601243335D+00,  0.82417897350702D-02,&
                   -0.22669451895219D-04,  0.17185218845417D-04/)
  case(085)
    nfun = 38
    alf(1:nfun)=(/  0.16662700108425D+08,  0.10098606126318D+08,  0.61203673492837D+07,  0.37093135450204D+07,&
                    0.22480688151639D+07,  0.13624659485842D+07,  0.82573693853586D+06,  0.50044662941568D+06,&
                    0.30330098752465D+06,  0.18381878031797D+06,  0.11140532140483D+06,  0.67518376608988D+05,&
                    0.40920228247872D+05,  0.24800138332043D+05,  0.15030386867905D+05,  0.91093253744880D+04,&
                    0.55208032572654D+04,  0.33459413680397D+04,  0.20278432533574D+04,  0.12289959111257D+04,&
                    0.74484600674283D+03,  0.45142182226838D+03,  0.27358898319296D+03,  0.16581150496543D+03,&
                    0.10049182119117D+03,  0.60904134055255D+02,  0.36911596397124D+02,  0.22370664483105D+02,&
                    0.13557978474609D+02,  0.82169566512784D+01,  0.49799737280475D+01,  0.30181658957864D+01,&
                    0.18291914519917D+01,  0.11086008799950D+01,  0.67187932120909D+00,  0.40719958861157D+00,&
                    0.24678762946156D+00,  0.14956826027973D+00/)
    coe(1:nfun)=(/  0.20079466475317D+08, -0.31108607764942D+08,  0.32005748374681D+08, -0.22850123536009D+08,&
                    0.16071968314632D+08, -0.89295850926549D+07,  0.59474242756271D+07, -0.27838060952253D+07,&
                    0.20943292665159D+07, -0.72703086052328D+06,  0.78446058329535D+06, -0.11473057713907D+06,&
                    0.33191523126337D+06,  0.33586517486606D+05,  0.14903303108071D+06,  0.34148266564884D+05,&
                    0.48722191905162D+05, -0.92625414548698D+03,  0.21391599325321D+04, -0.27071523880921D+04,&
                    0.10850257578406D+05,  0.85390249945096D+04, -0.12238892956811D+03, -0.37970602674533D+04,&
                    0.31443473707571D+04,  0.17038023907290D+04, -0.10809542842243D+04,  0.90177393531651D+02,&
                    0.34895133618406D+03, -0.64755816564525D+00, -0.11862993191252D+02,  0.11605663060848D+02,&
                    0.53364974699279D+01,  0.10951724224084D+01,  0.94510068319209D-01,  0.46094990276906D-02,&
                   -0.12199429258875D-03,  0.18587906874479D-04/)
  case(086)
    nfun = 38
    alf(1:nfun)=(/  0.16662700108425D+08,  0.10098606126318D+08,  0.61203673492837D+07,  0.37093135450204D+07,&
                    0.22480688151639D+07,  0.13624659485842D+07,  0.82573693853586D+06,  0.50044662941568D+06,&
                    0.30330098752465D+06,  0.18381878031797D+06,  0.11140532140483D+06,  0.67518376608988D+05,&
                    0.40920228247872D+05,  0.24800138332043D+05,  0.15030386867905D+05,  0.91093253744880D+04,&
                    0.55208032572654D+04,  0.33459413680397D+04,  0.20278432533574D+04,  0.12289959111257D+04,&
                    0.74484600674283D+03,  0.45142182226838D+03,  0.27358898319296D+03,  0.16581150496543D+03,&
                    0.10049182119117D+03,  0.60904134055255D+02,  0.36911596397124D+02,  0.22370664483105D+02,&
                    0.13557978474609D+02,  0.82169566512784D+01,  0.49799737280475D+01,  0.30181658957864D+01,&
                    0.18291914519917D+01,  0.11086008799950D+01,  0.67187932120909D+00,  0.40719958861157D+00,&
                    0.24678762946156D+00,  0.14956826027973D+00/)
    coe(1:nfun)=(/  0.19940392765511D+08, -0.30182341732844D+08,  0.31119095278997D+08, -0.22001183780369D+08,&
                    0.15585546518696D+08, -0.85489779873316D+07,  0.57825942847371D+07, -0.26407082825749D+07,&
                    0.20524815039275D+07, -0.67235317612907D+06,  0.77840778454661D+06, -0.92415375967965D+05,&
                    0.33349898242204D+06,  0.42301991891991D+05,  0.14988166875399D+06,  0.36025648852956D+05,&
                    0.47605916834552D+05, -0.14631866588590D+04,  0.16795018101521D+04, -0.19717903016402D+04,&
                    0.11660235274038D+05,  0.85157664259005D+04, -0.77977319837698D+03, -0.36523077395401D+04,&
                    0.35670352988434D+04,  0.14796366899402D+04, -0.11513929350242D+04,  0.18132160606776D+03,&
                    0.34734244851317D+03, -0.16533465667481D+02, -0.91873936219007D+01,  0.13192424196196D+02,&
                    0.52099982007415D+01,  0.88307486987796D+00,  0.58732358762541D-01,  0.26587447223889D-02,&
                   -0.13102590249048D-03,  0.16161261975789D-04/)
  case(087)
    nfun = 37
    alf(1:nfun)=(/  0.16662700108425D+08,  0.10098606126318D+08,  0.61203673492837D+07,  0.37093135450204D+07,&
                    0.22480688151639D+07,  0.13624659485842D+07,  0.82573693853586D+06,  0.50044662941568D+06,&
                    0.30330098752465D+06,  0.18381878031797D+06,  0.11140532140483D+06,  0.67518376608988D+05,&
                    0.40920228247872D+05,  0.24800138332043D+05,  0.15030386867905D+05,  0.91093253744880D+04,&
                    0.55208032572654D+04,  0.33459413680397D+04,  0.20278432533574D+04,  0.12289959111257D+04,&
                    0.74484600674283D+03,  0.45142182226838D+03,  0.27358898319296D+03,  0.16581150496543D+03,&
                    0.10049182119117D+03,  0.60904134055255D+02,  0.36911596397124D+02,  0.22370664483105D+02,&
                    0.13557978474609D+02,  0.82169566512784D+01,  0.49799737280475D+01,  0.30181658957864D+01,&
                    0.18291914519917D+01,  0.11086008799950D+01,  0.67187932120909D+00,  0.40719958861157D+00,&
                    0.24678762946156D+00/)
    coe(1:nfun)=(/  0.18039130541893D+08, -0.26152012421993D+08,  0.27045079584763D+08, -0.18588233685059D+08,&
                    0.13375569321638D+08, -0.71241056780162D+07,  0.50261627097149D+07, -0.21553557026975D+07,&
                    0.18147849588819D+07, -0.50293476407015D+06,  0.70949140547967D+06, -0.34942958197266D+05,&
                    0.31660818310119D+06,  0.61266130531214D+05,  0.14507660592522D+06,  0.40925937920030D+05,&
                    0.44655298579317D+05, -0.10409386246898D+04,  0.73112889160169D+03, -0.85420240028040D+03,&
                    0.12297632962839D+05,  0.85152410120890D+04, -0.15042827659884D+04, -0.34062477517664D+04,&
                    0.39535753961242D+04,  0.12345452202879D+04, -0.12060977656356D+04,  0.27770054820886D+03,&
                    0.33855142268983D+03, -0.31441244253991D+02, -0.56186321996711D+01,  0.14604360563245D+02,&
                    0.49686238132244D+01,  0.67666954082804D+00,  0.36886729590549D-01,  0.20468571512922D-03,&
                    0.53934060117555D-04/)
  case(088)
    nfun = 36
    alf(1:nfun)=(/  0.16662700108425D+08,  0.10098606126318D+08,  0.61203673492837D+07,  0.37093135450204D+07,&
                    0.22480688151639D+07,  0.13624659485842D+07,  0.82573693853586D+06,  0.50044662941568D+06,&
                    0.30330098752465D+06,  0.18381878031797D+06,  0.11140532140483D+06,  0.67518376608988D+05,&
                    0.40920228247872D+05,  0.24800138332043D+05,  0.15030386867905D+05,  0.91093253744880D+04,&
                    0.55208032572654D+04,  0.33459413680397D+04,  0.20278432533574D+04,  0.12289959111257D+04,&
                    0.74484600674283D+03,  0.45142182226838D+03,  0.27358898319296D+03,  0.16581150496543D+03,&
                    0.10049182119117D+03,  0.60904134055255D+02,  0.36911596397124D+02,  0.22370664483105D+02,&
                    0.13557978474609D+02,  0.82169566512784D+01,  0.49799737280475D+01,  0.30181658957864D+01,&
                    0.18291914519917D+01,  0.11086008799950D+01,  0.67187932120909D+00,  0.40719958861157D+00/)
    coe(1:nfun)=(/  0.16860526863998D+08, -0.22107997879135D+08,  0.22653951513931D+08, -0.14855877459633D+08,&
                    0.10943357716547D+08, -0.55207450469551D+07,  0.41491216675425D+07, -0.15945589429288D+07,&
                    0.15418118137472D+07, -0.31491458495593D+06,  0.63213207011831D+06,  0.28624088903487D+05,&
                    0.29629310140148D+06,  0.82567746427014D+05,  0.13887750541874D+06,  0.46463951369472D+05,&
                    0.41212663417095D+05, -0.40733025107533D+03, -0.27633875897806D+03,  0.40807414508937D+03,&
                    0.12888744761232D+05,  0.84588622502292D+04, -0.22448936050126D+04, -0.30823922611374D+04,&
                    0.43105687358358D+04,  0.96438730693953D+03, -0.12398006463536D+04,  0.37610170452145D+03,&
                    0.32302054720099D+03, -0.44993339628872D+02, -0.13014719935004D+01,  0.15876556668070D+02,&
                    0.45557670113625D+01,  0.51309274839946D+00,  0.16559124561169D-01,  0.29127646640958D-03/)
  case(089)
    nfun = 42
    alf(1:nfun)=(/  0.91541269670523D+09,  0.33623974167318D+09,  0.12350403734552D+09,  0.74850931724558D+08,&
                    0.45364201045187D+08,  0.27493455178901D+08,  0.16662700108425D+08,  0.10098606126318D+08,&
                    0.61203673492837D+07,  0.37093135450204D+07,  0.22480688151639D+07,  0.13624659485842D+07,&
                    0.82573693853586D+06,  0.50044662941568D+06,  0.30330098752465D+06,  0.18381878031797D+06,&
                    0.11140532140483D+06,  0.67518376608988D+05,  0.40920228247872D+05,  0.24800138332043D+05,&
                    0.15030386867905D+05,  0.91093253744880D+04,  0.55208032572654D+04,  0.33459413680397D+04,&
                    0.20278432533574D+04,  0.12289959111257D+04,  0.74484600674283D+03,  0.45142182226838D+03,&
                    0.27358898319296D+03,  0.16581150496543D+03,  0.10049182119117D+03,  0.60904134055255D+02,&
                    0.36911596397124D+02,  0.22370664483105D+02,  0.13557978474609D+02,  0.82169566512784D+01,&
                    0.49799737280475D+01,  0.30181658957864D+01,  0.18291914519917D+01,  0.11086008799950D+01,&
                    0.67187932120909D+00,  0.40719958861157D+00/)
    coe(1:nfun)=(/  0.22223957639252D+09, -0.15973490010736D+09,  0.18892620509058D+09, -0.24040982160278D+09,&
                    0.20648301664337D+09, -0.14145158244754D+09,  0.92703865757874D+08, -0.53450558912308D+08,&
                    0.32913909991104D+08, -0.16941477145233D+08,  0.10850135990637D+08, -0.48930204386425D+07,&
                    0.37135742275734D+07, -0.12411945420856D+07,  0.13757420842669D+07, -0.18003573442817D+06,&
                    0.58404140620524D+06,  0.77433656655490D+05,  0.28455205669317D+06,  0.99246045157865D+05,&
                    0.13525008731387D+06,  0.50405677059822D+05,  0.38509450886391D+05, -0.23168319629412D+03,&
                   -0.96676210442930D+03,  0.16125072084595D+04,  0.13536237416050D+05,  0.82840715970423D+04,&
                   -0.29607369531456D+04, -0.26983090152829D+04,  0.46411602335244D+04,  0.66716722531150D+03,&
                   -0.12479395206186D+04,  0.47264660181509D+03,  0.30231322592528D+03, -0.57612169282816D+02,&
                    0.40788237983152D+01,  0.16728123921367D+02,  0.41136047966759D+01,  0.35105089601598D+00,&
                    0.82721852090657D-02,  0.76282934724503D-04/)
  case(090)
    nfun = 39
    alf(1:nfun)=(/  0.45364201045187D+08,  0.27493455178901D+08,  0.16662700108425D+08,  0.10098606126318D+08,&
                    0.61203673492837D+07,  0.37093135450204D+07,  0.22480688151639D+07,  0.13624659485842D+07,&
                    0.82573693853586D+06,  0.50044662941568D+06,  0.30330098752465D+06,  0.18381878031797D+06,&
                    0.11140532140483D+06,  0.67518376608988D+05,  0.40920228247872D+05,  0.24800138332043D+05,&
                    0.15030386867905D+05,  0.91093253744880D+04,  0.55208032572654D+04,  0.33459413680397D+04,&
                    0.20278432533574D+04,  0.12289959111257D+04,  0.74484600674283D+03,  0.45142182226838D+03,&
                    0.27358898319296D+03,  0.16581150496543D+03,  0.10049182119117D+03,  0.60904134055255D+02,&
                    0.36911596397124D+02,  0.22370664483105D+02,  0.13557978474609D+02,  0.82169566512784D+01,&
                    0.49799737280475D+01,  0.30181658957864D+01,  0.18291914519917D+01,  0.11086008799950D+01,&
                    0.67187932120909D+00,  0.40719958861157D+00,  0.24678762946156D+00/)
    coe(1:nfun)=(/  0.20277255298592D+08, -0.29559287772160D+08,  0.33294860086166D+08, -0.22750244594776D+08,&
                    0.17235281488178D+08, -0.85429313735542D+07,  0.64705938821201D+07, -0.24202854701314D+07,&
                    0.24402964200685D+07, -0.47738838538778D+06,  0.10036253331671D+07,  0.65072805829465D+05,&
                    0.47841120532119D+06,  0.15831692632456D+06,  0.25570389662889D+06,  0.12553362168073D+06,&
                    0.12623047969935D+06,  0.57176986597851D+05,  0.34093451774011D+05,  0.84641494469717D+03,&
                   -0.20772815205853D+04,  0.31655833734402D+04,  0.14007623189470D+05,  0.81129581961589D+04,&
                   -0.37113221232033D+04, -0.22167967338325D+04,  0.49172758895211D+04,  0.35897665755612D+03,&
                   -0.12362411888402D+04,  0.56944977310499D+03,  0.27431980725588D+03, -0.67624664621618D+02,&
                    0.98221105264902D+01,  0.17335592364743D+02,  0.35687091445423D+01,  0.22704017131576D+00,&
                    0.48666660582626D-02, -0.15664804785942D-03,  0.20593534206803D-04/)
  case(091)
    nfun = 39
    alf(1:nfun)=(/  0.45364201045187D+08,  0.27493455178901D+08,  0.16662700108425D+08,  0.10098606126318D+08,&
                    0.61203673492837D+07,  0.37093135450204D+07,  0.22480688151639D+07,  0.13624659485842D+07,&
                    0.82573693853586D+06,  0.50044662941568D+06,  0.30330098752465D+06,  0.18381878031797D+06,&
                    0.11140532140483D+06,  0.67518376608988D+05,  0.40920228247872D+05,  0.24800138332043D+05,&
                    0.15030386867905D+05,  0.91093253744880D+04,  0.55208032572654D+04,  0.33459413680397D+04,&
                    0.20278432533574D+04,  0.12289959111257D+04,  0.74484600674283D+03,  0.45142182226838D+03,&
                    0.27358898319296D+03,  0.16581150496543D+03,  0.10049182119117D+03,  0.60904134055255D+02,&
                    0.36911596397124D+02,  0.22370664483105D+02,  0.13557978474609D+02,  0.82169566512784D+01,&
                    0.49799737280475D+01,  0.30181658957864D+01,  0.18291914519917D+01,  0.11086008799950D+01,&
                    0.67187932120909D+00,  0.40719958861157D+00,  0.24678762946156D+00/)
    coe(1:nfun)=(/  0.66538828315353D+07, -0.52656179318161D+07,  0.98493878281250D+07, -0.48085229293060D+07,&
                    0.56848221779652D+07, -0.13404775213604D+07,  0.23973774330044D+07,  0.17661034177303D+05,&
                    0.11508432679982D+07,  0.31237744555962D+06,  0.61768403658128D+06,  0.32078452689463D+06,&
                    0.36820983120197D+06,  0.24294156366505D+06,  0.22511513657439D+06,  0.15302795649936D+06,&
                    0.11643329834751D+06,  0.64224213227673D+05,  0.29364242630757D+05,  0.20697277139928D+04,&
                   -0.31874003144160D+04,  0.48401454662195D+04,  0.14416065653089D+05,  0.78781226563218D+04,&
                   -0.44484753060621D+04, -0.16577860726603D+04,  0.51341097302585D+04,  0.42625668632183D+02,&
                   -0.11963967840672D+04,  0.65840986332180D+03,  0.23965715108403D+03, -0.73028371923222D+02,&
                    0.15812236694122D+02,  0.16865517542495D+02,  0.31192368195753D+01,  0.18220613981006D+00,&
                    0.51280429644791D-02, -0.31279258378625D-03,  0.36078162217290D-04/)
  case(092)
    nfun = 40
    alf(1:nfun)=(/  0.74850931724558D+08,  0.45364201045187D+08,  0.27493455178901D+08,  0.16662700108425D+08,&
                    0.10098606126318D+08,  0.61203673492837D+07,  0.37093135450204D+07,  0.22480688151639D+07,&
                    0.13624659485842D+07,  0.82573693853586D+06,  0.50044662941568D+06,  0.30330098752465D+06,&
                    0.18381878031797D+06,  0.11140532140483D+06,  0.67518376608988D+05,  0.40920228247872D+05,&
                    0.24800138332043D+05,  0.15030386867905D+05,  0.91093253744880D+04,  0.55208032572654D+04,&
                    0.33459413680397D+04,  0.20278432533574D+04,  0.12289959111257D+04,  0.74484600674283D+03,&
                    0.45142182226838D+03,  0.27358898319296D+03,  0.16581150496543D+03,  0.10049182119117D+03,&
                    0.60904134055255D+02,  0.36911596397124D+02,  0.22370664483105D+02,  0.13557978474609D+02,&
                    0.82169566512784D+01,  0.49799737280475D+01,  0.30181658957864D+01,  0.18291914519917D+01,&
                    0.11086008799950D+01,  0.67187932120909D+00,  0.40719958861157D+00,  0.24678762946156D+00/)
    coe(1:nfun)=(/  0.83935985315590D+07, -0.16927060079262D+08,  0.24470649337509D+08, -0.15347928631805D+08,&
                    0.13336538204987D+08, -0.56658321157112D+07,  0.56713850762939D+07, -0.15273914616976D+07,&
                    0.23653946618156D+07, -0.76383872395277D+05,  0.10713988996096D+07,  0.25120709416825D+06,&
                    0.56855282737213D+06,  0.26385389317024D+06,  0.32511637809268D+06,  0.19645594583190D+06,&
                    0.17950663957804D+06,  0.10712614689524D+06,  0.70812552331190D+05,  0.24742244242666D+05,&
                    0.32118812412196D+04, -0.41567272510035D+04,  0.65568431517616D+04,  0.14798123371825D+05,&
                    0.75542906828382D+04, -0.51516014700781D+04, -0.10359637687717D+04,  0.53033443667197D+04,&
                   -0.28381185914225D+03, -0.11340366162905D+04,  0.74293593068034D+03,  0.20051852585779D+03,&
                   -0.76156036631049D+02,  0.21590213775563D+02,  0.16400988591123D+02,  0.26721692520808D+01,&
                    0.12850951783784D+00,  0.49823092795691D-02, -0.47940774186195D-03,  0.51713257763319D-04/)
  case(093)
    nfun = 36
    alf(1:nfun)=(/  0.27493455178901D+08,  0.16662700108425D+08,  0.10098606126318D+08,  0.61203673492837D+07,&
                    0.37093135450204D+07,  0.22480688151639D+07,  0.13624659485842D+07,  0.82573693853586D+06,&
                    0.50044662941568D+06,  0.30330098752465D+06,  0.18381878031797D+06,  0.11140532140483D+06,&
                    0.67518376608988D+05,  0.40920228247872D+05,  0.24800138332043D+05,  0.15030386867905D+05,&
                    0.91093253744880D+04,  0.55208032572654D+04,  0.33459413680397D+04,  0.20278432533574D+04,&
                    0.12289959111257D+04,  0.74484600674283D+03,  0.45142182226838D+03,  0.27358898319296D+03,&
                    0.16581150496543D+03,  0.10049182119117D+03,  0.60904134055255D+02,  0.36911596397124D+02,&
                    0.22370664483105D+02,  0.13557978474609D+02,  0.82169566512784D+01,  0.49799737280475D+01,&
                    0.30181658957864D+01,  0.18291914519917D+01,  0.11086008799950D+01,  0.67187932120909D+00/)
    coe(1:nfun)=(/  0.12846653617805D+08, -0.15181270483643D+08,  0.19307587204636D+08, -0.11848239559456D+08,&
                    0.10578483195908D+08, -0.45693112948265D+07,  0.43369769490891D+07, -0.11223430050144D+07,&
                    0.17457405989571D+07, -0.67802693966389D+05,  0.79369080237365D+06,  0.17338537743306D+06,&
                    0.40074067765671D+06,  0.17187730911928D+06,  0.20387085951957D+06,  0.98877357690021D+05,&
                    0.76624753093019D+05,  0.20399859025959D+05,  0.41819273729082D+04, -0.49251755156368D+04,&
                    0.82815457616579D+04,  0.15162943595309D+05,  0.71322805868005D+04, -0.58042015847268D+04,&
                   -0.35816707861098D+03,  0.54170771804246D+04, -0.61360855707216D+03, -0.10470980851640D+04,&
                    0.81878425728587D+03,  0.15757176006616D+03, -0.75869346905021D+02,  0.26796608829663D+02,&
                    0.15711483506160D+02,  0.22627798176268D+01,  0.97833510099861D-01,  0.22797077412022D-02/)
  case(094)
    nfun = 36
    alf(1:nfun)=(/  0.27493455178901D+08,  0.16662700108425D+08,  0.10098606126318D+08,  0.61203673492837D+07,&
                    0.37093135450204D+07,  0.22480688151639D+07,  0.13624659485842D+07,  0.82573693853586D+06,&
                    0.50044662941568D+06,  0.30330098752465D+06,  0.18381878031797D+06,  0.11140532140483D+06,&
                    0.67518376608988D+05,  0.40920228247872D+05,  0.24800138332043D+05,  0.15030386867905D+05,&
                    0.91093253744880D+04,  0.55208032572654D+04,  0.33459413680397D+04,  0.20278432533574D+04,&
                    0.12289959111257D+04,  0.74484600674283D+03,  0.45142182226838D+03,  0.27358898319296D+03,&
                    0.16581150496543D+03,  0.10049182119117D+03,  0.60904134055255D+02,  0.36911596397124D+02,&
                    0.22370664483105D+02,  0.13557978474609D+02,  0.82169566512784D+01,  0.49799737280475D+01,&
                    0.30181658957864D+01,  0.18291914519917D+01,  0.11086008799950D+01,  0.67187932120909D+00/)
    coe(1:nfun)=(/  0.17286686283066D+08, -0.23859919028549D+08,  0.28914698900360D+08, -0.19149059391556D+08,&
                    0.15738306170364D+08, -0.75885626084366D+07,  0.62530554596405D+07, -0.21136119172430D+07,&
                    0.23901727066746D+07, -0.36529991803312D+06,  0.10090998361809D+07,  0.89966889498353D+05,&
                    0.47340229481351D+06,  0.14939803859255D+06,  0.22711077673715D+06,  0.91149941240311D+05,&
                    0.81931703230344D+05,  0.16198433894385D+05,  0.50729874401577D+04, -0.55341568292640D+04,&
                    0.10034821224913D+05,  0.15486770578361D+05,  0.66218047564381D+04, -0.63987684116696D+04,&
                    0.37118499960535D+03,  0.54681203592259D+04, -0.93929063896550D+03, -0.93572214844089D+03,&
                    0.88268263951178D+03,  0.11226007507118D+03, -0.71832580674578D+02,  0.31163330376767D+02,&
                    0.14672748510363D+02,  0.19666979233253D+01,  0.79427997725588D-01,  0.24574871673906D-02/)
  case(095)
    nfun = 36
    alf(1:nfun)=(/  0.27493455178901D+08,  0.16662700108425D+08,  0.10098606126318D+08,  0.61203673492837D+07,&
                    0.37093135450204D+07,  0.22480688151639D+07,  0.13624659485842D+07,  0.82573693853586D+06,&
                    0.50044662941568D+06,  0.30330098752465D+06,  0.18381878031797D+06,  0.11140532140483D+06,&
                    0.67518376608988D+05,  0.40920228247872D+05,  0.24800138332043D+05,  0.15030386867905D+05,&
                    0.91093253744880D+04,  0.55208032572654D+04,  0.33459413680397D+04,  0.20278432533574D+04,&
                    0.12289959111257D+04,  0.74484600674283D+03,  0.45142182226838D+03,  0.27358898319296D+03,&
                    0.16581150496543D+03,  0.10049182119117D+03,  0.60904134055255D+02,  0.36911596397124D+02,&
                    0.22370664483105D+02,  0.13557978474609D+02,  0.82169566512784D+01,  0.49799737280475D+01,&
                    0.30181658957864D+01,  0.18291914519917D+01,  0.11086008799950D+01,  0.67187932120909D+00/)
    coe(1:nfun)=(/  0.19431072616356D+08, -0.27862495153870D+08,  0.33726124468628D+08, -0.22668442263260D+08,&
                    0.18388553605095D+08, -0.90469866596260D+07,  0.72651837516117D+07, -0.25777488566246D+07,&
                    0.27421786275244D+07, -0.49412272280169D+06,  0.11332270294673D+07,  0.59179793602705D+05,&
                    0.51792161852866D+06,  0.14298423003986D+06,  0.24156839278185D+06,  0.88122503336966D+05,&
                    0.84433853525318D+05,  0.13403173280869D+05,  0.51888933476266D+04, -0.55866160791367D+04,&
                    0.11594331014389D+05,  0.15884555281939D+05,  0.59570399902584D+04, -0.68910582965867D+04,&
                    0.11249275351022D+04,  0.54709629157517D+04, -0.12657038656360D+04, -0.80010614873326D+03,&
                    0.93487574660126D+03,  0.66289882705238D+02, -0.65808049681063D+02,  0.34978763917157D+02,&
                    0.13694813336331D+02,  0.16755981018525D+01,  0.57210896051828D-01,  0.21589136391640D-02/)
  case(096)
    nfun = 36
    alf(1:nfun)=(/  0.27493455178901D+08,  0.16662700108425D+08,  0.10098606126318D+08,  0.61203673492837D+07,&
                    0.37093135450204D+07,  0.22480688151639D+07,  0.13624659485842D+07,  0.82573693853586D+06,&
                    0.50044662941568D+06,  0.30330098752465D+06,  0.18381878031797D+06,  0.11140532140483D+06,&
                    0.67518376608988D+05,  0.40920228247872D+05,  0.24800138332043D+05,  0.15030386867905D+05,&
                    0.91093253744880D+04,  0.55208032572654D+04,  0.33459413680397D+04,  0.20278432533574D+04,&
                    0.12289959111257D+04,  0.74484600674283D+03,  0.45142182226838D+03,  0.27358898319296D+03,&
                    0.16581150496543D+03,  0.10049182119117D+03,  0.60904134055255D+02,  0.36911596397124D+02,&
                    0.22370664483105D+02,  0.13557978474609D+02,  0.82169566512784D+01,  0.49799737280475D+01,&
                    0.30181658957864D+01,  0.18291914519917D+01,  0.11086008799950D+01,  0.67187932120909D+00/)
    coe(1:nfun)=(/  0.19138142315392D+08, -0.27122214819260D+08,  0.33784542972260D+08, -0.22528602774780D+08,&
                    0.18643694623543D+08, -0.90230666230354D+07,  0.74183512639751D+07, -0.25422541044502D+07,&
                    0.28191210441751D+07, -0.46532367406416D+06,  0.11723587202189D+07,  0.77531531986833D+05,&
                    0.53604330196869D+06,  0.15167687597582D+06,  0.24768677943785D+06,  0.89541972088516D+05,&
                    0.84259671121791D+05,  0.11948923407450D+05,  0.45873345648311D+04, -0.51078868036736D+04,&
                    0.12971280681931D+05,  0.16336837207668D+05,  0.51481604368538D+04, -0.72752574579492D+04,&
                    0.18974939470459D+04,  0.54229125585297D+04, -0.15880924360429D+04, -0.64169123147248D+03,&
                    0.97459224722506D+03,  0.20359046152262D+02, -0.58054371773880D+02,  0.38354777297079D+02,&
                    0.12752721937007D+02,  0.13803156565962D+01,  0.33882925867943D-01,  0.16653925295209D-02/)
  case(097)
    nfun = 36
    alf(1:nfun)=(/  0.27493455178901D+08,  0.16662700108425D+08,  0.10098606126318D+08,  0.61203673492837D+07,&
                    0.37093135450204D+07,  0.22480688151639D+07,  0.13624659485842D+07,  0.82573693853586D+06,&
                    0.50044662941568D+06,  0.30330098752465D+06,  0.18381878031797D+06,  0.11140532140483D+06,&
                    0.67518376608988D+05,  0.40920228247872D+05,  0.24800138332043D+05,  0.15030386867905D+05,&
                    0.91093253744880D+04,  0.55208032572654D+04,  0.33459413680397D+04,  0.20278432533574D+04,&
                    0.12289959111257D+04,  0.74484600674283D+03,  0.45142182226838D+03,  0.27358898319296D+03,&
                    0.16581150496543D+03,  0.10049182119117D+03,  0.60904134055255D+02,  0.36911596397124D+02,&
                    0.22370664483105D+02,  0.13557978474609D+02,  0.82169566512784D+01,  0.49799737280475D+01,&
                    0.30181658957864D+01,  0.18291914519917D+01,  0.11086008799950D+01,  0.67187932120909D+00/)
    coe(1:nfun)=(/  0.19428021261757D+08, -0.25510645885818D+08,  0.32358342884079D+08, -0.20924603274582D+08,&
                    0.17860738517868D+08, -0.83121665210838D+07,  0.71735270385933D+07, -0.22658018533535D+07,&
                    0.27654887469444D+07, -0.35804588667488D+06,  0.11706463214066D+07,  0.12059825976694D+06,&
                    0.54148333487898D+06,  0.16786627931675D+06,  0.24966495289826D+06,  0.93095025388882D+05,&
                    0.82649354111888D+05,  0.11177592923995D+05,  0.36241368992124D+04, -0.42746266141990D+04,&
                    0.14253984832866D+05,  0.16782962148812D+05,  0.42284089513819D+04, -0.75531651937183D+04,&
                    0.26865720217585D+04,  0.53091139918961D+04, -0.18903097434037D+04, -0.46254040086604D+03,&
                    0.99590480036369D+03, -0.23072797133067D+02, -0.47051822817004D+02,  0.40197549322065D+02,&
                    0.11642868372804D+02,  0.12084612185862D+01,  0.28542840418282D-01,  0.17469754635577D-02/)
  case(098)
    nfun = 36
    alf(1:nfun)=(/  0.27493455178901D+08,  0.16662700108425D+08,  0.10098606126318D+08,  0.61203673492837D+07,&
                    0.37093135450204D+07,  0.22480688151639D+07,  0.13624659485842D+07,  0.82573693853586D+06,&
                    0.50044662941568D+06,  0.30330098752465D+06,  0.18381878031797D+06,  0.11140532140483D+06,&
                    0.67518376608988D+05,  0.40920228247872D+05,  0.24800138332043D+05,  0.15030386867905D+05,&
                    0.91093253744880D+04,  0.55208032572654D+04,  0.33459413680397D+04,  0.20278432533574D+04,&
                    0.12289959111257D+04,  0.74484600674283D+03,  0.45142182226838D+03,  0.27358898319296D+03,&
                    0.16581150496543D+03,  0.10049182119117D+03,  0.60904134055255D+02,  0.36911596397124D+02,&
                    0.22370664483105D+02,  0.13557978474609D+02,  0.82169566512784D+01,  0.49799737280475D+01,&
                    0.30181658957864D+01,  0.18291914519917D+01,  0.11086008799950D+01,  0.67187932120909D+00/)
    coe(1:nfun)=(/  0.17358834488991D+08, -0.18956639079010D+08,  0.25863248805458D+08, -0.15349098649788D+08,&
                    0.14453202955254D+08, -0.59664483743134D+07,  0.59811282637653D+07, -0.14376138564882D+07,&
                    0.24076998978095D+07, -0.75828927730560D+05,  0.10743604685726D+07,  0.21820548983526D+06,&
                    0.51764246738505D+06,  0.20066619268236D+06,  0.24245686724950D+06,  0.10150634602486D+06,&
                    0.78131344201900D+05,  0.11879906261541D+05,  0.19018198812511D+04, -0.28715722282184D+04,&
                    0.15322869018299D+05,  0.17277035032867D+05,  0.31696522362622D+04, -0.77023997491112D+04,&
                    0.34742423820678D+04,  0.51445440586746D+04, -0.21790071634876D+04, -0.26497444373393D+03,&
                    0.10029603563693D+04, -0.63933354351389D+02, -0.35258484391387D+02,  0.41630602198410D+02,&
                    0.10642808276660D+02,  0.10215364015093D+01,  0.20067064309160D-01,  0.15371616528057D-02/)
  case(099)
    nfun = 36
    alf(1:nfun)=(/  0.27493455178901D+08,  0.16662700108425D+08,  0.10098606126318D+08,  0.61203673492837D+07,&
                    0.37093135450204D+07,  0.22480688151639D+07,  0.13624659485842D+07,  0.82573693853586D+06,&
                    0.50044662941568D+06,  0.30330098752465D+06,  0.18381878031797D+06,  0.11140532140483D+06,&
                    0.67518376608988D+05,  0.40920228247872D+05,  0.24800138332043D+05,  0.15030386867905D+05,&
                    0.91093253744880D+04,  0.55208032572654D+04,  0.33459413680397D+04,  0.20278432533574D+04,&
                    0.12289959111257D+04,  0.74484600674283D+03,  0.45142182226838D+03,  0.27358898319296D+03,&
                    0.16581150496543D+03,  0.10049182119117D+03,  0.60904134055255D+02,  0.36911596397124D+02,&
                    0.22370664483105D+02,  0.13557978474609D+02,  0.82169566512784D+01,  0.49799737280475D+01,&
                    0.30181658957864D+01,  0.18291914519917D+01,  0.11086008799950D+01,  0.67187932120909D+00/)
    coe(1:nfun)=(/  0.14534735548294D+08, -0.97774751823807D+07,  0.16401533182899D+08, -0.72952583628922D+07,&
                    0.93863003906631D+07, -0.25665503538322D+07,  0.41808826479340D+07, -0.24774794174480D+06,&
                    0.18525364361291D+07,  0.32221140039372D+06,  0.91703005671287D+06,  0.35173801090634D+06,&
                    0.47491852467048D+06,  0.24438350649121D+06,  0.22914548641150D+06,  0.11310257109047D+06,&
                    0.71580701998565D+05,  0.13608916766338D+05, -0.34269508365169D+03, -0.10088088106662D+04,&
                    0.16227637314840D+05,  0.17779907981248D+05,  0.19966546699119D+04, -0.77242577163429D+04,&
                    0.42557177051245D+04,  0.49238790879269D+04, -0.24450574694128D+04, -0.52203988314235D+02,&
                    0.99399887604769D+03, -0.10083439518311D+03, -0.22491504130762D+02,  0.42346398548662D+02,&
                    0.96934755258153D+01,  0.85923000861806D+00,  0.14458537866204D-01,  0.12956215785067D-02/)
  case(100)
    nfun = 37
    alf(1:nfun)=(/  0.45364201045187D+08,  0.27493455178901D+08,  0.16662700108425D+08,  0.10098606126318D+08,&
                    0.61203673492837D+07,  0.37093135450204D+07,  0.22480688151639D+07,  0.13624659485842D+07,&
                    0.82573693853586D+06,  0.50044662941568D+06,  0.30330098752465D+06,  0.18381878031797D+06,&
                    0.11140532140483D+06,  0.67518376608988D+05,  0.40920228247872D+05,  0.24800138332043D+05,&
                    0.15030386867905D+05,  0.91093253744880D+04,  0.55208032572654D+04,  0.33459413680397D+04,&
                    0.20278432533574D+04,  0.12289959111257D+04,  0.74484600674283D+03,  0.45142182226838D+03,&
                    0.27358898319296D+03,  0.16581150496543D+03,  0.10049182119117D+03,  0.60904134055255D+02,&
                    0.36911596397124D+02,  0.22370664483105D+02,  0.13557978474609D+02,  0.82169566512784D+01,&
                    0.49799737280475D+01,  0.30181658957864D+01,  0.18291914519917D+01,  0.11086008799950D+01,&
                    0.67187932120909D+00/)
    coe(1:nfun)=(/ -0.10996248851700D+07,  0.12905245155411D+08, -0.10998863327026D+06,  0.57376929338531D+07,&
                    0.19882886769485D+07,  0.34492736978798D+07,  0.14185694672012D+07,  0.20432666900578D+07,&
                    0.11485719649830D+07,  0.11881515003166D+07,  0.78668764934444D+06,  0.72591237713790D+06,&
                    0.50604531370902D+06,  0.42168740239143D+06,  0.29425638060147D+06,  0.21237623270610D+06,&
                    0.12634912292979D+06,  0.63884036778681D+05,  0.15858307935458D+05, -0.27924211351480D+04,&
                    0.11362513779001D+04,  0.17060192541025D+05,  0.18227571181496D+05,  0.74924771327409D+03,&
                   -0.76291943096306D+04,  0.50295783193966D+04,  0.46439803280625D+04, -0.26812966360008D+04,&
                    0.17112123238898D+03,  0.96977419804288D+03, -0.13310836243103D+03, -0.91697339588336D+01,&
                    0.42411837950994D+02,  0.88112514478187D+01,  0.71720498537582D+00,  0.11159556727165D-01,&
                    0.10133919159827D-02/)
  case(101)
    nfun = 37
    alf(1:nfun)=(/  0.45364201045187D+08,  0.27493455178901D+08,  0.16662700108425D+08,  0.10098606126318D+08,&
                    0.61203673492837D+07,  0.37093135450204D+07,  0.22480688151639D+07,  0.13624659485842D+07,&
                    0.82573693853586D+06,  0.50044662941568D+06,  0.30330098752465D+06,  0.18381878031797D+06,&
                    0.11140532140483D+06,  0.67518376608988D+05,  0.40920228247872D+05,  0.24800138332043D+05,&
                    0.15030386867905D+05,  0.91093253744880D+04,  0.55208032572654D+04,  0.33459413680397D+04,&
                    0.20278432533574D+04,  0.12289959111257D+04,  0.74484600674283D+03,  0.45142182226838D+03,&
                    0.27358898319296D+03,  0.16581150496543D+03,  0.10049182119117D+03,  0.60904134055255D+02,&
                    0.36911596397124D+02,  0.22370664483105D+02,  0.13557978474609D+02,  0.82169566512784D+01,&
                    0.49799737280475D+01,  0.30181658957864D+01,  0.18291914519917D+01,  0.11086008799950D+01,&
                    0.67187932120909D+00/)
    coe(1:nfun)=(/  0.21495472915428D+08, -0.19050657971802D+08,  0.30246405344543D+08, -0.15644412036316D+08,&
                    0.16286698308189D+08, -0.47356479649582D+07,  0.64699113950768D+07, -0.59548373308182D+06,&
                    0.28115617265396D+07,  0.39196079190159D+06,  0.13275663969641D+07,  0.49705536326051D+06,&
                    0.68310335085273D+06,  0.35704219756955D+06,  0.35080824563339D+06,  0.19181647576556D+06,&
                    0.14144792988236D+06,  0.54890501471609D+05,  0.18737526562009D+05, -0.55070889801662D+04,&
                    0.36098422905859D+04,  0.17784731695826D+05,  0.18629175186557D+05, -0.57305638575961D+03,&
                   -0.74062041889149D+04,  0.57819578294416D+04,  0.43093653612793D+04, -0.28845782265787D+04,&
                    0.40182059236576D+03,  0.93072893377959D+03, -0.16003815605424D+03,  0.42414382851202D+01,&
                    0.41931550142448D+02,  0.79983500797039D+01,  0.59289764630887D+00,  0.97209122539539D-02,&
                    0.69140314460370D-03/)
  case(102)
    nfun = 37
    alf(1:nfun)=(/  0.45364201045187D+08,  0.27493455178901D+08,  0.16662700108425D+08,  0.10098606126318D+08,&
                    0.61203673492837D+07,  0.37093135450204D+07,  0.22480688151639D+07,  0.13624659485842D+07,&
                    0.82573693853586D+06,  0.50044662941568D+06,  0.30330098752465D+06,  0.18381878031797D+06,&
                    0.11140532140483D+06,  0.67518376608988D+05,  0.40920228247872D+05,  0.24800138332043D+05,&
                    0.15030386867905D+05,  0.91093253744880D+04,  0.55208032572654D+04,  0.33459413680397D+04,&
                    0.20278432533574D+04,  0.12289959111257D+04,  0.74484600674283D+03,  0.45142182226838D+03,&
                    0.27358898319296D+03,  0.16581150496543D+03,  0.10049182119117D+03,  0.60904134055255D+02,&
                    0.36911596397124D+02,  0.22370664483105D+02,  0.13557978474609D+02,  0.82169566512784D+01,&
                    0.49799737280475D+01,  0.30181658957864D+01,  0.18291914519917D+01,  0.11086008799950D+01,&
                    0.67187932120909D+00/)
    coe(1:nfun)=(/  0.45994672714508D+08, -0.53559455837143D+08,  0.63018224520401D+08, -0.38698048094025D+08,&
                    0.31696527454933D+08, -0.13541316136608D+08,  0.11903014878658D+08, -0.34327295232410D+07,&
                    0.45999047773752D+07, -0.46520633867502D+06,  0.19078555275435D+07,  0.25041947066128D+06,&
                    0.87202128402698D+06,  0.28689158357531D+06,  0.41076156593862D+06,  0.16921795626745D+06,&
                    0.15741262130731D+06,  0.45148315415073D+05,  0.21952168062748D+05, -0.83002453133818D+04,&
                    0.63107593545886D+04,  0.18448447463061D+05,  0.18944454231992D+05, -0.19391162697127D+04,&
                   -0.70631062483985D+04,  0.65086369642775D+04,  0.39199428725536D+04, -0.30496696587000D+04,&
                    0.63539659764315D+03,  0.87833652222486D+03, -0.18121768568722D+03,  0.17323233420412D+02,&
                    0.40994392875870D+02,  0.72546381807408D+01,  0.48447395985943D+00,  0.95106015050703D-02,&
                    0.35544364495976D-03/)
  case(103)
    nfun = 38
    alf(1:nfun)=(/  0.45364201045187D+08,  0.27493455178901D+08,  0.16662700108425D+08,  0.10098606126318D+08,&
                    0.61203673492837D+07,  0.37093135450204D+07,  0.22480688151639D+07,  0.13624659485842D+07,&
                    0.82573693853586D+06,  0.50044662941568D+06,  0.30330098752465D+06,  0.18381878031797D+06,&
                    0.11140532140483D+06,  0.67518376608988D+05,  0.40920228247872D+05,  0.24800138332043D+05,&
                    0.15030386867905D+05,  0.91093253744880D+04,  0.55208032572654D+04,  0.33459413680397D+04,&
                    0.20278432533574D+04,  0.12289959111257D+04,  0.74484600674283D+03,  0.45142182226838D+03,&
                    0.27358898319296D+03,  0.16581150496543D+03,  0.10049182119117D+03,  0.60904134055255D+02,&
                    0.36911596397124D+02,  0.22370664483105D+02,  0.13557978474609D+02,  0.82169566512784D+01,&
                    0.49799737280475D+01,  0.30181658957864D+01,  0.18291914519917D+01,  0.11086008799950D+01,&
                    0.67187932120909D+00,  0.40719958861157D+00/)
    coe(1:nfun)=(/  0.62634466854797D+08, -0.80587367226952D+08,  0.90875520265396D+08, -0.58973067998699D+08,&
                    0.45649324852997D+08, -0.21543530871952D+08,  0.16931289777293D+08, -0.60311234333115D+07,&
                    0.62686739240673D+07, -0.12467541264577D+07,  0.24531626400797D+07,  0.25854830507815D+05,&
                    0.10515407545335D+07,  0.22275922561605D+06,  0.46759575402862D+06,  0.14830649137831D+06,&
                    0.17205426710164D+06,  0.35916641257595D+05,  0.24808623888036D+05, -0.10771891863624D+05,&
                    0.90267130817212D+04,  0.19153202576569D+05,  0.19108545390383D+05, -0.33061191346624D+04,&
                   -0.66163534975814D+04,  0.72110958626366D+04,  0.34771354104534D+04, -0.31749987979363D+04,&
                    0.86868319542441D+03,  0.81511751233403D+03, -0.19798148529164D+03,  0.29928379617013D+02,&
                    0.40050369398153D+02,  0.65416294322885D+01,  0.36048687871619D+00,  0.94867680128488D-02,&
                   -0.38245221393041D-03,  0.51918787968397D-04/)
  case(104)
    nfun = 38
    alf(1:nfun)=(/  0.33623974167318D+09,  0.12350403734552D+09,  0.45364201045187D+08,  0.27493455178901D+08,&
                    0.16662700108425D+08,  0.10098606126318D+08,  0.61203673492837D+07,  0.37093135450204D+07,&
                    0.22480688151639D+07,  0.13624659485842D+07,  0.82573693853586D+06,  0.50044662941568D+06,&
                    0.30330098752465D+06,  0.18381878031797D+06,  0.11140532140483D+06,  0.67518376608988D+05,&
                    0.40920228247872D+05,  0.24800138332043D+05,  0.15030386867905D+05,  0.91093253744880D+04,&
                    0.55208032572654D+04,  0.33459413680397D+04,  0.20278432533574D+04,  0.12289959111257D+04,&
                    0.74484600674283D+03,  0.45142182226838D+03,  0.27358898319296D+03,  0.16581150496543D+03,&
                    0.10049182119117D+03,  0.60904134055255D+02,  0.36911596397124D+02,  0.22370664483105D+02,&
                    0.13557978474609D+02,  0.82169566512784D+01,  0.49799737280475D+01,  0.30181658957864D+01,&
                    0.18291914519917D+01,  0.11086008799950D+01/)
    coe(1:nfun)=(/  0.31318159126953D+08, -0.77491038641602D+08,  0.13504902154440D+09, -0.15502674235425D+09,&
                    0.14264919391895D+09, -0.88128995925903D+08,  0.62797946620621D+08, -0.30921880494476D+08,&
                    0.22646459576767D+08, -0.88368161399040D+07,  0.79820805765572D+07, -0.20424279564075D+07,&
                    0.30397232203283D+07, -0.22995102337813D+06,  0.12495494294081D+07,  0.15417390715039D+06,&
                    0.52434397243708D+06,  0.12819377280277D+06,  0.18581921588568D+06,  0.26838262506910D+05,&
                    0.27554282004528D+05, -0.13055260098446D+05,  0.11830204095049D+05,  0.19854037966660D+05,&
                    0.19133730125485D+05, -0.46708855309184D+04, -0.60583698261491D+04,  0.78747066105132D+04,&
                    0.29862281606917D+04, -0.32569980104158D+04,  0.10981978673121D+04,  0.74118460103658D+03,&
                   -0.20925613898750D+03,  0.41797040520429D+02,  0.39006126331383D+02,  0.57929913574212D+01,&
                    0.26577240016900D+00,  0.59482911072735D-02/)
  case(105)
    nfun = 36
    alf(1:nfun)=(/  0.45364201045187D+08,  0.27493455178901D+08,  0.16662700108425D+08,  0.10098606126318D+08,&
                    0.61203673492837D+07,  0.37093135450204D+07,  0.22480688151639D+07,  0.13624659485842D+07,&
                    0.82573693853586D+06,  0.50044662941568D+06,  0.30330098752465D+06,  0.18381878031797D+06,&
                    0.11140532140483D+06,  0.67518376608988D+05,  0.40920228247872D+05,  0.24800138332043D+05,&
                    0.15030386867905D+05,  0.91093253744880D+04,  0.55208032572654D+04,  0.33459413680397D+04,&
                    0.20278432533574D+04,  0.12289959111257D+04,  0.74484600674283D+03,  0.45142182226838D+03,&
                    0.27358898319296D+03,  0.16581150496543D+03,  0.10049182119117D+03,  0.60904134055255D+02,&
                    0.36911596397124D+02,  0.22370664483105D+02,  0.13557978474609D+02,  0.82169566512784D+01,&
                    0.49799737280475D+01,  0.30181658957864D+01,  0.18291914519917D+01,  0.11086008799950D+01/)
    coe(1:nfun)=(/  0.78543215161682D+08, -0.11786675339368D+09,  0.13308166211975D+09, -0.89728895086029D+08,&
                    0.67936580737198D+08, -0.34654305929001D+08,  0.25620079048698D+08, -0.10326595041958D+08,&
                    0.90875428474312D+07, -0.25011696148081D+07,  0.34250099564123D+07, -0.35903032585502D+06,&
                    0.13805144266944D+07,  0.11939652403378D+06,  0.56528505294412D+06,  0.11563776896724D+06,&
                    0.19521327710947D+06,  0.19957174320459D+05,  0.29031498365376D+05, -0.14470948794328D+05,&
                    0.14357511663856D+05,  0.20723704059198D+05,  0.18916024811941D+05, -0.59643330993054D+04,&
                   -0.54189456147619D+04,  0.85051037864550D+04,  0.24431413346719D+04, -0.32875294335674D+04,&
                    0.13172828540529D+04,  0.65899989015279D+03, -0.21537277964655D+03,  0.53059496095826D+02,&
                    0.37672632019505D+02,  0.50876539182429D+01,  0.18333384952837D+00,  0.50258267374232D-02/)
  case(106)
    nfun = 36
    alf(1:nfun)=(/  0.12350403734552D+09,  0.45364201045187D+08,  0.16662700108425D+08,  0.10098606126318D+08,&
                    0.61203673492837D+07,  0.37093135450204D+07,  0.22480688151639D+07,  0.13624659485842D+07,&
                    0.82573693853586D+06,  0.50044662941568D+06,  0.30330098752465D+06,  0.18381878031797D+06,&
                    0.11140532140483D+06,  0.67518376608988D+05,  0.40920228247872D+05,  0.24800138332043D+05,&
                    0.15030386867905D+05,  0.91093253744880D+04,  0.55208032572654D+04,  0.33459413680397D+04,&
                    0.20278432533574D+04,  0.12289959111257D+04,  0.74484600674283D+03,  0.45142182226838D+03,&
                    0.27358898319296D+03,  0.16581150496543D+03,  0.10049182119117D+03,  0.60904134055255D+02,&
                    0.36911596397124D+02,  0.22370664483105D+02,  0.13557978474609D+02,  0.82169566512784D+01,&
                    0.49799737280475D+01,  0.30181658957864D+01,  0.18291914519917D+01,  0.11086008799950D+01/)
    coe(1:nfun)=(/  0.14690617221451D+08, -0.12893668325615D+08,  0.57425318695290D+08, -0.54434436850342D+08,&
                    0.56297670275177D+08, -0.31211487586395D+08,  0.25605747442291D+08, -0.10487493876541D+08,&
                    0.95983448335552D+07, -0.26592938093643D+07,  0.36633334262366D+07, -0.40545686039996D+06,&
                    0.14687707594581D+07,  0.10902315525806D+06,  0.59308134230322D+06,  0.10996274427047D+06,&
                    0.20049560806571D+06,  0.15079700427420D+05,  0.29383327770084D+05, -0.15085692949533D+05,&
                    0.16640636458013D+05,  0.21735055432331D+05,  0.18460582849578D+05, -0.71767648009032D+04,&
                   -0.46976656196037D+04,  0.90912896578188D+04,  0.18552266644489D+04, -0.32658387038326D+04,&
                    0.15225504826021D+04,  0.57030555132894D+03, -0.21650753741526D+03,  0.63550677400708D+02,&
                    0.36122709236390D+02,  0.44099375581300D+01,  0.11855744540117D+00,  0.46845930415742D-02/)
  case(107)
    nfun = 36
    alf(1:nfun)=(/  0.45364201045187D+08,  0.27493455178901D+08,  0.16662700108425D+08,  0.10098606126318D+08,&
                    0.61203673492837D+07,  0.37093135450204D+07,  0.22480688151639D+07,  0.13624659485842D+07,&
                    0.82573693853586D+06,  0.50044662941568D+06,  0.30330098752465D+06,  0.18381878031797D+06,&
                    0.11140532140483D+06,  0.67518376608988D+05,  0.40920228247872D+05,  0.24800138332043D+05,&
                    0.15030386867905D+05,  0.91093253744880D+04,  0.55208032572654D+04,  0.33459413680397D+04,&
                    0.20278432533574D+04,  0.12289959111257D+04,  0.74484600674283D+03,  0.45142182226838D+03,&
                    0.27358898319296D+03,  0.16581150496543D+03,  0.10049182119117D+03,  0.60904134055255D+02,&
                    0.36911596397124D+02,  0.22370664483105D+02,  0.13557978474609D+02,  0.82169566512784D+01,&
                    0.49799737280475D+01,  0.30181658957864D+01,  0.18291914519917D+01,  0.11086008799950D+01/)
    coe(1:nfun)=(/  0.80415002123505D+08, -0.12526599417166D+09,  0.14697481376086D+09, -0.99144824306442D+08,&
                    0.76258221060440D+08, -0.38606454124863D+08,  0.28928860052860D+08, -0.11490424133945D+08,&
                    0.10260837603512D+08, -0.27825246632519D+07,  0.38566170059643D+07, -0.41156513199806D+06,&
                    0.15334689125645D+07,  0.11395926043570D+06,  0.61210530449438D+06,  0.10900719661120D+06,&
                    0.20271714876907D+06,  0.11662188663743D+05,  0.28926465782806D+05, -0.15061477866469D+05,&
                    0.18771479894668D+05,  0.22821599090605D+05,  0.17794174761386D+05, -0.83077237417896D+04,&
                   -0.38907598846444D+04,  0.96201705992130D+04,  0.12318389348375D+04, -0.31926785485361D+04,&
                    0.17114174384560D+04,  0.47650187480186D+03, -0.21273950454795D+03,  0.73128596115950D+02,&
                    0.34378345033313D+02,  0.37701265929165D+01,  0.69983880128177D-01,  0.45901975175298D-02/)
  case(108)
    nfun = 36
    alf(1:nfun)=(/  0.45364201045187D+08,  0.27493455178901D+08,  0.16662700108425D+08,  0.10098606126318D+08,&
                    0.61203673492837D+07,  0.37093135450204D+07,  0.22480688151639D+07,  0.13624659485842D+07,&
                    0.82573693853586D+06,  0.50044662941568D+06,  0.30330098752465D+06,  0.18381878031797D+06,&
                    0.11140532140483D+06,  0.67518376608988D+05,  0.40920228247872D+05,  0.24800138332043D+05,&
                    0.15030386867905D+05,  0.91093253744880D+04,  0.55208032572654D+04,  0.33459413680397D+04,&
                    0.20278432533574D+04,  0.12289959111257D+04,  0.74484600674283D+03,  0.45142182226838D+03,&
                    0.27358898319296D+03,  0.16581150496543D+03,  0.10049182119117D+03,  0.60904134055255D+02,&
                    0.36911596397124D+02,  0.22370664483105D+02,  0.13557978474609D+02,  0.82169566512784D+01,&
                    0.49799737280475D+01,  0.30181658957864D+01,  0.18291914519917D+01,  0.11086008799950D+01/)
    coe(1:nfun)=(/  0.79687931328796D+08, -0.12171377262280D+09,  0.14555550904755D+09, -0.96768673471680D+08,&
                    0.75678213352539D+08, -0.37518687999039D+08,  0.28821842963230D+08, -0.11018032307549D+08,&
                    0.10279989212721D+08, -0.25863143001719D+07,  0.38956920259323D+07, -0.33291544485831D+06,&
                    0.15554302030628D+07,  0.14275393602419D+06,  0.61810605679661D+06,  0.11492852563447D+06,&
                    0.20070504754646D+06,  0.10364858342424D+05,  0.27326292803738D+05, -0.14190819645118D+05,&
                    0.20628306592925D+05,  0.24031034667023D+05,  0.16883595158939D+05, -0.93225433228903D+04,&
                   -0.30140093171491D+04,  0.10088696299824D+05,  0.57796472279535D+03, -0.30670355952695D+04,&
                    0.18803233090726D+04,  0.37974442971005D+03, -0.20462026653452D+03,  0.81821915022776D+02,&
                    0.32444395000976D+02,  0.31759345446433D+01,  0.35570280026093D-01,  0.43745539106936D-02/)
  case(109)
    nfun = 36
    alf(1:nfun)=(/  0.45364201045187D+08,  0.27493455178901D+08,  0.16662700108425D+08,  0.10098606126318D+08,&
                    0.61203673492837D+07,  0.37093135450204D+07,  0.22480688151639D+07,  0.13624659485842D+07,&
                    0.82573693853586D+06,  0.50044662941568D+06,  0.30330098752465D+06,  0.18381878031797D+06,&
                    0.11140532140483D+06,  0.67518376608988D+05,  0.40920228247872D+05,  0.24800138332043D+05,&
                    0.15030386867905D+05,  0.91093253744880D+04,  0.55208032572654D+04,  0.33459413680397D+04,&
                    0.20278432533574D+04,  0.12289959111257D+04,  0.74484600674283D+03,  0.45142182226838D+03,&
                    0.27358898319296D+03,  0.16581150496543D+03,  0.10049182119117D+03,  0.60904134055255D+02,&
                    0.36911596397124D+02,  0.22370664483105D+02,  0.13557978474609D+02,  0.82169566512784D+01,&
                    0.49799737280475D+01,  0.30181658957864D+01,  0.18291914519917D+01,  0.11086008799950D+01/)
    coe(1:nfun)=(/  0.77656051750717D+08, -0.11234711409766D+09,  0.13754940667978D+09, -0.88761151141617D+08,&
                    0.71324427973389D+08, -0.34013188166451D+08,  0.27340758464001D+08, -0.97230671225605D+07,&
                    0.98528939728642D+07, -0.21245734110012D+07,  0.37955726452460D+07, -0.17263310746217D+06,&
                    0.15350336637398D+07,  0.19586316043556D+06,  0.61063039506477D+06,  0.12799308542237D+06,&
                    0.19428739586978D+06,  0.11282542920999D+05,  0.24544577381175D+05, -0.12434499007659D+05,&
                    0.22186388295695D+05,  0.25360866831848D+05,  0.15725081587906D+05, -0.10204550182032D+05,&
                   -0.20743342842135D+04,  0.10488576495547D+05, -0.97494465052499D+02, -0.28899048254244D+04,&
                    0.20261360350640D+04,  0.28205824748709D+03, -0.19239199527887D+03,  0.89426712724877D+02,&
                    0.30368969824762D+02,  0.26404778343006D+01,  0.11756499427065D-01,  0.42798920341696D-02/)
  case(110)
    nfun = 36
    alf(1:nfun)=(/  0.45364201045187D+08,  0.27493455178901D+08,  0.16662700108425D+08,  0.10098606126318D+08,&
                    0.61203673492837D+07,  0.37093135450204D+07,  0.22480688151639D+07,  0.13624659485842D+07,&
                    0.82573693853586D+06,  0.50044662941568D+06,  0.30330098752465D+06,  0.18381878031797D+06,&
                    0.11140532140483D+06,  0.67518376608988D+05,  0.40920228247872D+05,  0.24800138332043D+05,&
                    0.15030386867905D+05,  0.91093253744880D+04,  0.55208032572654D+04,  0.33459413680397D+04,&
                    0.20278432533574D+04,  0.12289959111257D+04,  0.74484600674283D+03,  0.45142182226838D+03,&
                    0.27358898319296D+03,  0.16581150496543D+03,  0.10049182119117D+03,  0.60904134055255D+02,&
                    0.36911596397124D+02,  0.22370664483105D+02,  0.13557978474609D+02,  0.82169566512784D+01,&
                    0.49799737280475D+01,  0.30181658957864D+01,  0.18291914519917D+01,  0.11086008799950D+01/)
    coe(1:nfun)=(/  0.73776768201492D+08, -0.98152304154724D+08,  0.12412548276059D+09, -0.76280731742432D+08,&
                    0.63974531735733D+08, -0.28578950104027D+08,  0.24738588398422D+08, -0.77568313719101D+07,&
                    0.90657148337536D+07, -0.14514633860912D+07,  0.35833574810052D+07,  0.55225680268049D+05,&
                    0.14781935016484D+07,  0.26979086466974D+06,  0.59160643720168D+06,  0.14688200679576D+06,&
                    0.18428792607322D+06,  0.13994076181725D+05,  0.20864127284721D+05, -0.99302727934904D+04,&
                    0.23506852684978D+05,  0.26748815428089D+05,  0.14352799467733D+05, -0.10951347869641D+05,&
                   -0.10752850789694D+04,  0.10810055904632D+05, -0.78313140928809D+03, -0.26653980206132D+04,&
                    0.21479438877032D+04,  0.18481198552498D+03, -0.17650034345785D+03,  0.95948906048410D+02,&
                    0.28182877800236D+02,  0.21620485548203D+01, -0.28933942348885D-02,  0.40611406740254D-02/)
  case(111)
    nfun = 36
    alf(1:nfun)=(/  0.45364201045187D+08,  0.27493455178901D+08,  0.16662700108425D+08,  0.10098606126318D+08,&
                    0.61203673492837D+07,  0.37093135450204D+07,  0.22480688151639D+07,  0.13624659485842D+07,&
                    0.82573693853586D+06,  0.50044662941568D+06,  0.30330098752465D+06,  0.18381878031797D+06,&
                    0.11140532140483D+06,  0.67518376608988D+05,  0.40920228247872D+05,  0.24800138332043D+05,&
                    0.15030386867905D+05,  0.91093253744880D+04,  0.55208032572654D+04,  0.33459413680397D+04,&
                    0.20278432533574D+04,  0.12289959111257D+04,  0.74484600674283D+03,  0.45142182226838D+03,&
                    0.27358898319296D+03,  0.16581150496543D+03,  0.10049182119117D+03,  0.60904134055255D+02,&
                    0.36911596397124D+02,  0.22370664483105D+02,  0.13557978474609D+02,  0.82169566512784D+01,&
                    0.49799737280475D+01,  0.30181658957864D+01,  0.18291914519917D+01,  0.11086008799950D+01/)
    coe(1:nfun)=(/  0.69752708938553D+08, -0.80060012686646D+08,  0.10627075665659D+09, -0.59822311179581D+08,&
                    0.54022330405960D+08, -0.21417099475960D+08,  0.21187209463840D+08, -0.52017598725567D+07,&
                    0.79708203566132D+07, -0.58602797665596D+06,  0.32751213327198D+07,  0.34148179757929D+06,&
                    0.13924772073972D+07,  0.36074044213396D+06,  0.56304772609678D+06,  0.17071001385626D+06,&
                    0.17104649563022D+06,  0.18289121864103D+05,  0.16386592701010D+05, -0.67196174265407D+04,&
                    0.24619206440263D+05,  0.28170450240132D+05,  0.12772710690316D+05, -0.11554345638712D+05,&
                   -0.19904069376760D+02,  0.11044729537493D+05, -0.14700155266101D+04, -0.23956572062856D+04,&
                    0.22437134242266D+04,  0.89762687350769D+02, -0.15740570793802D+03,  0.10135508227657D+03,&
                    0.25915039899093D+02,  0.17453177401385D+01, -0.11060169247003D-01,  0.37503149202180D-02/)
  case(112)
    nfun = 36
    alf(1:nfun)=(/  0.45364201045187D+08,  0.27493455178901D+08,  0.16662700108425D+08,  0.10098606126318D+08,&
                    0.61203673492837D+07,  0.37093135450204D+07,  0.22480688151639D+07,  0.13624659485842D+07,&
                    0.82573693853586D+06,  0.50044662941568D+06,  0.30330098752465D+06,  0.18381878031797D+06,&
                    0.11140532140483D+06,  0.67518376608988D+05,  0.40920228247872D+05,  0.24800138332043D+05,&
                    0.15030386867905D+05,  0.91093253744880D+04,  0.55208032572654D+04,  0.33459413680397D+04,&
                    0.20278432533574D+04,  0.12289959111257D+04,  0.74484600674283D+03,  0.45142182226838D+03,&
                    0.27358898319296D+03,  0.16581150496543D+03,  0.10049182119117D+03,  0.60904134055255D+02,&
                    0.36911596397124D+02,  0.22370664483105D+02,  0.13557978474609D+02,  0.82169566512784D+01,&
                    0.49799737280475D+01,  0.30181658957864D+01,  0.18291914519917D+01,  0.11086008799950D+01/)
    coe(1:nfun)=(/  0.64833824909164D+08, -0.58823491133118D+08,  0.84960799786011D+08, -0.40271056037567D+08,&
                    0.42052045951828D+08, -0.12903728135895D+08,  0.16881608681316D+08, -0.21660045148125D+07,&
                    0.66291163337345D+07,  0.43175529811668D+06,  0.28916986503353D+07,  0.67590006761384D+06,&
                    0.12815465119923D+07,  0.46669094809043D+06,  0.52608364964530D+06,  0.19857853577296D+06,&
                    0.15516532039127D+06,  0.23857316782348D+05,  0.11335132627271D+05, -0.29118228313792D+04,&
                    0.25569710523671D+05,  0.29570101661545D+05,  0.11019538263314D+05, -0.12011834966755D+05,&
                    0.10857696194807D+04,  0.11185222067597D+05, -0.21466032424929D+04, -0.20861220365510D+04,&
                    0.23132778623078D+04, -0.17205077506769D+01, -0.13569710973438D+03,  0.10567384774165D+03,&
                    0.23618330513851D+02,  0.13849323506652D+01, -0.16121169920496D-01,  0.32697036357590D-02/)
  case(113)
    nfun = 37
    alf(1:nfun)=(/  0.74850931724558D+08,  0.45364201045187D+08,  0.27493455178901D+08,  0.16662700108425D+08,&
                    0.10098606126318D+08,  0.61203673492837D+07,  0.37093135450204D+07,  0.22480688151639D+07,&
                    0.13624659485842D+07,  0.82573693853586D+06,  0.50044662941568D+06,  0.30330098752465D+06,&
                    0.18381878031797D+06,  0.11140532140483D+06,  0.67518376608988D+05,  0.40920228247872D+05,&
                    0.24800138332043D+05,  0.15030386867905D+05,  0.91093253744880D+04,  0.55208032572654D+04,&
                    0.33459413680397D+04,  0.20278432533574D+04,  0.12289959111257D+04,  0.74484600674283D+03,&
                    0.45142182226838D+03,  0.27358898319296D+03,  0.16581150496543D+03,  0.10049182119117D+03,&
                    0.60904134055255D+02,  0.36911596397124D+02,  0.22370664483105D+02,  0.13557978474609D+02,&
                    0.82169566512784D+01,  0.49799737280475D+01,  0.30181658957864D+01,  0.18291914519917D+01,&
                    0.11086008799950D+01/)
    coe(1:nfun)=(/ -0.66473985991974D+08,  0.14233074477417D+09, -0.86652024218414D+08,  0.84007196419617D+08,&
                   -0.26628702136917D+08,  0.31150835130875D+08, -0.38814123221664D+07,  0.11956590230377D+08,&
                    0.13113735407295D+07,  0.50827777178888D+07,  0.15991433890667D+07,  0.24033690849252D+07,&
                    0.10856397638271D+07,  0.11339318205887D+07,  0.58832833833951D+06,  0.48270893778816D+06,&
                    0.22890135695618D+06,  0.13743697639816D+06,  0.30367629960813D+05,  0.59266459951177D+04,&
                    0.13811435938850D+04,  0.26402501188511D+05,  0.30901782330127D+05,  0.91194714293981D+04,&
                   -0.12315751949960D+05,  0.22306322472353D+04,  0.11226008519369D+05, -0.28017585354797D+04,&
                   -0.17415131447939D+04,  0.23558537272034D+04, -0.88481184647215D+02, -0.11143584771505D+03,&
                    0.10871290458345D+03,  0.21315835715913D+02,  0.10924728516097D+01, -0.17968956369771D-01,&
                    0.29014051364129D-02/)
  case(114)
    nfun = 37
    alf(1:nfun)=(/  0.74850931724558D+08,  0.45364201045187D+08,  0.27493455178901D+08,  0.16662700108425D+08,&
                    0.10098606126318D+08,  0.61203673492837D+07,  0.37093135450204D+07,  0.22480688151639D+07,&
                    0.13624659485842D+07,  0.82573693853586D+06,  0.50044662941568D+06,  0.30330098752465D+06,&
                    0.18381878031797D+06,  0.11140532140483D+06,  0.67518376608988D+05,  0.40920228247872D+05,&
                    0.24800138332043D+05,  0.15030386867905D+05,  0.91093253744880D+04,  0.55208032572654D+04,&
                    0.33459413680397D+04,  0.20278432533574D+04,  0.12289959111257D+04,  0.74484600674283D+03,&
                    0.45142182226838D+03,  0.27358898319296D+03,  0.16581150496543D+03,  0.10049182119117D+03,&
                    0.60904134055255D+02,  0.36911596397124D+02,  0.22370664483105D+02,  0.13557978474609D+02,&
                    0.82169566512784D+01,  0.49799737280475D+01,  0.30181658957864D+01,  0.18291914519917D+01,&
                    0.11086008799950D+01/)
    coe(1:nfun)=(/ -0.25497909097137D+08,  0.84000913428650D+08, -0.27466831493805D+08,  0.42670783980469D+08,&
                    0.29228963427429D+07,  0.14665589201202D+08,  0.70046157987137D+07,  0.65702097274055D+07,&
                    0.49874031317997D+07,  0.34559335478401D+07,  0.28071774611220D+07,  0.19342282408748D+07,&
                    0.14778015366848D+07,  0.99668931819189D+06,  0.70970557731724D+06,  0.43667086433798D+06,&
                    0.26074448859893D+06,  0.11839426380751D+06,  0.37415974892087D+05,  0.40848550653458D+03,&
                    0.60544367403146D+04,  0.27157288210934D+05,  0.32128706204882D+05,  0.70974947434450D+04,&
                   -0.12465981887302D+05,  0.34092778333018D+04,  0.11162936291470D+05, -0.34265557016966D+04,&
                   -0.13668295857447D+04,  0.23715686235383D+04, -0.16909453866344D+03, -0.85453296866410D+02,&
                    0.11062347008299D+03,  0.19051459559556D+02,  0.85375229757619D+00, -0.18673885262956D-01,&
                    0.25966779671833D-02/)
  case(115)
    nfun = 37
    alf(1:nfun)=(/  0.74850931724558D+08,  0.45364201045187D+08,  0.27493455178901D+08,  0.16662700108425D+08,&
                    0.10098606126318D+08,  0.61203673492837D+07,  0.37093135450204D+07,  0.22480688151639D+07,&
                    0.13624659485842D+07,  0.82573693853586D+06,  0.50044662941568D+06,  0.30330098752465D+06,&
                    0.18381878031797D+06,  0.11140532140483D+06,  0.67518376608988D+05,  0.40920228247872D+05,&
                    0.24800138332043D+05,  0.15030386867905D+05,  0.91093253744880D+04,  0.55208032572654D+04,&
                    0.33459413680397D+04,  0.20278432533574D+04,  0.12289959111257D+04,  0.74484600674283D+03,&
                    0.45142182226838D+03,  0.27358898319296D+03,  0.16581150496543D+03,  0.10049182119117D+03,&
                    0.60904134055255D+02,  0.36911596397124D+02,  0.22370664483105D+02,  0.13557978474609D+02,&
                    0.82169566512784D+01,  0.49799737280475D+01,  0.30181658957864D+01,  0.18291914519917D+01,&
                    0.11086008799950D+01/)
    coe(1:nfun)=(/  0.16109229117050D+08,  0.24905025033752D+08,  0.33064322577698D+08,  0.48656478527832D+06,&
                    0.33320972056198D+08, -0.22135479015808D+07,  0.18236549009094D+08,  0.10535467482185D+07,&
                    0.87863128949394D+07,  0.17903524319143D+07,  0.40555308848548D+07,  0.14534000971298D+07,&
                    0.18820949419363D+07,  0.85509412719870D+06,  0.83398949390274D+06,  0.38864187890843D+06,&
                    0.29307705169480D+06,  0.98605673387177D+05,  0.44788883909389D+05, -0.50669749703668D+04,&
                    0.11025321307415D+05,  0.27863180726897D+05,  0.33208943732299D+05,  0.49837763175033D+04,&
                   -0.12457478674857D+05,  0.46086776896151D+04,  0.10993942573247D+05, -0.40101696525528D+04,&
                   -0.96919895295172D+03,  0.23616936008676D+04, -0.24278939015891D+03, -0.58202804301291D+02,&
                    0.11143620662232D+03,  0.16862234240162D+02,  0.66116190895741D+00, -0.18574145183607D-01,&
                    0.23402813801354D-02/)
  case(116)
    nfun = 37
    alf(1:nfun)=(/  0.74850931724558D+08,  0.45364201045187D+08,  0.27493455178901D+08,  0.16662700108425D+08,&
                    0.10098606126318D+08,  0.61203673492837D+07,  0.37093135450204D+07,  0.22480688151639D+07,&
                    0.13624659485842D+07,  0.82573693853586D+06,  0.50044662941568D+06,  0.30330098752465D+06,&
                    0.18381878031797D+06,  0.11140532140483D+06,  0.67518376608988D+05,  0.40920228247872D+05,&
                    0.24800138332043D+05,  0.15030386867905D+05,  0.91093253744880D+04,  0.55208032572654D+04,&
                    0.33459413680397D+04,  0.20278432533574D+04,  0.12289959111257D+04,  0.74484600674283D+03,&
                    0.45142182226838D+03,  0.27358898319296D+03,  0.16581150496543D+03,  0.10049182119117D+03,&
                    0.60904134055255D+02,  0.36911596397124D+02,  0.22370664483105D+02,  0.13557978474609D+02,&
                    0.82169566512784D+01,  0.49799737280475D+01,  0.30181658957864D+01,  0.18291914519917D+01,&
                    0.11086008799950D+01/)
    coe(1:nfun)=(/  0.56273527727829D+08, -0.30646318321442D+08,  0.90523620046234D+08, -0.39080852403961D+08,&
                    0.62277152833328D+08, -0.18066081674744D+08,  0.28991104658791D+08, -0.41174542693787D+07,&
                    0.12443640538887D+08,  0.23583203055096D+06,  0.52617724683642D+07,  0.10071147820599D+07,&
                    0.22730017802529D+07,  0.72320062913299D+06,  0.95332095301992D+06,  0.34290868052116D+06,&
                    0.32346328069982D+06,  0.79443362418741D+05,  0.51752025486343D+05, -0.10054353193879D+05,&
                    0.16052394947872D+05,  0.28635362481509D+05,  0.34054507430523D+05,  0.28385767608180D+04,&
                   -0.12305636525831D+05,  0.58262423469711D+04,  0.10713595448229D+05, -0.45408763863974D+04,&
                   -0.55638150274765D+03,  0.23280155768492D+04, -0.30900173755162D+03, -0.30118795480304D+02,&
                    0.11119865590013D+03,  0.14777692228789D+02,  0.50775190211220D+00, -0.17916387404171D-01,&
                    0.21131234485168D-02/)
  case(117)
    nfun = 37
    alf(1:nfun)=(/  0.74850931724558D+08,  0.45364201045187D+08,  0.27493455178901D+08,  0.16662700108425D+08,&
                    0.10098606126318D+08,  0.61203673492837D+07,  0.37093135450204D+07,  0.22480688151639D+07,&
                    0.13624659485842D+07,  0.82573693853586D+06,  0.50044662941568D+06,  0.30330098752465D+06,&
                    0.18381878031797D+06,  0.11140532140483D+06,  0.67518376608988D+05,  0.40920228247872D+05,&
                    0.24800138332043D+05,  0.15030386867905D+05,  0.91093253744880D+04,  0.55208032572654D+04,&
                    0.33459413680397D+04,  0.20278432533574D+04,  0.12289959111257D+04,  0.74484600674283D+03,&
                    0.45142182226838D+03,  0.27358898319296D+03,  0.16581150496543D+03,  0.10049182119117D+03,&
                    0.60904134055255D+02,  0.36911596397124D+02,  0.22370664483105D+02,  0.13557978474609D+02,&
                    0.82169566512784D+01,  0.49799737280475D+01,  0.30181658957864D+01,  0.18291914519917D+01,&
                    0.11086008799950D+01/)
    coe(1:nfun)=(/  0.96371689989166D+08, -0.85319925086700D+08,  0.14711896344989D+09, -0.77275372201202D+08,&
                    0.90498747018387D+08, -0.33260686980972D+08,  0.39453371579880D+08, -0.90248926922531D+07,&
                    0.16010675632853D+08, -0.12374612086029D+07,  0.64431456216497D+07,  0.58769184447241D+06,&
                    0.26541894374282D+07,  0.59887700427568D+06,  0.10690034459468D+07,  0.29851060921827D+06,&
                    0.35246456100617D+06,  0.60593817039959D+05,  0.58515908953533D+05, -0.14636130257335D+05,&
                    0.21192098495165D+05,  0.29412905163195D+05,  0.34681494630654D+05,  0.66398712552106D+03,&
                   -0.11992483277683D+05,  0.70377830191946D+04,  0.10330616971098D+05, -0.50148838509644D+04,&
                   -0.13262476829617D+03,  0.22710629277519D+04, -0.36673523415563D+03, -0.18059196456306D+01,&
                    0.11002260407427D+03,  0.12810209900304D+02,  0.38926824028022D+00, -0.17239186038978D-01,&
                    0.19377641915810D-02/)
  case(118)
    nfun = 37
    alf(1:nfun)=(/  0.74850931724558D+08,  0.45364201045187D+08,  0.27493455178901D+08,  0.16662700108425D+08,&
                    0.10098606126318D+08,  0.61203673492837D+07,  0.37093135450204D+07,  0.22480688151639D+07,&
                    0.13624659485842D+07,  0.82573693853586D+06,  0.50044662941568D+06,  0.30330098752465D+06,&
                    0.18381878031797D+06,  0.11140532140483D+06,  0.67518376608988D+05,  0.40920228247872D+05,&
                    0.24800138332043D+05,  0.15030386867905D+05,  0.91093253744880D+04,  0.55208032572654D+04,&
                    0.33459413680397D+04,  0.20278432533574D+04,  0.12289959111257D+04,  0.74484600674283D+03,&
                    0.45142182226838D+03,  0.27358898319296D+03,  0.16581150496543D+03,  0.10049182119117D+03,&
                    0.60904134055255D+02,  0.36911596397124D+02,  0.22370664483105D+02,  0.13557978474609D+02,&
                    0.82169566512784D+01,  0.49799737280475D+01,  0.30181658957864D+01,  0.18291914519917D+01,&
                    0.11086008799950D+01/)
    coe(1:nfun)=(/  0.13332441439940D+09, -0.13292054992203D+09,  0.19767940028568D+09, -0.11090606092899D+09,&
                    0.11617891137413D+09, -0.46772795548157D+08,  0.49112760319489D+08, -0.13403637089462D+08,&
                    0.19339503905104D+08, -0.25386059284592D+07,  0.75508834830251D+07,  0.21982515664864D+06,&
                    0.30132402419416D+07,  0.48913154456902D+06,  0.11765074325371D+07,  0.25824772375613D+06,&
                    0.37833110709566D+06,  0.43083287134528D+05,  0.64566384530146D+05, -0.18513265806641D+05,&
                    0.26295422626469D+05,  0.30246355918323D+05,  0.35041052549356D+05, -0.14944544913271D+04,&
                   -0.11529163900958D+05,  0.82365941900268D+04,  0.98435661532416D+04, -0.54218577325545D+04,&
                    0.29347583791698D+03,  0.21938312859351D+04, -0.41604778246687D+03,  0.26447557414358D+02,&
                    0.10795408128474D+03,  0.10985013004526D+02,  0.29776156372759D+00, -0.16268359769072D-01,&
                    0.17652067257217D-02/)
end select

return
end

!-----------------------------------------------------------------------------------------------------------------------------------
!---  EDF data library: ncore = 92, iz =104-120
!-----------------------------------------------------------------------------------------------------------------------------------
subroutine EDFLIB92(iz,nfun,alf,coe)
implicit real(kind=8) (a-h,o-z)
real(kind=8) :: alf(*),coe(*)

select case(iz)
  case(104)
    nfun = 39
    alf(1:nfun)=(/  0.45364201045187D+08,  0.27493455178901D+08,  0.16662700108425D+08,  0.10098606126318D+08,&
                    0.61203673492837D+07,  0.37093135450204D+07,  0.22480688151639D+07,  0.13624659485842D+07,&
                    0.82573693853586D+06,  0.50044662941568D+06,  0.30330098752465D+06,  0.18381878031797D+06,&
                    0.11140532140483D+06,  0.67518376608988D+05,  0.40920228247872D+05,  0.24800138332043D+05,&
                    0.15030386867905D+05,  0.91093253744880D+04,  0.55208032572654D+04,  0.33459413680397D+04,&
                    0.20278432533574D+04,  0.12289959111257D+04,  0.74484600674283D+03,  0.45142182226838D+03,&
                    0.27358898319296D+03,  0.16581150496543D+03,  0.10049182119117D+03,  0.60904134055255D+02,&
                    0.36911596397124D+02,  0.22370664483105D+02,  0.13557978474609D+02,  0.82169566512784D+01,&
                    0.49799737280475D+01,  0.30181658957864D+01,  0.18291914519917D+01,  0.11086008799950D+01,&
                    0.67187932120909D+00,  0.40719958861157D+00,  0.24678762946156D+00/)
    coe(1:nfun)=(/  0.83964718243233D+08, -0.10827403953196D+09,  0.11429758668667D+09, -0.73168327044289D+08,&
                    0.55188003550468D+08, -0.27014838668383D+08,  0.20599828478711D+08, -0.77431811080246D+07,&
                    0.73895455030007D+07, -0.17183501396797D+07,  0.28615016351920D+07, -0.13160454803991D+06,&
                    0.11951780753014D+07,  0.18426896169019D+06,  0.50767494838984D+06,  0.13743104886817D+06,&
                    0.18069818436453D+06,  0.29678705393519D+05,  0.25977838148603D+05, -0.12179731635175D+05,&
                    0.11343459611323D+05,  0.20124777383807D+05,  0.18983361557638D+05, -0.45827250512997D+04,&
                   -0.60978133095553D+04,  0.79020559378661D+04,  0.29280137915920D+04, -0.32794400504466D+04,&
                    0.11267342623162D+04,  0.80631552775410D+03, -0.23275575557031D+03,  0.16453980092969D+02,&
                    0.42231903155252D+02,  0.12522610021522D+02,  0.30082293016475D+01,  0.64412082409862D+00,&
                    0.87759568821411D-01,  0.60842936743484D-02,  0.25076168288685D-03/)
  case(105)
    nfun = 39
    alf(1:nfun)=(/  0.45364201045187D+08,  0.27493455178901D+08,  0.16662700108425D+08,  0.10098606126318D+08,&
                    0.61203673492837D+07,  0.37093135450204D+07,  0.22480688151639D+07,  0.13624659485842D+07,&
                    0.82573693853586D+06,  0.50044662941568D+06,  0.30330098752465D+06,  0.18381878031797D+06,&
                    0.11140532140483D+06,  0.67518376608988D+05,  0.40920228247872D+05,  0.24800138332043D+05,&
                    0.15030386867905D+05,  0.91093253744880D+04,  0.55208032572654D+04,  0.33459413680397D+04,&
                    0.20278432533574D+04,  0.12289959111257D+04,  0.74484600674283D+03,  0.45142182226838D+03,&
                    0.27358898319296D+03,  0.16581150496543D+03,  0.10049182119117D+03,  0.60904134055255D+02,&
                    0.36911596397124D+02,  0.22370664483105D+02,  0.13557978474609D+02,  0.82169566512784D+01,&
                    0.49799737280475D+01,  0.30181658957864D+01,  0.18291914519917D+01,  0.11086008799950D+01,&
                    0.67187932120909D+00,  0.40719958861157D+00,  0.24678762946156D+00/)
    coe(1:nfun)=(/  0.85332138576115D+08, -0.11565392406123D+09,  0.12567031486488D+09, -0.81831102537575D+08,&
                    0.61992123828655D+08, -0.30749705890276D+08,  0.23253752204846D+08, -0.89451185801950D+07,&
                    0.83004032304444D+07, -0.20576634589996D+07,  0.31770615554984D+07, -0.22087941342616D+06,&
                    0.13037366785296D+07,  0.16202322187634D+06,  0.54163876192668D+06,  0.12875161227999D+06,&
                    0.18794224685174D+06,  0.23988736894885D+05,  0.26795978339666D+05, -0.13231213039283D+05,&
                    0.13669909307106D+05,  0.21104808768858D+05,  0.18705543257919D+05, -0.58429584475626D+04,&
                   -0.54745656286257D+04,  0.85396976921900D+04,  0.23739077375062D+04, -0.33090024962065D+04,&
                    0.13544469415726D+04,  0.72868307104091D+03, -0.24759933262280D+03,  0.27835182805616D+02,&
                    0.42715702140632D+02,  0.12496652567271D+02,  0.28369561228982D+01,  0.54016497861562D+00,&
                    0.54988475989006D-01,  0.27462791413164D-02,  0.11661629844939D-03/)
  case(106)
    nfun = 38
    alf(1:nfun)=(/  0.45364201045187D+08,  0.27493455178901D+08,  0.16662700108425D+08,  0.10098606126318D+08,&
                    0.61203673492837D+07,  0.37093135450204D+07,  0.22480688151639D+07,  0.13624659485842D+07,&
                    0.82573693853586D+06,  0.50044662941568D+06,  0.30330098752465D+06,  0.18381878031797D+06,&
                    0.11140532140483D+06,  0.67518376608988D+05,  0.40920228247872D+05,  0.24800138332043D+05,&
                    0.15030386867905D+05,  0.91093253744880D+04,  0.55208032572654D+04,  0.33459413680397D+04,&
                    0.20278432533574D+04,  0.12289959111257D+04,  0.74484600674283D+03,  0.45142182226838D+03,&
                    0.27358898319296D+03,  0.16581150496543D+03,  0.10049182119117D+03,  0.60904134055255D+02,&
                    0.36911596397124D+02,  0.22370664483105D+02,  0.13557978474609D+02,  0.82169566512784D+01,&
                    0.49799737280475D+01,  0.30181658957864D+01,  0.18291914519917D+01,  0.11086008799950D+01,&
                    0.67187932120909D+00,  0.40719958861157D+00/)
    coe(1:nfun)=(/  0.97601304920128D+08, -0.12958049572989D+09,  0.13902314991460D+09, -0.89922752206955D+08,&
                    0.67847836855515D+08, -0.33598035423298D+08,  0.25343020799643D+08, -0.97730479869862D+07,&
                    0.90120103497305D+07, -0.22718653530431D+07,  0.34287620417469D+07, -0.26910685912478D+06,&
                    0.13912233326409D+07,  0.15263025865901D+06,  0.56871947064745D+06,  0.12352574201800D+06,&
                    0.19296049665813D+06,  0.19261162867829D+05,  0.27064681506123D+05, -0.13800797713153D+05,&
                    0.15929096246870D+05,  0.22128314276352D+05,  0.18244621877482D+05, -0.70525551475126D+04,&
                   -0.47523569850302D+04,  0.91230560328232D+04,  0.17799378888548D+04, -0.32887024727228D+04,&
                    0.15713221226061D+04,  0.64239534707789D+03, -0.25771406132737D+03,  0.39073634190594D+02,&
                    0.43107325280043D+02,  0.12419603277962D+02,  0.25960079256631D+01,  0.44114135212749D+00,&
                    0.29820293814637D-01,  0.18786434175464D-02/)
  case(107)
    nfun = 38
    alf(1:nfun)=(/  0.45364201045187D+08,  0.27493455178901D+08,  0.16662700108425D+08,  0.10098606126318D+08,&
                    0.61203673492837D+07,  0.37093135450204D+07,  0.22480688151639D+07,  0.13624659485842D+07,&
                    0.82573693853586D+06,  0.50044662941568D+06,  0.30330098752465D+06,  0.18381878031797D+06,&
                    0.11140532140483D+06,  0.67518376608988D+05,  0.40920228247872D+05,  0.24800138332043D+05,&
                    0.15030386867905D+05,  0.91093253744880D+04,  0.55208032572654D+04,  0.33459413680397D+04,&
                    0.20278432533574D+04,  0.12289959111257D+04,  0.74484600674283D+03,  0.45142182226838D+03,&
                    0.27358898319296D+03,  0.16581150496543D+03,  0.10049182119117D+03,  0.60904134055255D+02,&
                    0.36911596397124D+02,  0.22370664483105D+02,  0.13557978474609D+02,  0.82169566512784D+01,&
                    0.49799737280475D+01,  0.30181658957864D+01,  0.18291914519917D+01,  0.11086008799950D+01,&
                    0.67187932120909D+00,  0.40719958861157D+00/)
    coe(1:nfun)=(/  0.91466431611053D+08, -0.12856824104767D+09,  0.14427154699718D+09, -0.94494832548004D+08,&
                    0.72351025561775D+08, -0.35907946887814D+08,  0.27257190168262D+08, -0.10501946022288D+08,&
                    0.96941940504112D+07, -0.24620206760087D+07,  0.36771133037741D+07, -0.31143004093623D+06,&
                    0.14777919236236D+07,  0.14487957365650D+06,  0.59495384564346D+06,  0.11851673084605D+06,&
                    0.19744780202667D+06,  0.14580727409899D+05,  0.27311085602188D+05, -0.14168349140845D+05,&
                    0.18278545299810D+05,  0.23092411429091D+05,  0.17647502105993D+05, -0.82221773318141D+04,&
                   -0.39212597358103D+04,  0.96355092260141D+04,  0.11575174439283D+04, -0.32204245156719D+04,&
                    0.17755738336260D+04,  0.54804863527444D+03, -0.26259951857714D+03,  0.49809558177075D+02,&
                    0.43527375517852D+02,  0.12199232695317D+02,  0.23299404715029D+01,  0.34352286956988D+00,&
                    0.15438931402699D-01,  0.11950850487743D-02/)
  case(108)
    nfun = 38
    alf(1:nfun)=(/  0.45364201045187D+08,  0.27493455178901D+08,  0.16662700108425D+08,  0.10098606126318D+08,&
                    0.61203673492837D+07,  0.37093135450204D+07,  0.22480688151639D+07,  0.13624659485842D+07,&
                    0.82573693853586D+06,  0.50044662941568D+06,  0.30330098752465D+06,  0.18381878031797D+06,&
                    0.11140532140483D+06,  0.67518376608988D+05,  0.40920228247872D+05,  0.24800138332043D+05,&
                    0.15030386867905D+05,  0.91093253744880D+04,  0.55208032572654D+04,  0.33459413680397D+04,&
                    0.20278432533574D+04,  0.12289959111257D+04,  0.74484600674283D+03,  0.45142182226838D+03,&
                    0.27358898319296D+03,  0.16581150496543D+03,  0.10049182119117D+03,  0.60904134055255D+02,&
                    0.36911596397124D+02,  0.22370664483105D+02,  0.13557978474609D+02,  0.82169566512784D+01,&
                    0.49799737280475D+01,  0.30181658957864D+01,  0.18291914519917D+01,  0.11086008799950D+01,&
                    0.67187932120909D+00,  0.40719958861157D+00/)
    coe(1:nfun)=(/  0.91739496242653D+08, -0.12951982994127D+09,  0.14830890842442D+09, -0.96695834676811D+08,&
                    0.74931221853378D+08, -0.36803560091278D+08,  0.28326792871550D+08, -0.10707694095578D+08,&
                    0.10097338448365D+08, -0.24813025774112D+07,  0.38364361224055D+07, -0.29969430376053D+06,&
                    0.15369208656257D+07,  0.15304628258279D+06,  0.61239694300216D+06,  0.11809188430677D+06,&
                    0.19895547053205D+06,  0.11330680400196D+05,  0.26794814052887D+05, -0.13899892595062D+05,&
                    0.20470436894316D+05,  0.24114905484834D+05,  0.16842001563710D+05, -0.92954701029580D+04,&
                   -0.30091629987076D+04,  0.10080705406867D+05,  0.50800159251227D+03, -0.31009655232868D+04,&
                    0.19624137706388D+04,  0.44805740622402D+03, -0.26276177362898D+03,  0.60185647228303D+02,&
                    0.43885003836329D+02,  0.11825595719643D+02,  0.20408978219610D+01,  0.26262312328305D+00,&
                    0.72603062945618D-02,  0.87118564925781D-03/)
  case(109)
    nfun = 38
    alf(1:nfun)=(/  0.45364201045187D+08,  0.27493455178901D+08,  0.16662700108425D+08,  0.10098606126318D+08,&
                    0.61203673492837D+07,  0.37093135450204D+07,  0.22480688151639D+07,  0.13624659485842D+07,&
                    0.82573693853586D+06,  0.50044662941568D+06,  0.30330098752465D+06,  0.18381878031797D+06,&
                    0.11140532140483D+06,  0.67518376608988D+05,  0.40920228247872D+05,  0.24800138332043D+05,&
                    0.15030386867905D+05,  0.91093253744880D+04,  0.55208032572654D+04,  0.33459413680397D+04,&
                    0.20278432533574D+04,  0.12289959111257D+04,  0.74484600674283D+03,  0.45142182226838D+03,&
                    0.27358898319296D+03,  0.16581150496543D+03,  0.10049182119117D+03,  0.60904134055255D+02,&
                    0.36911596397124D+02,  0.22370664483105D+02,  0.13557978474609D+02,  0.82169566512784D+01,&
                    0.49799737280475D+01,  0.30181658957864D+01,  0.18291914519917D+01,  0.11086008799950D+01,&
                    0.67187932120909D+00,  0.40719958861157D+00/)
    coe(1:nfun)=(/  0.94123061747383D+08, -0.12757057438174D+09,  0.14749078675447D+09, -0.94078524383720D+08,&
                    0.74112077210403D+08, -0.35450355113262D+08,  0.28103536031702D+08, -0.10130958254831D+08,&
                    0.10075076701344D+08, -0.22460738972645D+07,  0.38625352344830D+07, -0.20958003452992D+06,&
                    0.15554823100249D+07,  0.18454117102197D+06,  0.61690885892291D+06,  0.12450891297339D+06,&
                    0.19622427795222D+06,  0.10203521638252D+05,  0.25147892106105D+05, -0.12773854055269D+05,&
                    0.22379057669376D+05,  0.25249286928365D+05,  0.15793291892933D+05, -0.10238453291119D+05,&
                   -0.20326383964795D+04,  0.10455688974841D+05, -0.16289498626936D+03, -0.29294560454306D+04,&
                    0.21278086427944D+04,  0.34491919823362D+03, -0.25866294422444D+03,  0.70119511832681D+02,&
                    0.44122939042994D+02,  0.11354318321833D+02,  0.17441819788373D+01,  0.19473034322132D+00,&
                    0.23745083023883D-02,  0.67912226610467D-03/)
  case(110)
    nfun = 38
    alf(1:nfun)=(/  0.45364201045187D+08,  0.27493455178901D+08,  0.16662700108425D+08,  0.10098606126318D+08,&
                    0.61203673492837D+07,  0.37093135450204D+07,  0.22480688151639D+07,  0.13624659485842D+07,&
                    0.82573693853586D+06,  0.50044662941568D+06,  0.30330098752465D+06,  0.18381878031797D+06,&
                    0.11140532140483D+06,  0.67518376608988D+05,  0.40920228247872D+05,  0.24800138332043D+05,&
                    0.15030386867905D+05,  0.91093253744880D+04,  0.55208032572654D+04,  0.33459413680397D+04,&
                    0.20278432533574D+04,  0.12289959111257D+04,  0.74484600674283D+03,  0.45142182226838D+03,&
                    0.27358898319296D+03,  0.16581150496543D+03,  0.10049182119117D+03,  0.60904134055255D+02,&
                    0.36911596397124D+02,  0.22370664483105D+02,  0.13557978474609D+02,  0.82169566512784D+01,&
                    0.49799737280475D+01,  0.30181658957864D+01,  0.18291914519917D+01,  0.11086008799950D+01,&
                    0.67187932120909D+00,  0.40719958861157D+00/)
    coe(1:nfun)=(/  0.94947916652168D+08, -0.12106467607828D+09,  0.14143539149239D+09, -0.87087026139534D+08,&
                    0.70349550926079D+08, -0.32196052561523D+08,  0.26774023348019D+08, -0.88909445906992D+07,&
                    0.96971031577058D+07, -0.18018736017916D+07,  0.37778613712120D+07, -0.52619471016645D+05,&
                    0.15380039879596D+07,  0.23663128626284D+06,  0.60999588120833D+06,  0.13668221431012D+06,&
                    0.18994874398319D+06,  0.10849675115366D+05,  0.22613418118024D+05, -0.10905942187347D+05,&
                    0.24053265334784D+05,  0.26440131347879D+05,  0.14531680202555D+05, -0.11046528439017D+05,&
                   -0.99656012243824D+03,  0.10751454606646D+05, -0.84412505826096D+03, -0.27095223562418D+04,&
                    0.22704319345008D+04,  0.23984000773229D+03, -0.25044784185540D+03,  0.79593522840521D+02,&
                    0.44223252851279D+02,  0.10754539223077D+02,  0.14551899224727D+01,  0.14276788523765D+00,&
                   -0.21980602441118D-03,  0.55669926378243D-03/)
  case(111)
    nfun = 38
    alf(1:nfun)=(/  0.45364201045187D+08,  0.27493455178901D+08,  0.16662700108425D+08,  0.10098606126318D+08,&
                    0.61203673492837D+07,  0.37093135450204D+07,  0.22480688151639D+07,  0.13624659485842D+07,&
                    0.82573693853586D+06,  0.50044662941568D+06,  0.30330098752465D+06,  0.18381878031797D+06,&
                    0.11140532140483D+06,  0.67518376608988D+05,  0.40920228247872D+05,  0.24800138332043D+05,&
                    0.15030386867905D+05,  0.91093253744880D+04,  0.55208032572654D+04,  0.33459413680397D+04,&
                    0.20278432533574D+04,  0.12289959111257D+04,  0.74484600674283D+03,  0.45142182226838D+03,&
                    0.27358898319296D+03,  0.16581150496543D+03,  0.10049182119117D+03,  0.60904134055255D+02,&
                    0.36911596397124D+02,  0.22370664483105D+02,  0.13557978474609D+02,  0.82169566512784D+01,&
                    0.49799737280475D+01,  0.30181658957864D+01,  0.18291914519917D+01,  0.11086008799950D+01,&
                    0.67187932120909D+00,  0.40719958861157D+00/)
    coe(1:nfun)=(/  0.94744572603958D+08, -0.10953618263952D+09,  0.12999770763196D+09, -0.75465694204399D+08,&
                    0.63577455426197D+08, -0.26973658497921D+08,  0.24356988171812D+08, -0.69839151600075D+07,&
                    0.89675706662426D+07, -0.11408977788849D+07,  0.35835636150913D+07,  0.17029258556426D+06,&
                    0.14874572361360D+07,  0.30806855880621D+06,  0.59225853021424D+06,  0.15450982124029D+06,&
                    0.18003448416868D+06,  0.13299540085230D+05,  0.19159559002804D+05, -0.82635960184634D+04,&
                    0.25481473459702D+05,  0.27685848160216D+05,  0.13050404225584D+05, -0.11703968241339D+05,&
                    0.91966305494891D+02,  0.10961875370952D+05, -0.15280670883838D+04, -0.24423210764948D+04,&
                    0.23875650004742D+04,  0.13472937435077D+03, -0.23847596446793D+03,  0.88594592388510D+02,&
                    0.44130617774692D+02,  0.10043792420084D+02,  0.11872874090012D+01,  0.10439933591417D+00,&
                   -0.15226409202782D-02,  0.47006896265600D-03/)
  case(112)
    nfun = 40
    alf(1:nfun)=(/  0.12350403734552D+09,  0.74850931724558D+08,  0.45364201045187D+08,  0.27493455178901D+08,&
                    0.16662700108425D+08,  0.10098606126318D+08,  0.61203673492837D+07,  0.37093135450204D+07,&
                    0.22480688151639D+07,  0.13624659485842D+07,  0.82573693853586D+06,  0.50044662941568D+06,&
                    0.30330098752465D+06,  0.18381878031797D+06,  0.11140532140483D+06,  0.67518376608988D+05,&
                    0.40920228247872D+05,  0.24800138332043D+05,  0.15030386867905D+05,  0.91093253744880D+04,&
                    0.55208032572654D+04,  0.33459413680397D+04,  0.20278432533574D+04,  0.12289959111257D+04,&
                    0.74484600674283D+03,  0.45142182226838D+03,  0.27358898319296D+03,  0.16581150496543D+03,&
                    0.10049182119117D+03,  0.60904134055255D+02,  0.36911596397124D+02,  0.22370664483105D+02,&
                    0.13557978474609D+02,  0.82169566512784D+01,  0.49799737280475D+01,  0.30181658957864D+01,&
                    0.18291914519917D+01,  0.11086008799950D+01,  0.67187932120909D+00,  0.40719958861157D+00/)
    coe(1:nfun)=(/  0.46159854346283D+08, -0.15388393092792D+09,  0.25139671190805D+09, -0.18848814474625D+09,&
                    0.15694129305305D+09, -0.76365301017067D+08,  0.59977468717983D+08, -0.21933565347885D+08,&
                    0.21570627244921D+08, -0.46530382769556D+07,  0.79726459361019D+07, -0.30108406756139D+06,&
                    0.32943551827388D+07,  0.45391626556897D+06,  0.14042429956197D+07,  0.39879740627125D+06,&
                    0.56368921553272D+06,  0.17773874185304D+06,  0.16672121040676D+06,  0.17445287784388D+05,&
                    0.14896713834127D+05, -0.48934346698225D+04,  0.26675143542498D+05,  0.28950110360836D+05,&
                    0.11373406779888D+05, -0.12203004152012D+05,  0.12234318604242D+04,  0.11081523034086D+05,&
                   -0.22043014922471D+04, -0.21323987148475D+04,  0.24783739089482D+04,  0.31130852320131D+02,&
                   -0.22324051005486D+03,  0.97156096067280D+02,  0.43816476400884D+02,  0.92401046591207D+01,&
                    0.94811113530005D+00,  0.76629908429027D-01, -0.21258554262928D-02,  0.40406805353932D-03/)
  case(113)
    nfun = 40
    alf(1:nfun)=(/  0.12350403734552D+09,  0.74850931724558D+08,  0.45364201045187D+08,  0.27493455178901D+08,&
                    0.16662700108425D+08,  0.10098606126318D+08,  0.61203673492837D+07,  0.37093135450204D+07,&
                    0.22480688151639D+07,  0.13624659485842D+07,  0.82573693853586D+06,  0.50044662941568D+06,&
                    0.30330098752465D+06,  0.18381878031797D+06,  0.11140532140483D+06,  0.67518376608988D+05,&
                    0.40920228247872D+05,  0.24800138332043D+05,  0.15030386867905D+05,  0.91093253744880D+04,&
                    0.55208032572654D+04,  0.33459413680397D+04,  0.20278432533574D+04,  0.12289959111257D+04,&
                    0.74484600674283D+03,  0.45142182226838D+03,  0.27358898319296D+03,  0.16581150496543D+03,&
                    0.10049182119117D+03,  0.60904134055255D+02,  0.36911596397124D+02,  0.22370664483105D+02,&
                    0.13557978474609D+02,  0.82169566512784D+01,  0.49799737280475D+01,  0.30181658957864D+01,&
                    0.18291914519917D+01,  0.11086008799950D+01,  0.67187932120909D+00,  0.40719958861157D+00/)
    coe(1:nfun)=(/  0.13045354562668D+08, -0.71451174388702D+08,  0.16822703340286D+09, -0.11989995205856D+09,&
                    0.11313052767708D+09, -0.47098579088646D+08,  0.44152286214741D+08, -0.11635111604420D+08,&
                    0.16442055727791D+08, -0.12325181142817D+07,  0.65121072747440D+07,  0.80118350201654D+06,&
                    0.28475837798729D+07,  0.83886806490183D+06,  0.12709053696073D+07,  0.51234911274299D+06,&
                    0.52484782480933D+06,  0.20553242147028D+06,  0.15039953727762D+06,  0.23174308616830D+05,&
                    0.99218084599189D+04, -0.84107157748193D+03,  0.27641481989844D+05,  0.30207566965233D+05,&
                    0.95153195177421D+04, -0.12529071259622D+05,  0.23830673105483D+04,  0.11106863850550D+05,&
                   -0.28628557068482D+04, -0.17836423980788D+04,  0.25415066271393D+04, -0.69571543163334D+02,&
                   -0.20477540281903D+03,  0.10508691100458D+03,  0.43214599240944D+02,  0.84094057891335D+01,&
                    0.74282286286461D+00,  0.54128697852036D-01, -0.24460967811564D-02,  0.32964560133131D-03/)
  case(114)
    nfun = 39
    alf(1:nfun)=(/  0.74850931724558D+08,  0.45364201045187D+08,  0.27493455178901D+08,  0.16662700108425D+08,&
                    0.10098606126318D+08,  0.61203673492837D+07,  0.37093135450204D+07,  0.22480688151639D+07,&
                    0.13624659485842D+07,  0.82573693853586D+06,  0.50044662941568D+06,  0.30330098752465D+06,&
                    0.18381878031797D+06,  0.11140532140483D+06,  0.67518376608988D+05,  0.40920228247872D+05,&
                    0.24800138332043D+05,  0.15030386867905D+05,  0.91093253744880D+04,  0.55208032572654D+04,&
                    0.33459413680397D+04,  0.20278432533574D+04,  0.12289959111257D+04,  0.74484600674283D+03,&
                    0.45142182226838D+03,  0.27358898319296D+03,  0.16581150496543D+03,  0.10049182119117D+03,&
                    0.60904134055255D+02,  0.36911596397124D+02,  0.22370664483105D+02,  0.13557978474609D+02,&
                    0.82169566512784D+01,  0.49799737280475D+01,  0.30181658957864D+01,  0.18291914519917D+01,&
                    0.11086008799950D+01,  0.67187932120909D+00,  0.40719958861157D+00/)
    coe(1:nfun)=(/ -0.16449720593361D+08,  0.10172091940712D+09, -0.57371565210266D+08,  0.70753231423691D+08,&
                   -0.17342724488335D+08,  0.27694449383278D+08, -0.81372468953514D+06,  0.11107761949446D+08,&
                    0.24093244103346D+07,  0.49058840237565D+07,  0.19972522249258D+07,  0.23852401354935D+07,&
                    0.12272098739699D+07,  0.11357958504418D+07,  0.63253937048468D+06,  0.47946879507387D+06,&
                    0.23701014522518D+06,  0.13155918704706D+06,  0.30110677945374D+05,  0.44655702654729D+04,&
                    0.37979748421544D+04,  0.28415180694360D+05,  0.31424076705455D+05,  0.74997303827406D+04,&
                   -0.12681246830673D+05,  0.35648686879374D+04,  0.11034164260988D+05, -0.34951166083649D+04,&
                   -0.14004813143101D+04,  0.25764766435347D+04, -0.16580817134805D+03, -0.18377120420267D+03,&
                    0.11250012265674D+03,  0.42316372887902D+02,  0.75625838971441D+01,  0.56363816856862D+00,&
                    0.38436259941221D-01, -0.25460740613070D-02,  0.29257736551536D-03/)
  case(115)
    nfun = 39
    alf(1:nfun)=(/  0.74850931724558D+08,  0.45364201045187D+08,  0.27493455178901D+08,  0.16662700108425D+08,&
                    0.10098606126318D+08,  0.61203673492837D+07,  0.37093135450204D+07,  0.22480688151639D+07,&
                    0.13624659485842D+07,  0.82573693853586D+06,  0.50044662941568D+06,  0.30330098752465D+06,&
                    0.18381878031797D+06,  0.11140532140483D+06,  0.67518376608988D+05,  0.40920228247872D+05,&
                    0.24800138332043D+05,  0.15030386867905D+05,  0.91093253744880D+04,  0.55208032572654D+04,&
                    0.33459413680397D+04,  0.20278432533574D+04,  0.12289959111257D+04,  0.74484600674283D+03,&
                    0.45142182226838D+03,  0.27358898319296D+03,  0.16581150496543D+03,  0.10049182119117D+03,&
                    0.60904134055255D+02,  0.36911596397124D+02,  0.22370664483105D+02,  0.13557978474609D+02,&
                    0.82169566512784D+01,  0.49799737280475D+01,  0.30181658957864D+01,  0.18291914519917D+01,&
                    0.11086008799950D+01,  0.67187932120909D+00,  0.40719958861157D+00/)
    coe(1:nfun)=(/  0.22806323753914D+08,  0.43784006465317D+08,  0.31158049763336D+07,  0.28122298639999D+08,&
                    0.13505150282661D+08,  0.10483031441704D+08,  0.10629700813168D+08,  0.54641277363977D+07,&
                    0.62815377363014D+07,  0.31986545212965D+07,  0.32689814223146D+07,  0.18913521347464D+07,&
                    0.16387707154406D+07,  0.99016147939691D+06,  0.75906553423354D+06,  0.43019516532040D+06,&
                    0.27003360227060D+06,  0.11138682705387D+06,  0.37696956836751D+05, -0.11287093636021D+04,&
                    0.88351687081619D+04,  0.29083889112755D+05,  0.32525360817591D+05,  0.53746709161425D+04,&
                   -0.12664338419696D+05,  0.47612088074369D+04,  0.10858316819944D+05, -0.40884954994802D+04,&
                   -0.99077387844303D+03,  0.25846680056483D+04, -0.25698548461734D+03, -0.16035928546637D+03,&
                    0.11926675181847D+03,  0.41149874979362D+02,  0.67091512315558D+01,  0.41331986309115D+00,&
                    0.27911374215677D-01, -0.25110512072004D-02,  0.26912514184740D-03/)
  case(116)
    nfun = 37
    alf(1:nfun)=(/  0.74850931724558D+08,  0.45364201045187D+08,  0.27493455178901D+08,  0.16662700108425D+08,&
                    0.10098606126318D+08,  0.61203673492837D+07,  0.37093135450204D+07,  0.22480688151639D+07,&
                    0.13624659485842D+07,  0.82573693853586D+06,  0.50044662941568D+06,  0.30330098752465D+06,&
                    0.18381878031797D+06,  0.11140532140483D+06,  0.67518376608988D+05,  0.40920228247872D+05,&
                    0.24800138332043D+05,  0.15030386867905D+05,  0.91093253744880D+04,  0.55208032572654D+04,&
                    0.33459413680397D+04,  0.20278432533574D+04,  0.12289959111257D+04,  0.74484600674283D+03,&
                    0.45142182226838D+03,  0.27358898319296D+03,  0.16581150496543D+03,  0.10049182119117D+03,&
                    0.60904134055255D+02,  0.36911596397124D+02,  0.22370664483105D+02,  0.13557978474609D+02,&
                    0.82169566512784D+01,  0.49799737280475D+01,  0.30181658957864D+01,  0.18291914519917D+01,&
                    0.11086008799950D+01/)
    coe(1:nfun)=(/  0.33112859893250D+08,  0.15835177618530D+08,  0.42550155291199D+08, -0.18042907484741D+07,&
                    0.37407209910675D+08, -0.27639219714966D+07,  0.20001374277420D+08,  0.10325934230576D+07,&
                    0.95360868274994D+07,  0.18643778463440D+07,  0.43538545676451D+07,  0.15120183296556D+07,&
                    0.19926362674189D+07,  0.87876359100842D+06,  0.86704444460177D+06,  0.39074929916251D+06,&
                    0.29693679039630D+06,  0.94153463921934D+05,  0.43592055846736D+05, -0.55251063662842D+04,&
                    0.13535528299201D+05,  0.30036335231081D+05,  0.33271849499839D+05,  0.32843873189609D+04,&
                   -0.12540240676675D+05,  0.59951728168402D+04,  0.10559665761717D+05, -0.46233684605259D+04,&
                   -0.56636416306940D+03,  0.25697829532634D+04, -0.34365836852806D+03, -0.13413934789656D+03,&
                    0.12497110193741D+03,  0.39902239350551D+02,  0.57844272759326D+01,  0.32517798372328D+00,&
                    0.10117643746312D-01/)
  case(117)
    nfun = 37
    alf(1:nfun)=(/  0.74850931724558D+08,  0.45364201045187D+08,  0.27493455178901D+08,  0.16662700108425D+08,&
                    0.10098606126318D+08,  0.61203673492837D+07,  0.37093135450204D+07,  0.22480688151639D+07,&
                    0.13624659485842D+07,  0.82573693853586D+06,  0.50044662941568D+06,  0.30330098752465D+06,&
                    0.18381878031797D+06,  0.11140532140483D+06,  0.67518376608988D+05,  0.40920228247872D+05,&
                    0.24800138332043D+05,  0.15030386867905D+05,  0.91093253744880D+04,  0.55208032572654D+04,&
                    0.33459413680397D+04,  0.20278432533574D+04,  0.12289959111257D+04,  0.74484600674283D+03,&
                    0.45142182226838D+03,  0.27358898319296D+03,  0.16581150496543D+03,  0.10049182119117D+03,&
                    0.60904134055255D+02,  0.36911596397124D+02,  0.22370664483105D+02,  0.13557978474609D+02,&
                    0.82169566512784D+01,  0.49799737280475D+01,  0.30181658957864D+01,  0.18291914519917D+01,&
                    0.11086008799950D+01/)
    coe(1:nfun)=(/  0.73272556941864D+08, -0.41830405679077D+08,  0.10316219433490D+09, -0.43507293210693D+08,&
                    0.68091874106049D+08, -0.19518769627045D+08,  0.31393377108688D+08, -0.44121568762207D+07,&
                    0.13407776870747D+08,  0.21996846143150D+06,  0.56307521016703D+07,  0.10394241175909D+07,&
                    0.24033620833936D+07,  0.73804458005047D+06,  0.99182206504846D+06,  0.34130647319794D+06,&
                    0.32873628874126D+06,  0.73751294571549D+05,  0.51217937785566D+05, -0.10585962791555D+05,&
                    0.18941947586145D+05,  0.30664897160498D+05,  0.33982606172839D+05,  0.10630985845788D+04,&
                   -0.12197972315752D+05,  0.71910136446282D+04,  0.10175411793121D+05, -0.51115843556338D+04,&
                   -0.12265227376920D+03,  0.25271581730045D+04, -0.42204795917442D+03, -0.10697219469052D+03,&
                    0.13033340038424D+03,  0.38213184638268D+02,  0.49773327436501D+01,  0.22811410543818D+00,&
                    0.62863572424163D-02/)
  case(118)
    nfun = 37
    alf(1:nfun)=(/  0.74850931724558D+08,  0.45364201045187D+08,  0.27493455178901D+08,  0.16662700108425D+08,&
                    0.10098606126318D+08,  0.61203673492837D+07,  0.37093135450204D+07,  0.22480688151639D+07,&
                    0.13624659485842D+07,  0.82573693853586D+06,  0.50044662941568D+06,  0.30330098752465D+06,&
                    0.18381878031797D+06,  0.11140532140483D+06,  0.67518376608988D+05,  0.40920228247872D+05,&
                    0.24800138332043D+05,  0.15030386867905D+05,  0.91093253744880D+04,  0.55208032572654D+04,&
                    0.33459413680397D+04,  0.20278432533574D+04,  0.12289959111257D+04,  0.74484600674283D+03,&
                    0.45142182226838D+03,  0.27358898319296D+03,  0.16581150496543D+03,  0.10049182119117D+03,&
                    0.60904134055255D+02,  0.36911596397124D+02,  0.22370664483105D+02,  0.13557978474609D+02,&
                    0.82169566512784D+01,  0.49799737280475D+01,  0.30181658957864D+01,  0.18291914519917D+01,&
                    0.11086008799950D+01/)
    coe(1:nfun)=(/  0.11420874745679D+09, -0.96423931190735D+08,  0.16061572709473D+09, -0.82358817068237D+08,&
                    0.97212539991394D+08, -0.35132152518219D+08,  0.42282644410553D+08, -0.94938418381195D+07,&
                    0.17133008819267D+08, -0.13030406535816D+07,  0.68621371685247D+07,  0.60281139500713D+06,&
                    0.28005835447545D+07,  0.60712042498755D+06,  0.11110726314778D+07,  0.29452919721532D+06,&
                    0.35821586694646D+06,  0.54236291408002D+05,  0.58381123854876D+05, -0.15081419182085D+05,&
                    0.24389462430522D+05,  0.31306114485338D+05,  0.34450252551096D+05, -0.11557787508853D+04,&
                   -0.11697381509866D+05,  0.83689551867417D+04,  0.96891596153872D+04, -0.55344339090961D+04,&
                    0.32736409641380D+03,  0.24618367979728D+04, -0.49332418761605D+03, -0.78319022446976D+02,&
                    0.13489860060415D+03,  0.36298767980030D+02,  0.42153349273672D+01,  0.15460150708793D+00,&
                    0.40536321589144D-02/)
  case(119)
    nfun = 36
    alf(1:nfun)=(/  0.27493455178901D+08,  0.16662700108425D+08,  0.10098606126318D+08,  0.61203673492837D+07,&
                    0.37093135450204D+07,  0.22480688151639D+07,  0.13624659485842D+07,  0.82573693853586D+06,&
                    0.50044662941568D+06,  0.30330098752465D+06,  0.18381878031797D+06,  0.11140532140483D+06,&
                    0.67518376608988D+05,  0.40920228247872D+05,  0.24800138332043D+05,  0.15030386867905D+05,&
                    0.91093253744880D+04,  0.55208032572654D+04,  0.33459413680397D+04,  0.20278432533574D+04,&
                    0.12289959111257D+04,  0.74484600674283D+03,  0.45142182226838D+03,  0.27358898319296D+03,&
                    0.16581150496543D+03,  0.10049182119117D+03,  0.60904134055255D+02,  0.36911596397124D+02,&
                    0.22370664483105D+02,  0.13557978474609D+02,  0.82169566512784D+01,  0.49799737280475D+01,&
                    0.30181658957864D+01,  0.18291914519917D+01,  0.11086008799950D+01,  0.67187932120909D+00/)
    coe(1:nfun)=(/  0.23461466894828D+09, -0.26841415085935D+09,  0.29460910029617D+09, -0.18434836964632D+09,&
                    0.14313588233511D+09, -0.69643171213146D+08,  0.53084798857014D+08, -0.21166817178737D+08,&
                    0.18502809936180D+08, -0.56447442045028D+07,  0.64527803133231D+07, -0.13158199660330D+07,&
                    0.22324771637782D+07, -0.30114043244311D+06,  0.69808879322405D+06, -0.13243475369383D+06,&
                    0.16277861800820D+06, -0.70057101451369D+05,  0.59613440810631D+05,  0.14991578446841D+05,&
                    0.42860249785210D+05, -0.83232744181554D+04, -0.77263066955996D+04,  0.78738427608060D+04,&
                    0.97338043435935D+04, -0.62104238772464D+04,  0.10390973687351D+04,  0.21897856880951D+04,&
                   -0.46536664850264D+03, -0.77600374543115D+02,  0.14374560783319D+03,  0.33985407040635D+02,&
                    0.32988373103292D+01,  0.20070142337175D+00, -0.18308131557603D-01,  0.22238342211338D-02/)
  case(120)
    nfun = 34
    alf(1:nfun)=(/  0.27493455178901D+08,  0.16662700108425D+08,  0.10098606126318D+08,  0.61203673492837D+07,&
                    0.37093135450204D+07,  0.22480688151639D+07,  0.13624659485842D+07,  0.82573693853586D+06,&
                    0.50044662941568D+06,  0.30330098752465D+06,  0.18381878031797D+06,  0.11140532140483D+06,&
                    0.67518376608988D+05,  0.40920228247872D+05,  0.24800138332043D+05,  0.15030386867905D+05,&
                    0.91093253744880D+04,  0.55208032572654D+04,  0.33459413680397D+04,  0.20278432533574D+04,&
                    0.12289959111257D+04,  0.74484600674283D+03,  0.45142182226838D+03,  0.27358898319296D+03,&
                    0.16581150496543D+03,  0.10049182119117D+03,  0.60904134055255D+02,  0.36911596397124D+02,&
                    0.22370664483105D+02,  0.13557978474609D+02,  0.82169566512784D+01,  0.49799737280475D+01,&
                    0.30181658957864D+01,  0.18291914519917D+01/)
    coe(1:nfun)=(/  0.25500022160287D+09, -0.28831077790323D+09,  0.31663208784598D+09, -0.19715684268198D+09,&
                    0.15341610174687D+09, -0.74396797093719D+08,  0.56821657971733D+08, -0.22634198693882D+08,&
                    0.19761614330992D+08, -0.60720376368361D+07,  0.68589004486456D+07, -0.14461824809818D+07,&
                    0.23526124356861D+07, -0.34920941128361D+06,  0.72743533030385D+06, -0.15225434301406D+06,&
                    0.17053882948497D+06, -0.74045170333624D+05,  0.65362317625314D+05,  0.15302773034826D+05,&
                    0.42913491375364D+05, -0.10446054031812D+05, -0.68614030156541D+04,  0.88939820975256D+04,&
                    0.90953348552752D+04, -0.64822523946550D+04,  0.14708817847709D+04,  0.20917816349430D+04,&
                   -0.52334642407068D+03, -0.46788464342768D+02,  0.14695541526303D+03,  0.31387894884935D+02,&
                    0.28543375162367D+01,  0.85693524799069D-01/)
end select

return
end

